package ru.yandex.solomon.gateway.tasks.removeShard;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import ru.yandex.gateway.api.task.RemoveShardParams;
import ru.yandex.gateway.api.task.RemoveShardProgress.RemoveConf;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardFromConf implements AutoCloseable {
    private final RetryContext retryContext;
    private final RemoveShardParams params;
    private final ShardsDao dao;

    private final AtomicReference<RemoveConf> progress;

    public RemoveShardFromConf(RetryConfig retry, ShardsDao dao, RemoveShardParams params, RemoveConf progress) {
        this.retryContext = new RetryContext(retry);
        this.dao = dao;
        this.params = params;
        this.progress = new AtomicReference<>(progress);
    }

    public CompletableFuture<Void> start() {
        if (progress.get().getComplete()) {
            return completedFuture(null);
        }

        try {
            return findAndRemoveShard()
                    .whenComplete((ignore, e) -> updateProgress(e == null));
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public RemoveConf progress() {
        return progress.get();
    }

    private CompletableFuture<Optional<Shard>> findShard() {
        return retry(() -> dao.findOne(params.getProjectId(), "", params.getShardId()));
    }

    private CompletableFuture<Void> findAndRemoveShard() {
        return retry(() -> findShard().thenCompose(shard -> removeShard(shard.orElse(null))));
    }

    private CompletableFuture<Void> removeShard(@Nullable Shard shard) {
        if (shard == null) {
            return completedFuture(null);
        }

        if (shard.getNumId() != params.getNumId()) {
            return completedFuture(null);
        }

        // TODO: remove by numId (@gordiychuk)
        return dao.deleteOne(shard.getProjectId(), shard.getFolderId(), shard.getId())
                .thenAccept(deleted -> {
                    if (!deleted) {
                        throw new RuntimeException("Failed to delete shard " + shard.getId() + " in project " + shard.getProjectId());
                    }
                });
    }

    private void updateProgress(boolean complete) {
        var prev = progress.get();
        progress.set(RemoveConf.newBuilder()
                .setComplete(complete || prev.getComplete())
                .build());
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryContext.retry(supplier);
    }

    @Override
    public void close() {
        retryContext.close();
    }
}
