package ru.yandex.solomon.gateway.tasks.removeShard;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.gateway.api.task.RemoteTaskProgressOrBuilder;
import ru.yandex.gateway.api.task.RemoveShardParams;
import ru.yandex.monitoring.coremon.TRemoveShardRequest;
import ru.yandex.monitoring.coremon.TRemoveShardResponse;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.gateway.tasks.RemoteTask;
import ru.yandex.solomon.gateway.tasks.RemoteTask.RemoteTaskClient;
import ru.yandex.solomon.gateway.tasks.SubTask;
import ru.yandex.solomon.scheduler.proto.GetTaskRequest;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Vladimir Gordiychuk
 */
final class RemoveShardFromReplica implements SubTask<RemoteTaskProgress> {

    private final RemoteTask remoteTask;

    RemoveShardFromReplica(
        RetryConfig retry,
        CoremonClient coremonClient,
        Executor executor,
        ScheduledExecutorService timer,
        RemoveShardParams params,
        RemoteTaskProgress progress)
    {
        this.remoteTask = new RemoteTask(
            "remove_shard",
            executor,
            timer,
            progress,
            new RemoveShardTaskClient(retry, coremonClient, progress.getClusterId(), params),
            RemoveShardFromReplica::isComplete);
    }

    @Override
    public CompletableFuture<Void> start(boolean interrupt, Runnable onProgressChange) {
        return remoteTask.start(interrupt, onProgressChange);
    }

    @Override
    public boolean isIdle() {
        return remoteTask.isIdle();
    }

    @Override
    public RemoteTaskProgress progress() {
        return remoteTask.progress();
    }

    @Override
    public void close()  {
        remoteTask.close();
    }

    private static boolean isComplete(RemoteTaskProgressOrBuilder progress) {
        return progress.getComplete()
            || progress.getRemoteTaskCompletedAt() > 0 && progress.getRemoteTaskRemovedAt() > 0;
    }

    private static final class RemoveShardTaskClient extends RemoteTaskClient {

        private final CoremonClient coremonClient;
        private final String clusterId;
        private final RemoveShardParams params;

        RemoveShardTaskClient(
            RetryConfig retry,
            CoremonClient coremonClient,
            String clusterId,
            RemoveShardParams params)
        {
            super(retry);
            this.coremonClient = coremonClient;
            this.clusterId = clusterId;
            this.params = params;
        }

        @Override
        public CompletableFuture<String> schedule() {
            var request = request();
            return retry(() -> coremonClient.removeShard(clusterId, request))
                .thenApply(TRemoveShardResponse::getTaskId);
        }

        @Override
        public CompletableFuture<Task> getTask(GetTaskRequest request) {
            return retry(() -> coremonClient.getTask(clusterId, request));
        }

        private TRemoveShardRequest request() {
            return TRemoveShardRequest.newBuilder()
                .setProjectId(params.getProjectId())
                .setShardId(params.getShardId())
                .setNumId(params.getNumId())
                .build();
        }
    }

}
