package ru.yandex.solomon.gateway.tasks.removeShard;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import com.google.protobuf.TextFormat;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.gateway.api.task.RemoveShardParams;
import ru.yandex.gateway.api.task.RemoveShardProgress;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.grpc.Proto;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.time.DurationUtils;

import static ru.yandex.solomon.util.time.DurationUtils.backoff;
import static ru.yandex.solomon.util.time.DurationUtils.randomize;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardTask implements AutoCloseable {
    private static final long BACKOFF_DELAY_MILLIS = TimeUnit.MINUTES.toMillis(10);
    private static final long BACKOFF_MAX_DELAY_MILLIS = TimeUnit.DAYS.toMillis(1);
    private static final Logger logger = LoggerFactory.getLogger(RemoveShardTask.class);

    private final ExecutionContext context;
    private final RemoveShardParams params;
    private final AtomicReference<RemoveShardProgress> progress;
    private final ShardsDao dao;

    private final RemoveShardFromConf removeFromConf;
    private final RemoveShardFromReplicas removeShardFromReplicas;

    private final PingActorRunner actor;

    public RemoveShardTask(
            RetryConfig retryConfig,
            CoremonClient coremonClient,
            ShardsDao shardsDao,
            Executor executor,
            ScheduledExecutorService timer,
            ExecutionContext context)
    {
        this.context = context;
        this.params = RemoveShardTaskProto.params(context.task().params());
        this.progress = new AtomicReference<>(RemoveShardTaskProto.progress(context.task().progress()));
        this.dao = shardsDao;

        retryConfig = retryConfig
                .withStats((timeSpentMillis, cause) -> logger.warn("{} error", logPrefix(), cause));

        this.removeFromConf = new RemoveShardFromConf(retryConfig, shardsDao, params, progress.get().getRemoveConf());
        this.removeShardFromReplicas = new RemoveShardFromReplicas(retryConfig, coremonClient, executor, timer, params, progress.get().getRemoveReplicaList());
        this.actor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofSeconds(15))
                .operation("remove_shard_save_progress")
                .executor(executor)
                .timer(timer)
                .onPing(this::safeProgress)
                .build();
    }

    public RemoveShardProgress progress() {
        updateProgress();
        return progress.get();
    }

    public CompletableFuture<Void> start() {
        actor.schedule();
        var root = new CompletableFuture<>();
        var future = root.thenCompose(ignore -> removeFromConf.start())
                .thenRun(actor::forcePing)
                .thenCompose(ignore -> removeShardFromReplicas.start())
                .thenCompose(ignore -> onCompleteRemoteTasks())
                .handle((ignore, e) -> e != null ? Status.fromThrowable(e) : Status.OK)
                .thenCompose(status -> {
                    updateProgress(status);
                    if (context.isDone()) {
                        return CompletableFuture.completedFuture(null);
                    } else if (status.isOk()) {
                        return complete();
                    } else {
                        return reschedule();
                    }
                })
                .whenComplete((ignore, e) -> {
                    if (e != null && logger.isWarnEnabled()) {
                        logger.warn("{} failed on complete", logPrefix(), e);
                    }

                    close();
                });

        root.complete(null);
        return future;
    }

    private CompletableFuture<Void> onCompleteRemoteTasks() {
        updateProgress();

        var executeAt = removeShardFromReplicas.progress()
                .stream()
                .filter(replica -> replica.getRemoteTaskCompletedAt() == 0)
                .mapToLong(value -> value.getRemoteTask().getExecuteAt())
                .min();

        if (executeAt.isPresent()) {
            return reschedule(Math.max(executeAt.getAsLong(), System.currentTimeMillis()) + DurationUtils.randomize(5_000));
        }

        var anyError = removeShardFromReplicas.progress()
                .stream()
                .filter(replica -> replica.getRemoteTaskCompletedAt() != 0)
                .map(replica -> Proto.fromProto(replica.getRemoteTask().getStatus()))
                .filter(status -> !status.isOk())
                .findAny();

        if (anyError.isPresent()) {
            return fail(anyError.get());
        }

        var completedNotRemoved = removeShardFromReplicas.progress()
                .stream()
                .filter(replica -> replica.getRemoteTaskRemovedAt() == 0)
                .mapToLong(RemoteTaskProgress::getRemoteTaskCompletedAt)
                .min();

        if (completedNotRemoved.isPresent()) {
            long shouldBeRemovedAt = completedNotRemoved.getAsLong() + TimeUnit.DAYS.toMillis(7);
            return reschedule(Math.max(shouldBeRemovedAt, System.currentTimeMillis()) + DurationUtils.randomize(TimeUnit.DAYS.toMillis(1)));
        }

        var removedAt = removeShardFromReplicas.progress()
                .stream()
                .mapToLong(RemoteTaskProgress::getRemoteTaskRemovedAt)
                .filter(value -> value > 0)
                .max();

        if (removedAt.isEmpty()) {
            return reschedule(System.currentTimeMillis() + DurationUtils.randomize(TimeUnit.DAYS.toMillis(1)));
        }

        var releaseAt = removedAt.getAsLong() + TimeUnit.DAYS.toMillis(30L);
        if (releaseAt < System.currentTimeMillis()) {
            return releaseNumId();
        }

        return reschedule(releaseAt + DurationUtils.randomize(TimeUnit.DAYS.toMillis(1)));
    }

    private CompletableFuture<Void> releaseNumId() {
        return dao.releaseNumId(params.getProjectId(), params.getShardId(), params.getNumId())
                .thenAccept(success -> {
                    if (!success) {
                        throw Status.FAILED_PRECONDITION
                                .withDescription("Unable release numId for " + TextFormat.shortDebugString(params))
                                .asRuntimeException();
                    }
                });
    }

    private CompletableFuture<Void> fail(Status status) {
        return context.fail(status.asRuntimeException())
                .thenRun(() -> logger.debug("{} fail {}, latest progress ({})",
                        logPrefix(), status, TextFormat.shortDebugString(progress.get())));
    }

    private CompletableFuture<Void> complete() {
        var result = RemoveShardTaskProto.mergeResult(progress());
        actor.close();
        return context.complete(result)
                .thenRun(() -> logger.info("{} completed ({})",
                        logPrefix(), TextFormat.shortDebugString(result)));
    }

    private CompletableFuture<Void> reschedule(long executeAt) {
        var progress = progress();
        actor.close();
        return context.reschedule(executeAt, progress)
                .thenRun(() -> logger.debug("{} rescheduled on {}, latest progress ({})",
                        logPrefix(), Instant.ofEpochMilli(executeAt), TextFormat.shortDebugString(progress)));
    }

    private CompletableFuture<Void> reschedule() {
        var progress = progress();
        long delayMillis = randomize(backoff(BACKOFF_DELAY_MILLIS, BACKOFF_MAX_DELAY_MILLIS, progress.getAttempt()));
        long executeAt = System.currentTimeMillis() + delayMillis;
        return reschedule(executeAt);
    }

    private void updateProgress() {
        RemoveShardProgress prev;
        RemoveShardProgress update;
        do {
            prev = progress.get();
            update = prev.toBuilder()
                    .setRemoveConf(removeFromConf.progress())
                    .clearRemoveReplica()
                    .addAllRemoveReplica(removeShardFromReplicas.progress())
                    .build();

            if (prev.equals(update)) {
                return;
            }
        } while (!progress.compareAndSet(prev, update));
        if (logger.isDebugEnabled()) {
            logger.debug("{} progress ({})", logPrefix(), TextFormat.shortDebugString(update));
        }
    }

    private void updateProgress(Status status) {
        RemoveShardProgress prev;
        RemoveShardProgress update;
        do {
            prev = progress.get();
            var updateBuilder = prev.toBuilder()
                    .setRemoveConf(removeFromConf.progress())
                    .clearRemoveReplica()
                    .addAllRemoveReplica(removeShardFromReplicas.progress());

            if (status.isOk()) {
                updateBuilder.clearAttempt();
                updateBuilder.clearStatus();
            } else {
                updateBuilder.setAttempt(prev.getAttempt() + 1)
                        .setStatus(Proto.toProto(status));
            }

            update = updateBuilder.build();
            if (prev.equals(update)) {
                return;
            }
        } while (!progress.compareAndSet(prev, update));
        if (logger.isDebugEnabled()) {
            logger.debug("{} progress ({})", logPrefix(), TextFormat.shortDebugString(update));
        }
    }

    private CompletableFuture<?> safeProgress(int attempt) {
        if (context.isDone()) {
            close();
            return CompletableFuture.completedFuture(null);
        }

        return context.progress(progress())
                .whenComplete((ignore, e) -> {
                    if (context.isDone()) {
                        close();
                    }
                });
    }

    private String logPrefix() {
        return "remove_shard(task_id: \"" + context.task().id() + "\" " + TextFormat.shortDebugString(params) + ")";
    }

    @Override
    public void close() {
        if (logger.isDebugEnabled()) {
            logger.debug("{} close", logPrefix());
        }

        if (!context.isDone()) {
            context.cancel();
        }

        removeFromConf.close();
        removeShardFromReplicas.close();
        actor.close();
    }
}
