package ru.yandex.solomon.gateway.tasks.removeShard;

import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.Descriptors.Descriptor;

import ru.yandex.gateway.api.task.RemoveShardParams;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.Permit;
import ru.yandex.solomon.scheduler.PermitLimiter;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.TaskHandler;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class RemoveShardTaskHandler implements TaskHandler {
    private static final String TYPE = "remove_shard";
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(10)
            .withDelay(TimeUnit.SECONDS.toMillis(1))
            .withMaxDelay(TimeUnit.MINUTES.toMillis(1));

    private final PermitLimiter permitLimiter = new PermitLimiter(10);
    private final CoremonClient coremon;
    private final ShardsDao shardsDao;
    private final Executor executor;
    private final ScheduledExecutorService timer;

    // for manager ui
    private final ConcurrentMap<String, RemoveShardTask> running = new ConcurrentHashMap<>();

    public RemoveShardTaskHandler(
            CoremonClient coremon,
            ShardsDao shardsDao,
            Executor executor,
            ScheduledExecutorService timer)
    {
        this.coremon = coremon;
        this.shardsDao = shardsDao;
        this.executor = executor;
        this.timer = timer;
    }

    @Override
    public String type() {
        return TYPE;
    }

    @Override
    @Nullable
    public Permit acquire(String id, Any params) {
        if (coremon.clusterIds().isEmpty()) {
            return null;
        }

        return permitLimiter.acquire();
    }

    @Override
    public void execute(ExecutionContext context) {
        var taskId = context.task().id();
        var process = new RemoveShardTask(RETRY_CONFIG, coremon, shardsDao, executor, timer, context);
        running.put(taskId, process);
        try {
            process.start().whenComplete((ignore, e) -> {
                running.remove(taskId, process);
                process.close();
            });
        } catch (Throwable e) {
            running.remove(taskId, process);
            throw new RuntimeException(e);
        }
    }

    @SuppressWarnings("UnnecessaryFullyQualifiedName")
    @Override
    public List<Descriptor> descriptors() {
        return List.of(
                /// gateway
                ru.yandex.gateway.api.task.RemoveShardParams.getDescriptor(),
                ru.yandex.gateway.api.task.RemoveShardProgress.getDescriptor(),
                ru.yandex.gateway.api.task.RemoveShardResult.getDescriptor(),
                // coremon
                ru.yandex.coremon.api.task.RemoveShardParams.getDescriptor(),
                ru.yandex.coremon.api.task.RemoveShardProgress.getDescriptor(),
                ru.yandex.coremon.api.task.RemoveShardResult.getDescriptor());
    }

    public static Task removeShardTask(String projectId, String shardId, int numId) {
        var params = Any.pack(RemoveShardParams.newBuilder()
                .setNumId(numId)
                .setProjectId(projectId)
                .setShardId(shardId)
                .build());

        return Task.newBuilder()
                .setId(IdGenerator.generateInternalId())
                .setType(TYPE)
                .setExecuteAt(System.currentTimeMillis())
                .setParams(params)
                .build();
    }

}
