package ru.yandex.solomon.gateway.utils.conf;

import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.gateway.utils.WwwBooleans;
import ru.yandex.solomon.gateway.utils.conf.source.GraphSettingsSource;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class GraphSetting {

    static final String DEFAULT_ON_LABEL = "On";
    static final String DEFAULT_OFF_LABEL = "Off";

    public enum ControlKind {
        BOOLEAN_DROPDOWN,
        DROPDOWN,
        TEXTFIELD,
        MULTILINETEXT,
        NUMBER_FORMAT,
    }

    enum BooleanDropdown { BD }

    enum TextField { F, ML }

    enum Dropdown { D }

    enum NumberFormatField { NF }

    public final ControlKind kind;
    public final String queryArg;
    public final String title;
    public final String hint;
    private final GraphSettingOption[] options;
    public final String defaultValue;
    public final Predicate<GraphSettingsSource> showIf;

    private Object protobufField;

    public String defaultValue() {
        return defaultValue;
    }

    public boolean isDefaultValue(String value, GraphSettingsSource defaults) {
        if (value == null || value.isEmpty()) {
            return true;
        }

        String defaultValue = defaults.getRaw(queryArg);
        if (defaultValue == null || defaultValue.isEmpty()) {
            defaultValue = defaultValue();
        }

        value = normalize(value);
        defaultValue = normalize(defaultValue);
        return value.equals(defaultValue);
    }

    private static String normalize(String value) {
        return value.replace("\r", "");
    }

    GraphSetting(TextField f, String queryArg, String title, String hint) {
        this(f, queryArg, title, hint, s -> true);
    }

    public GraphSetting(TextField f, String queryArg, String title, String hint, Predicate<GraphSettingsSource> showIf) {
        this(f, queryArg, title, hint, "", showIf);
    }

    public GraphSetting(TextField f, String queryArg, String title, String hint, String defaultValue, Predicate<GraphSettingsSource> showIf) {
        if (f == TextField.F) {
            this.kind = ControlKind.TEXTFIELD;
        } else {
            this.kind = ControlKind.MULTILINETEXT;
        }
        this.queryArg = queryArg;
        this.title = title;
        this.hint = hint;
        this.options = new GraphSettingOption[0];
        this.defaultValue = defaultValue;
        this.showIf = showIf;
    }

    public GraphSetting(NumberFormatField f, String queryArg, String title, String hint) {
        this.kind = ControlKind.NUMBER_FORMAT;
        this.queryArg = queryArg;
        this.title = title;
        this.hint = hint;
        this.options = new GraphSettingOption[0];
        this.defaultValue = "3|auto";
        this.showIf = s -> true;
    }

    public GraphSetting(
        BooleanDropdown bd,
        String queryArg,
        String title,
        boolean defaultValue,
        Predicate<GraphSettingsSource> showIf,
        String onLabel,
        String offLabel)
    {
        this.kind = ControlKind.BOOLEAN_DROPDOWN;
        this.queryArg = queryArg;
        this.title = title;
        this.hint = "";
        this.options = new GraphSettingOption[]{
            new GraphSettingOption("", "Default (" + (defaultValue ? onLabel : offLabel) + ")"),
            new GraphSettingOption(WwwBooleans.formatBoolean(true), onLabel),
            new GraphSettingOption(WwwBooleans.formatBoolean(false), offLabel)};
        this.defaultValue = "";
        this.showIf = showIf;
    }

    public GraphSetting(BooleanDropdown bd, String queryArg, String title, boolean defaultValue) {
        this(bd, queryArg, title, defaultValue, DEFAULT_ON_LABEL, DEFAULT_OFF_LABEL);
    }

    public GraphSetting(BooleanDropdown bd, String queryArg, String title, boolean defaultValue, String onLabel, String offLabel) {
        this(bd, queryArg, title, defaultValue, s -> true, onLabel, offLabel);
    }

    public GraphSetting(Dropdown d, String queryArg, String title,  GraphSettingOption[] options) {
        this(d, queryArg, title, graphSettingsSource -> true, options);
    }

    public GraphSetting(
        Dropdown d,
        String queryArg,
        String title,
        Predicate<GraphSettingsSource> showIf,
        GraphSettingOption[] options)
    {
        this.kind = ControlKind.DROPDOWN;
        this.queryArg = queryArg;
        this.title = title;
        this.hint = "";
        this.options = options;
        this.defaultValue = options[0].getValue();
        this.showIf = showIf;
    }

    public Object getProtobufFieldRaw() {
        return protobufField;
    }

    public String getFrom(GraphSettingsSource source) {
        return GraphSettingsSource.getRawOrEmpty(source, queryArg);
    }

    public String getFromOrDefault(GraphSettingsSource source) {
        String r = GraphSettingsSource.getRawOrEmpty(source, queryArg);
        if (r.isEmpty()) {
            r = defaultValue();
        }
        return r;
    }

    public boolean getBoolFrom(GraphSettingsSource source) {
        return WwwBooleans.isTrue(getFromOrDefault(source));
    }

    public void initProtobufField(Object protobufField) {
        if (this.protobufField != null) {
            throw new IllegalStateException("protobuf field already initialized");
        }
        this.protobufField = protobufField;
    }
}
