package ru.yandex.solomon.gateway.utils.conf;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.solomon.gateway.utils.GraphMode;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.utils.conf.source.GraphSettingsSource;

/**
 * @author Stepan Koltsov
 */
public class GraphSettings {
    public static final GraphSetting description = new GraphSetting(GraphSetting.TextField.ML, UserLinksBasic.GRAPH_DESCRIPTION, "Graph description", "description");
    public static final GraphSetting max = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.MAX_PARAM, "Max", "19G");
    public static final GraphSetting min = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.MIN_PARAM, "Min", "17K");
    public static final GraphSetting norm = new GraphSetting(GraphSetting.BooleanDropdown.BD, UserLinksBasic.NORM_PARAM, "Normalize", false);
    public static final GraphSetting cs = new GraphSetting(GraphSetting.Dropdown.D, UserLinksBasic.COLOR_SCHEME_QUERY_ARG, "Gradient", new GraphSettingOption[]{
        new GraphSettingOption("", "Auto"),
        new GraphSettingOption("default", "Default"),
        new GraphSettingOption("gradient", "Gradient"),
    });
    public static final GraphSetting dropNans = new GraphSetting(GraphSetting.BooleanDropdown.BD, UserLinksBasic.DROP_NANS_PARAM, "Drop NaN value", false);
    public static final GraphSetting stack = new GraphSetting(GraphSetting.BooleanDropdown.BD, UserLinksBasic.STACK_PARAM, "Stack/Area", true, GraphSettingsSource::isAnyGraphLines, GraphSetting.DEFAULT_ON_LABEL, GraphSetting.DEFAULT_OFF_LABEL);
    public static final GraphSetting aggr = new GraphSetting(GraphSetting.Dropdown.D, UserLinksBasic.AGGR_PARAM, "Aggregation function", new GraphSettingOption[]{
        new GraphSettingOption("avg", "avg"),
        new GraphSettingOption("min", "min"),
        new GraphSettingOption("max", "max"),
        new GraphSettingOption("sum", "sum"),
        new GraphSettingOption("last", "last"),
    });

    public static final GraphSetting graphMode = new GraphSetting(
        GraphSetting.Dropdown.D, UserLinksBasic.GRAPH_MODE, "Main graph", GraphMode.graphSettingOptions(true));
    public static final GraphSetting secondaryGraphMode = new GraphSetting(
        GraphSetting.Dropdown.D, UserLinksBasic.SECONDARY_GRAPH_MODE, "Secondary graph", GraphMode.graphSettingOptions(false));

    public static final GraphSetting interpolate = new GraphSetting(GraphSetting.Dropdown.D, UserLinksBasic.INTERPOLATE_PARAM, "Interpolate", new GraphSettingOption[]{
        new GraphSettingOption("linear", "Linear"),
        new GraphSettingOption("left", "Left"),
        new GraphSettingOption("right", "Right"),
        new GraphSettingOption("none", "None"),
    });

    public static final GraphSetting transform = new GraphSetting(
        GraphSetting.Dropdown.D,
        UserLinksBasic.TRANSFORM,
        "Transform",
        GraphSettingsSource::isAnyGraphLines,
        new GraphSettingOption[]{
            new GraphSettingOption("none", "None"),
            new GraphSettingOption("differentiate", "Differentiate"),
            new GraphSettingOption("differentiate_with_negative", "Differentiate with negative"),
            new GraphSettingOption("integrate", "Integrate"),
            new GraphSettingOption("moving_average", "Moving average"),
            new GraphSettingOption("moving_percentile", "Moving percentile"),
            new GraphSettingOption("diff", "Diff between values"),
            new GraphSettingOption("asap", "Auto Smoother"),
        }
    );

    public static final  GraphSetting overLinesTransform = new GraphSetting(
        GraphSetting.Dropdown.D,
        UserLinksBasic.OVER_LINES_TRANSFORM,
        "Over lines transform",
        GraphSettingsSource::isAnyGraphLines,
        new GraphSettingOption[]{
            new GraphSettingOption("NONE", "None"),
            new GraphSettingOption("PERCENTILE", "Percentile"),
            new GraphSettingOption("WEIGHTED_PERCENTILE", "Weighted Percentile"),
            new GraphSettingOption("SUMMARY", "Summary"),
        }
    );

    public static final GraphSetting scale = new GraphSetting(GraphSetting.Dropdown.D, UserLinksBasic.SCALE_PARAM, "Scale", new GraphSettingOption[]{
        new GraphSettingOption(UserLinksBasic.SCALE_NATURAL, "Natural"),
        new GraphSettingOption(UserLinksBasic.SCALE_LOG, "Logarithmic"),
    });

    public static final GraphSetting numberFormat = new GraphSetting(GraphSetting.NumberFormatField.NF, UserLinksBasic.GRAPH_NUMBER_FORMAT, "Number format", ".0,K.3");

    public static final GraphSetting green = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.GREEN_QUERY_ARG, "Green", "label", GraphSettingsSource::isGradient);
    public static final GraphSetting yellow = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.YELLOW_QUERY_ARG, "Yellow", "label", GraphSettingsSource::isGradient);
    public static final GraphSetting red = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.RED_QUERY_ARG, "Red", "label", GraphSettingsSource::isGradient);
    public static final GraphSetting violet = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.VIOLET_QUERY_ARG, "Violet", "label", GraphSettingsSource::isGradient);

    public static final GraphSetting greenValue = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.HEATMAP_GREEN_VALUE_QUERY_ARG, "Green", "10k", GraphSettingsSource::isAnyGraphHeatmap);
    public static final GraphSetting yellowValue = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.HEATMAP_YELLOW_VALUE_QUERY_ARG, "Yellow", "20m", GraphSettingsSource::isAnyGraphHeatmap);
    public static final GraphSetting redValue = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.HEATMAP_RED_VALUE_QUERY_ARG, "Red", "30g", GraphSettingsSource::isAnyGraphHeatmap);
    public static final GraphSetting violetValue = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.HEATMAP_VIOLET_VALUE_QUERY_ARG, "Violet", "40g", GraphSettingsSource::isAnyGraphHeatmap);

    public static final GraphSetting limit = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.LIMIT_PARAM, "Limit metric count", "100");

    public static final GraphSetting percentiles = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.PERCENTILE_PARAM, "Percentiles", "50,90,99", "50,90,99", GraphSettingsSource::isOverLinesPercentile);
    public static final GraphSetting ignoreInf = new GraphSetting(GraphSetting.BooleanDropdown.BD, UserLinksBasic.IGNORE_INF_PARAM, "Ignore inf", false, source -> GraphSettingsSource.getRawOrEmpty(source, UserLinksBasic.OVER_LINES_TRANSFORM).equals("WEIGHTED_PERCENTILE"), "On", "Off");
    public static final GraphSetting bucketLabel = new GraphSetting(GraphSetting.TextField.F, UserLinksBasic.BUCKET_LABEL_PARAM, "Bucket label", "bin", "", source -> GraphSettingsSource.getRawOrEmpty(source, UserLinksBasic.OVER_LINES_TRANSFORM).equals("WEIGHTED_PERCENTILE"));

    public static final GraphSetting showGraphsLegend = new GraphSetting(GraphSetting.BooleanDropdown.BD, UserLinksBasic.LEGEND_QUERY_ARG, "Show graphs legend", false);

    public static final GraphSetting filter = new GraphSetting(
        GraphSetting.Dropdown.D,
        UserLinksBasic.FILTER,
        "Filter",
        new GraphSettingOption[] {
            new GraphSettingOption("none", "None"),
            new GraphSettingOption("top", "Top"),
            new GraphSettingOption("bottom", "Bottom")
        }
    );

    public static final GraphSetting filterBy = new GraphSetting(
        GraphSetting.Dropdown.D,
        UserLinksBasic.FILTER_BY,
        "Filter by",
        source -> {
            String filterKind = filter.getFromOrDefault(source);
            return "top".equals(filterKind) || "bottom".equals(filterKind);
        },
        new GraphSettingOption[]{
            new GraphSettingOption("avg", "avg"),
            new GraphSettingOption("min", "min"),
            new GraphSettingOption("max", "max"),
            new GraphSettingOption("sum", "sum"),
            new GraphSettingOption("last", "last"),
            new GraphSettingOption("std", "std"),
            new GraphSettingOption("count", "count"),
            new GraphSettingOption("integrate", "integrate"),
            new GraphSettingOption("random", "random"),
        }
    );

    public static final GraphSetting filterLimit =
        new GraphSetting(
            GraphSetting.TextField.F,
            UserLinksBasic.FILTER_LIMIT,
            "Filter Limit",
            "3",
            "3",
            source -> {
                String filterKind = filter.getFromOrDefault(source);
                return "top".equals(filterKind) || "bottom".equals(filterKind);
            }
        );

    public static final GraphSetting downsampling = new GraphSetting(
        GraphSetting.Dropdown.D,
        UserLinksBasic.DOWNSAMPLING,
        "Downsampling",
        new GraphSettingOption[]{
            new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_AUTO, "Auto"), // Default
            new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_WITHOUT_GRID, "Without grid"),
            new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_BY_INTERVAL, "By interval"),
            new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_BY_POINTS, "By points"),
            new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_OFF, "Off"),
        }
    );

    public static final GraphSetting grid = new GraphSetting(
        GraphSetting.TextField.F,
        UserLinksBasic.GRID,
        "Grid interval",
        "1m30s",
        s -> UserLinksBasic.DOWNSAMPLING_PARAM_BY_INTERVAL.equals(downsampling.getFromOrDefault(s))
    );

    // We must support points and maxPoints parameters both,
    // otherwise "points" from args overrides "points" from graph proto config
    public static final GraphSetting maxPoints = new GraphSetting(
        GraphSetting.TextField.F,
        UserLinksBasic.MAX_POINTS,
        "Max points",
        "100",
        "100",
        s -> UserLinksBasic.DOWNSAMPLING_PARAM_BY_POINTS.equals(downsampling.getFromOrDefault(s))
    );

    public static final GraphSetting downSamplingAggr = new GraphSetting(
        GraphSetting.Dropdown.D,
        UserLinksBasic.DOWNSAMPLING_AGGR,
        "Aggregation function use for downsampling",
        s -> !UserLinksBasic.DOWNSAMPLING_PARAM_OFF.equals(downsampling.getFromOrDefault(s)),
        new GraphSettingOption[]{
            new GraphSettingOption("default", "default"),
            new GraphSettingOption("avg", "avg"),
            new GraphSettingOption("max", "max"),
            new GraphSettingOption("min", "min"),
            new GraphSettingOption("sum", "sum"),
            new GraphSettingOption("last", "last"),
        }
    );

    public static final GraphSetting downsamplingFill = new GraphSetting(
        GraphSetting.Dropdown.D,
        UserLinksBasic.DOWNSAMPLING_FILL,
        "Fill option",
        s -> !UserLinksBasic.DOWNSAMPLING_PARAM_OFF.equals(downsampling.getFromOrDefault(s)),
        new GraphSettingOption[]{
            new GraphSettingOption("null", "Null"),
            new GraphSettingOption("none", "None"),
            new GraphSettingOption("previous", "Previous"),
        }
    );

    public static final GraphSetting ignoreMinStepMillis = new GraphSetting(
        GraphSetting.BooleanDropdown.BD,
        UserLinksBasic.IGNORE_MIN_STEP_MILLIS,
        "Ignore min step interval",
        false,
        s -> !UserLinksBasic.DOWNSAMPLING_PARAM_OFF.equals(downsampling.getFromOrDefault(s)),
        "On",
        "Off");

    public static final GraphSetting movingWindow = new GraphSetting(
        GraphSetting.TextField.F,
        UserLinksBasic.MOVING_WINDOW,
        "Moving window",
        "5m",
        "5m",
        s -> transform.getFromOrDefault(s).startsWith("moving_")
    );

    public static final GraphSetting movingPercentile = new GraphSetting(
        GraphSetting.TextField.F,
        UserLinksBasic.MOVING_PERCENTILE,
        "Percentile",
        "80",
        "80",
        s -> "moving_percentile".equals(transform.getFromOrDefault(s))
    );

    public static final GraphSetting hideNoData = new GraphSetting(
        GraphSetting.BooleanDropdown.BD,
        UserLinksBasic.HIDE_NO_DATA_QUERY_ARG,
        "Hide no data",
        false
    );

    public static final GraphSetting threshold = new GraphSetting(
        GraphSetting.TextField.F,
        "threshold",
        "Threshold",
        "100.2"
    );

    static final GraphSetting[] settings = {
        description,
        graphMode,
        secondaryGraphMode,
        stack,
        scale,
        interpolate,
        aggr,
        dropNans,
        numberFormat,
        cs,
        green,
        yellow,
        red,
        violet,
        greenValue,
        yellowValue,
        redValue,
        violetValue,
        norm,
        min,
        max,
        limit,
        transform,
        movingWindow,
        movingPercentile,
        overLinesTransform,
        percentiles,
        bucketLabel,
        ignoreInf,
        filter,
        filterBy,
        filterLimit,
        downsampling,
        grid,
        maxPoints,
        downSamplingAggr,
        downsamplingFill,
        ignoreMinStepMillis,
        hideNoData,
        threshold,
    };

    static final GraphSetting[] dashboardSettings = {
        showGraphsLegend
    };

    public static GraphSetting[] settings() {
        return settings;
    }

    public static GraphSetting[] dashboardSettings() {
        return dashboardSettings;
    }

    public static List<String> settingNames() {
        ArrayList<String> r = new ArrayList<>();
        for (GraphSetting setting : settings) {
            r.add(setting.queryArg);
        }
        return r;
    }

    public static GraphSetting getByName(String name) {
        for (GraphSetting setting : settings) {
            if (setting.queryArg.equals(name)) {
                return setting;
            }
        }
        throw new IllegalArgumentException("setting not found by name: " + name);
    }
}
