package ru.yandex.solomon.gateway.utils.url;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

/**
 * @author Stepan Koltsov
 */
public class UrlParsed {
    public final String beforeQuery;
    public final ArrayList<QueryArg> queryArgs;

    public UrlParsed(String beforeQuery, ArrayList<QueryArg> queryArgs) {
        this.beforeQuery = beforeQuery;
        this.queryArgs = queryArgs;
    }

    public boolean hasArg(String name) {
        for (QueryArg queryArg : queryArgs) {
            if (queryArg.name.getDecoded().equals(name)) {
                return true;
            }
        }
        return false;
    }

    public String reconstruct() {
        if (queryArgs.isEmpty()) {
            return beforeQuery;
        } else {
            StringBuilder r = new StringBuilder();
            r.append(beforeQuery);
            r.append("?");
            UrlUtils.appendQueryArgsFromNvPairs(r, queryArgs);
            return r.toString();
        }
    }

    public UrlParsed removeQueryArg(String name) {
        ArrayList<QueryArg> r = new ArrayList<>();
        for (QueryArg queryArg : queryArgs) {
            if (queryArg.name.getDecoded().equals(name)) {
                continue;
            }
            r.add(queryArg);
        }
        return new UrlParsed(beforeQuery, r);
    }

    public UrlParsed removeQueryArgs(String... args) {
        UrlParsed r = this;
        for (String arg : args) {
            r = r.removeQueryArg(arg);
        }

        return r;
    }

    public UrlParsed addQueryArg(String name, String value) {
        ArrayList<QueryArg> copy = new ArrayList<>(queryArgs);
        copy.add(QueryArg.decoded(name, value));
        return new UrlParsed(beforeQuery, copy);
    }

    public UrlParsed updateQueryArg(QueryArg update) {
        if (update.name.isEmpty()) {
            throw new IllegalArgumentException("update name is empty");
        }

        ArrayList<QueryArg> copy = new ArrayList<>(queryArgs.size());
        boolean updated = false;
        for (QueryArg queryArg : queryArgs) {
            if (!queryArg.name.getDecoded().equals(update.name.getDecoded())) {
                copy.add(queryArg);
                continue;
            }

            if (updated) {
                continue;
            }

            copy.add(update);

            updated = true;
        }

        if (!updated) {
            copy.add(update);
        }

        return new UrlParsed(beforeQuery, copy);
    }

    public UrlParsed updateQueryArg(String name, String value) {
        return updateQueryArg(QueryArg.decoded(name, value));
    }

    public UrlParsed fillDefault(String name, @Nullable String value) {
        if (value == null) {
            return this;
        }

        if (hasArg(name)) {
            return this;
        } else {
            return addQueryArg(name, value);
        }
    }

    public UrlParsed fillDefaults(List<QueryArg> defaults) {
        UrlParsed r = this;
        for (QueryArg nameValue : defaults) {
            r = r.fillDefault(nameValue.name.getDecoded(), nameValue.value.getDecoded());
        }
        return r;
    }
}
