package ru.yandex.solomon.gateway.utils.url;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class UrlUtils {

    static String encodeArg(String value) {
        try {
            return URLEncoder.encode(value, "utf-8").replace("+", "%20");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    static String decodeArg(String value) {
        try {
            // SOLOMON-1902: URLDecoder.decode() escapes "+" to " "
            return URLDecoder.decode(value, "utf-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    public static LinkedHashMap<String, String> parseUrlQueryArgs(@Nullable String url) {
        if (url == null)
            return new LinkedHashMap<>();

        int searchIndex = url.indexOf('?');
        if (searchIndex < 0)
            return new LinkedHashMap<>();

        return parseQueryArgs(url.substring(searchIndex + 1));
    }

    public static LinkedHashMap<String, String> parseQueryArgs(@Nullable String args) {
        LinkedHashMap<String, String> result = new LinkedHashMap<>();

        if (args == null || args.isEmpty())
            return result;

        for (String pair: args.split("&")) {
            int index = pair.indexOf("=");
            if (index < 0) {
                continue;
            }

            String key = pair.substring(0, index);
            String value = pair.substring(index + 1);
            result.put(decodeArg(key), decodeArg(value));
        }
        return result;
    }

    public static String updateOrRemoveParameter(String url, String name, @Nullable String value) {
        if (value == null) {
            return removeQueryArgs(url, name);
        } else {
            return updateParameter(url, name, value);
        }
    }

    public static String updateParameter(String url, String name, String value) {
        return parseUrl(url).updateQueryArg(QueryArg.decoded(name, value)).reconstruct();
    }

    @ParametersAreNonnullByDefault
    private static class RemoteAndFramgment {
        private final String beforeFragment;
        // starts with slash if not empty
        private final String fragment;

        RemoteAndFramgment(String beforeFragment, String fragment) {
            this.beforeFragment = beforeFragment;
            this.fragment = fragment;
        }

        private static RemoteAndFramgment decode(String url) {
            int hash = url.indexOf('#');
            if (hash >= 0) {
                return new RemoteAndFramgment(url.substring(0, hash), url.substring(hash));
            } else {
                return new RemoteAndFramgment(url, "");
            }
        }
    }

    public static String addParameter(String url, String name, String value) {
        return addParameter(url, QueryArg.decoded(name, value));
    }

    public static String addParameter(String url, QueryArg nameValue) {
        if (url.isEmpty()) {
            throw new IllegalArgumentException("empty URL");
        }

        RemoteAndFramgment remoteAndFramgment = RemoteAndFramgment.decode(url);

        url = remoteAndFramgment.beforeFragment;
        String fragment = remoteAndFramgment.fragment;

        String tail = nameValue.reconstruct();
        if (url.endsWith("?") || url.endsWith("&")) {
            return url + tail + fragment;
        } else if (url.contains("?")) {
            return url + "&" + tail + fragment;
        } else {
            return url + "?" + tail + fragment;
        }
    }

    public static String addParameters(String url, Map<String, String> params) {
        for (Map.Entry<String, String> e : params.entrySet()) {
            url = addParameter(url, e.getKey(), e.getValue());
        }

        return url;
    }

    private static ArrayList<QueryArg> mapToNvPairs(Map<String, String> map) {
        ArrayList<QueryArg> queryArgs = new ArrayList<>();
        for (Map.Entry<String, String> entry : map.entrySet()) {
            queryArgs.add(QueryArg.decoded(entry.getKey(), entry.getValue()));
        }
        return queryArgs;
    }

    public static String makeQueryArgs(Map<String, String> map) {
        return makeQueryArgsFromNvPairs(mapToNvPairs(map));
    }

    private static String makeQueryArgsFromNvPairs(List<QueryArg> queryArgs) {
        StringBuilder r = new StringBuilder();
        appendQueryArgsFromNvPairs(r, queryArgs);
        return r.toString();
    }

    static void appendQueryArgsFromNvPairs(StringBuilder r, List<QueryArg> queryArgs) {
        for (int i = 0; i < queryArgs.size(); i++) {
            if (i != 0) {
                r.append("&");
            }
            QueryArg queryArg = queryArgs.get(i);
            queryArg.reconstructTo(r);
        }
    }

    public static String fillDefaultsMap(String url, Map<String, String> defaults) {
        return fillDefaults(url, mapToNvPairs(defaults));
    }

    private static String fillDefaults(String url, List<QueryArg> defaults) {
        return parseUrl(url).fillDefaults(defaults).reconstruct();
    }

    public static String fillDefault(String url, String name, String value) {
        return fillDefaults(url, Collections.singletonList(QueryArg.decoded(name, value)));
    }


    private static ArrayList<QueryArg> parseQueryArgsImpl(String queryArgs) {
        ArrayList<QueryArg> r = new ArrayList<>();

        String[] parts = queryArgs.split("\\&");
        for (String part : parts) {
            if (part.isEmpty()) {
                continue;
            }

            int eq = part.indexOf('=');
            if (eq < 0) {
                r.add(QueryArg.encoded(part, ""));
            } else {
                String name = part.substring(0, eq);
                String value = part.substring(eq + 1);
                r.add(QueryArg.encoded(name, value));
            }
        }
        return r;
    }

    // TODO: hash
    public static UrlParsed parseUrl(String url) {
        int q = url.indexOf('?');
        if (q < 0) {
            return new UrlParsed(url, new ArrayList<>());
        }

        return new UrlParsed(url.substring(0, q), parseQueryArgsImpl(url.substring(q + 1)));
    }

    public static String removeQueryArgs(String url, String... names) {
        int q = url.indexOf('?');
        if (q < 0) {
            return url;
        }

        return parseUrl(url).removeQueryArgs(names).reconstruct();
    }
}
