package ru.yandex.solomon.gateway.viewHistory;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Triple;

import ru.yandex.solomon.core.db.model.ViewHistory;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.utils.url.QueryArg;
import ru.yandex.solomon.gateway.utils.url.UrlParsed;
import ru.yandex.solomon.util.ComparableUtils;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;
import static ru.yandex.solomon.gateway.utils.url.UrlUtils.parseUrlQueryArgs;
import static ru.yandex.solomon.gateway.viewHistory.ViewHistoryUrlNormalizer.normalizeUrl;


/**
 * @author albazh
 */
@ParametersAreNonnullByDefault
public class ViewHistoryUrlUtils {

    public static Predicate<ViewHistory> matchByUrl(String url) {
        String normalizeUrl = normalizeUrl(url);
        return history -> normalizeUrl.equals(normalizeUrl(history.getUrl()));
    }

    public static Predicate<ViewHistory> matchByUrlPcsArgs(String url) {
        Triple<String, String, String> pcs = pcs(url);

        return history -> equalPcs(pcs, pcs(history.getUrl()));
    }

    static UrlParsed sortQueryArgsByName(UrlParsed urlParsed) {
        ArrayList<QueryArg> queryArgs = new ArrayList<>(urlParsed.queryArgs);
        if (!isQueryArgsSortedByName(queryArgs)) {
            queryArgs.sort(ViewHistoryUrlUtils::compare);
        }
        return new UrlParsed(urlParsed.beforeQuery, queryArgs);
    }

    static boolean isQueryArgsSortedByName(ArrayList<QueryArg> queryArgs) {
        for (int i = 1; i < queryArgs.size(); i++) {
            if (ViewHistoryUrlUtils.compare(queryArgs.get(i - 1), queryArgs.get(i)) > 0) {
                return false;
            }
        }
        return true;
    }

    static boolean equalPcs(Triple<String, String, String> first, Triple<String, String, String> second) {
        return equalPcsPart(first.getLeft(), second.getLeft()) &&
            equalPcsPart(first.getMiddle(), second.getMiddle()) &&
            equalPcsPart(first.getRight(), second.getRight());
    }

    private static int compare(QueryArg a, QueryArg b) {
        return ComparableUtils
            .compare(a.name.getDecoded(), a.value.getDecoded(), b.name.getDecoded(), b.value.getDecoded());
    }

    private static boolean equalPcsPart(String first, String second) {
        return StringUtils.isEmpty(first) || StringUtils.isEmpty(second) || first.equals(second);
    }

    static Triple<String, String, String> pcs(String url) {
        LinkedHashMap<String, String> queryArgs = parseUrlQueryArgs(normalizeUrl(url));

        String project = defaultIfEmpty(queryArgs.get(UserLinksBasic.PROJECT), EMPTY);
        String cluster = defaultIfEmpty(queryArgs.get(UserLinksBasic.CLUSTER), EMPTY);
        String service = defaultIfEmpty(queryArgs.get(UserLinksBasic.SERVICE), EMPTY);

        return Triple.of(project, cluster, service);
    }
}
