package ru.yandex.solomon.gateway.www;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.gateway.backend.client.data.WwwDataApiGet;
import ru.yandex.solomon.gateway.backend.client.data.WwwDataApiMetric;
import ru.yandex.solomon.gateway.backend.client.www.page.DownSamplingParams;
import ru.yandex.solomon.gateway.backend.storage.LocalDataApiManager;
import ru.yandex.solomon.gateway.utils.UserLinks;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.util.time.Interval;


/**
 * @author Sergey Polovko
 */
@Import({
    LocalDataApiManager.class,
})
@Component
public class BackendManager {

    private static final int METRICS_REQUEST_LIMIT = 100000;

    private final LocalDataApiManager dataApiManager;
    private final SolomonConfHolder solomonConfHolder;

    @Autowired
    public BackendManager(
            LocalDataApiManager dataApiManager,
            SolomonConfHolder solomonConfHolder)
    {
        this.dataApiManager = dataApiManager;
        this.solomonConfHolder = solomonConfHolder;
    }

    public CompletableFuture<WwwDataApiGet> processGet(
        ShardKey shardKey,
        Interval interval,
        DownSamplingParams downsamplingParams,
        @Nullable String clusterId,
        Map<String, String> params,
        String subjectId)
    {
        ShardConfDetailed shardConf = solomonConfHolder.getConfOrThrow().findShardByKey(shardKey);
        Deadline deadline = Deadline.fromThreadLocal("dataApiGet");
        return dataApiManager.processGet(shardConf, interval, downsamplingParams, params, clusterId, deadline, subjectId);
    }

    public CompletableFuture<List<WwwDataApiMetric>> processMetrics(
        ShardKey shardKey,
        Map<String, String> params)
    {
        Map<String, String> clearedParams = UserLinks.cleanParams(params);

        Deadline deadline = Deadline.fromThreadLocal("findMetricsDetailed");
        return dataApiManager.findMetrics(shardKey, clearedParams, deadline).thenApply(
            metrics -> {
                if (metrics.size() > METRICS_REQUEST_LIMIT) {
                    throw new RuntimeException("found more than " + METRICS_REQUEST_LIMIT + " metrics");
                } else {
                    return metrics;
                }
            }
        );
    }
}
