package ru.yandex.solomon.gateway.www;

import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.gateway.data.MetricsClientSubjectMetrics;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;

/**
 * @author Alexey Trushkin
 */
@RestController
@Import({
        ManagerWriterContext.class,
})
public class GatewaySubjectInfoWww {

    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final ManagerWriterContext context;
    private final MetricsClientSubjectMetrics metrics;

    public GatewaySubjectInfoWww(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            ManagerWriterContext context,
            MetricsClientSubjectMetrics metrics)
    {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.context = context;
        this.metrics = metrics;
    }

    @Bean
    public RootLink gatewayLocalSubjectsLinks() {
        return new RootLink("/local-subjects", "Local subjects");
    }

    @RequestMapping(value = "/local-subjects", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> localSubjects(
            @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
            @RequestParam(value = "limit", defaultValue = "25") int limit,
            ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
                .thenCompose(authorizer::authorize)
                .thenApply(account -> localSubjectsImpl(sortBy, limit));
    }

    private String localSubjectsImpl(int sortBy, int limit) {
        List<Record> records = metrics.getSubjectsInfo().stream()
                .map(Record::of)
                .sorted((o1, o2) -> String.CASE_INSENSITIVE_ORDER.compare(o1.subjectId, o2.subjectId))
                .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Local subjects(" + DataSize.shortString(records.size()) + ")", context, columns, records, sortBy, limit).genString();
    }

    private List<Column<Record>> makeColumns() {
        return List.of(
                Column.of(
                        "SubjectId",
                        r -> r.subjectId,
                        (o1, o2) -> String.CASE_INSENSITIVE_ORDER.compare(o1.subjectId, o2.subjectId)),
                Column.of(
                        "ReadBytes/sec",
                        r -> Long.toString(r.readBytes),
                        Comparator.comparingLong(o -> o.readBytes)),
                Column.of(
                        "ReadPoints/sec",
                        r -> Long.toString(r.readPoints),
                        Comparator.comparingLong(o -> o.readPoints)),
                Column.of(
                        "ReadTimeseries/sec",
                        r -> Long.toString(r.readTimeseries),
                        Comparator.comparingLong(o -> o.readTimeseries))
        );
    }

    private static class Record implements TableRecord {
        String subjectId;
        long readBytes;
        long readPoints;
        long readTimeseries;

        public static Record of(MetricsClientSubjectMetrics.SubjectInfo subjectInfo) {
            var r = new Record();
            r.subjectId = subjectInfo.subjectId();
            r.readBytes = Math.round(subjectInfo.readBytesRate());
            r.readPoints = Math.round(subjectInfo.readPointsRate());
            r.readTimeseries = Math.round(subjectInfo.readTimeseriesRate());
            return r;
        }
    }
}
