package ru.yandex.solomon.gateway.www.time;

import java.time.Duration;
import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.gateway.www.ser.WwwIntervalSerializer;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class UiInstant {

    /** Null means now */
    @Nullable
    private final Instant instant;
    @Nullable
    private final Duration relative;

    private UiInstant(@Nullable Instant instant, @Nullable Duration relative) {
        this.instant = instant;
        this.relative = relative;
    }

    @Nullable
    public Instant getInstant() {
        return instant;
    }

    @Nullable
    public Duration getRelative() {
        return relative;
    }

    public boolean isBefore(UiInstant that) {
        return this.dispatch(new Dispatch<Boolean>() {
            @Override
            public Boolean instant(@Nonnull Instant thisInstant) {
                return that.dispatch(new Dispatch<Boolean>() {
                    @Override
                    public Boolean instant(@Nonnull Instant thatInstant) {
                        return thisInstant.isBefore(thatInstant);
                    }

                    @Override
                    public Boolean relative(@Nonnull Duration thatRelative) {
                        return false;
                    }
                });
            }

            @Override
            public Boolean relative(@Nonnull Duration thisRelative) {
                return that.dispatch(new Dispatch<Boolean>() {
                    @Override
                    public Boolean instant(@Nonnull Instant thatInstant) {
                        return false;
                    }

                    @Override
                    public Boolean relative(@Nonnull Duration thatRelative) {
                        return thisRelative.compareTo(thatRelative) < 0;
                    }
                });
            }
        });
    }

    public interface Dispatch<R> {
        R instant(@Nonnull Instant instant);
        R relative(@Nonnull Duration relative);
    }

    public <R> R dispatch(Dispatch<R> dispatch) {
        if (instant != null) {
            return dispatch.instant(instant);
        } else if (relative != null) {
            return dispatch.relative(relative);
        } else {
            throw new RuntimeException("not instant and not relative in UI instant");
        }
    }

    public static UiInstant instant(@Nonnull Instant instant) {
        return new UiInstant(instant, null);
    }

    public static UiInstant relativeForward(@Nonnull Duration duration) {
        return new UiInstant(null, duration);
    }

    public static UiInstant relativeBackwards(@Nonnull Duration duration) {
        return relativeForward(duration.negated());
    }

    public static UiInstant now() {
        return relativeForward(Duration.ZERO);
    }


    public Instant toInstant(@Nonnull Instant now) {
        return dispatch(new Dispatch<Instant>() {
            @Override
            public Instant instant(@Nonnull Instant instant) {
                return instant;
            }

            @Override
            public Instant relative(@Nonnull Duration relative) {
                return now.plus(relative);
            }
        });
    }

    public UiInstant plus(@Nonnull Duration duration) {
        return dispatch(new Dispatch<UiInstant>() {
            @Override
            public UiInstant instant(@Nonnull Instant instant) {
                return UiInstant.instant(instant.plus(duration));
            }

            @Override
            public UiInstant relative(@Nonnull Duration relative) {
                return UiInstant.relativeForward(relative.plus(duration));
            }
        });
    }

    public UiInstant minus(Duration duration) {
        return plus(duration.negated());
    }

    @Override
    public boolean equals(@Nullable Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        UiInstant uiInstant = (UiInstant) o;

        if (instant != null ? !instant.equals(uiInstant.instant) : uiInstant.instant != null) return false;
        if (relative != null ? !relative.equals(uiInstant.relative) : uiInstant.relative != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = instant != null ? instant.hashCode() : 0;
        result = 31 * result + (relative != null ? relative.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return WwwIntervalSerializer.uiInstantToString(this);
    }
}
