package ru.yandex.solomon.gateway.www.time;

import java.time.Duration;
import java.time.Instant;

import javax.annotation.Nonnull;

import ru.yandex.solomon.util.time.Interval;

/**
 * @author Stepan Koltsov
 */
public class UiInterval {
    /** Relative to now */
    @Nonnull
    private final UiInstant begin;
    /** relative to now */
    @Nonnull
    private final UiInstant end;

    public UiInterval(@Nonnull UiInstant begin, @Nonnull UiInstant end) {
        this.begin = begin;
        this.end = end;

        if (duration().isNegative()) {
            throw new RuntimeException("negative interval: " + begin + ", " + end);
        }
    }

    public UiInterval(Duration begin, Duration end) {
        this(UiInstant.relativeForward(begin), UiInstant.relativeForward(end));
    }

    public static UiInterval toNow(Duration distance) {
        if (distance.isNegative()) {
            throw new IllegalArgumentException("negative distance: " + distance);
        }
        return new UiInterval(UiInstant.relativeBackwards(distance), UiInstant.now());
    }

    public static UiInterval between(UiInstant a, UiInstant b) {
        if (b.isBefore(a)) {
            return new UiInterval(b, a);
        } else {
            return new UiInterval(a, b);
        }
    }

    @Nonnull
    public UiInstant getBegin() {
        return begin;
    }

    @Nonnull
    public UiInstant getEnd() {
        return end;
    }

    public Interval toInterval(@Nonnull Instant now) {
        return new Interval(begin.toInstant(now), end.toInstant(now));
    }

    public Duration duration() {
        return begin.dispatch(new UiInstant.Dispatch<Duration>() {
            @Override
            public Duration instant(@Nonnull Instant beginInstant) {
                return end.dispatch(new UiInstant.Dispatch<Duration>() {
                    @Override
                    public Duration instant(@Nonnull Instant endInstant) {
                        return Duration.between(beginInstant, endInstant);
                    }

                    @Override
                    public Duration relative(@Nonnull Duration endRelative) {
                        return endRelative;
                    }
                });
            }

            @Override
            public Duration relative(@Nonnull Duration beginRelative) {
                return end.dispatch(new UiInstant.Dispatch<Duration>() {
                    @Override
                    public Duration instant(@Nonnull Instant endInstant) {
                        return beginRelative.negated();
                    }

                    @Override
                    public Duration relative(@Nonnull Duration endRelative) {
                        return endRelative.minus(beginRelative);
                    }
                });
            }
        });
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        UiInterval that = (UiInterval) o;

        if (!begin.equals(that.begin)) return false;
        if (!end.equals(that.end)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = begin.hashCode();
        result = 31 * result + end.hashCode();
        return result;
    }
}
