package ru.yandex.solomon.gateway.api.cloud.priv.v2;

import java.time.Instant;
import java.util.concurrent.CompletionException;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import yandex.cloud.priv.quota.PQ.BatchUpdateQuotaMetricsRequest;
import yandex.cloud.priv.quota.PQ.GetQuotaDefaultRequest;
import yandex.cloud.priv.quota.PQ.GetQuotaDefaultResponse;
import yandex.cloud.priv.quota.PQ.GetQuotaRequest;
import yandex.cloud.priv.quota.PQ.MetricLimit;
import yandex.cloud.priv.quota.PQ.Quota;
import yandex.cloud.priv.quota.PQ.QuotaMetric;
import yandex.cloud.priv.quota.PQ.UpdateQuotaMetricRequest;

import ru.yandex.solomon.core.db.dao.QuotasDao;
import ru.yandex.solomon.core.db.dao.memory.InMemoryQuotasDao;
import ru.yandex.solomon.quotas.manager.QuotaManager;
import ru.yandex.solomon.quotas.manager.fetcher.ManualAlertingQuotaFetcherStub;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.misc.concurrent.CompletableFutures.unwrapCompletionException;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class QuotaServiceTest {
    private QuotaService quotaService;
    private QuotaManager quotaManager;
    private ManualAlertingQuotaFetcherStub alertingQuotaFetcher;
    private QuotasDao quotasDao;

    @Before
    public void setUp() {
        quotasDao = new InMemoryQuotasDao();
        quotasDao.createSchemaForTests();
        alertingQuotaFetcher = new ManualAlertingQuotaFetcherStub();
        quotaManager = new QuotaManager(quotasDao, alertingQuotaFetcher);
        quotaService = new QuotaServiceImpl(quotaManager);
    }

    @After
    public void tearDown() {
        quotasDao.dropSchemaForTests();
    }

    private Quota syncGet(GetQuotaRequest request) {
        return join(quotaService.get(request));
    }

    private GetQuotaDefaultResponse syncGetDefault(GetQuotaDefaultRequest request) {
        return join(quotaService.getDefault(request));
    }

    private void syncUpdateMetric(UpdateQuotaMetricRequest request, String updatedBy, Instant updatedAt) {
        join(quotaService.updateMetric(request, updatedBy, updatedAt));
    }

    private void syncBatchUpdateMetric(BatchUpdateQuotaMetricsRequest request, String updatedBy, Instant updatedAt) {
        join(quotaService.batchUpdateMetrics(request, updatedBy, updatedAt));
    }

    private static QuotaMetric quotaMetric(String name, long value, long limit) {
        return QuotaMetric.newBuilder()
            .setName(name)
            .setLimit(limit)
            .setValue(value)
            .setUsage(value)
            .build();
    }

    private static MetricLimit metricLimit(String name, long limit) {
        return MetricLimit.newBuilder()
            .setName(name)
            .setLimit(limit)
            .build();
    }

    private void populate() {
        quotaManager.updateDefaultLimit("alerting", "project", "alert.count", 200);
        quotaManager.updateDefaultLimit("alerting", "project", "subalert.count", 2000);
        quotaManager.updateLimit("alerting", "project", "solomon", "subalert.count", 5000);

        alertingQuotaFetcher.getAlertCounter("junk").addAndGet(42);
        alertingQuotaFetcher.getSubalertCounter("junk").addAndGet(100500);

        alertingQuotaFetcher.getAlertCounter("solomon").addAndGet(10);
    }

    @Test
    public void testGet() {
        populate();

        var getJunk = syncGet(GetQuotaRequest.newBuilder()
                .setCloudId("junk")
                .build());

        var getSolomon = syncGet(GetQuotaRequest.newBuilder()
                .setCloudId("solomon")
                .build());

        assertThat(getJunk, equalTo(
            Quota.newBuilder()
                .setCloudId("junk")
                .addMetrics(quotaMetric("monitoring.alert.count", 42, 200))
                .addMetrics(quotaMetric("monitoring.subalert.count", 100500, 2000))
                .build()));

        assertThat(getSolomon, equalTo(
            Quota.newBuilder()
                .setCloudId("solomon")
                .addMetrics(quotaMetric("monitoring.alert.count", 10, 200))
                .addMetrics(quotaMetric("monitoring.subalert.count", 0, 5000))
                .build()));
    }

    @Test
    public void testGetDefaults() {
        populate();

        quotaManager.updateDefaultLimit("alerting", "project", "alert.count", 200);
        quotaManager.updateDefaultLimit("alerting", "project", "subalert.count", 2000);
        quotaManager.updateLimit("alerting", "project", "solomon", "subalert.count", 5000);

        alertingQuotaFetcher.getAlertCounter("junk").addAndGet(42);
        alertingQuotaFetcher.getSubalertCounter("junk").addAndGet(100500);

        alertingQuotaFetcher.getAlertCounter("solomon").addAndGet(10);

        var defaults = syncGetDefault(GetQuotaDefaultRequest.getDefaultInstance());

        assertThat(defaults, equalTo(
            GetQuotaDefaultResponse.newBuilder()
                .addMetrics(metricLimit("monitoring.alert.count", 200))
                .addMetrics(metricLimit("monitoring.subalert.count", 2000))
                .build()));
    }

    @Test
    public void testUpdateMetric() {
        populate();

        syncUpdateMetric(UpdateQuotaMetricRequest.newBuilder()
            .setCloudId("solomon")
            .setMetric(metricLimit("monitoring.alert.count", 100500))
            .build(),
            "uranix",
            Instant.now());

        syncUpdateMetric(UpdateQuotaMetricRequest.newBuilder()
                .setCloudId("junk")
                .setMetric(metricLimit("monitoring.channel.count", 42))
                .build(),
            "uranix",
            Instant.now());

        var getJunk = syncGet(GetQuotaRequest.newBuilder()
            .setCloudId("junk")
            .build());

        var getSolomon = syncGet(GetQuotaRequest.newBuilder()
            .setCloudId("solomon")
            .build());

        assertThat(getJunk, equalTo(
            Quota.newBuilder()
                .setCloudId("junk")
                .addMetrics(quotaMetric("monitoring.alert.count", 42, 200))
                .addMetrics(quotaMetric("monitoring.subalert.count", 100500, 2000))
                .addMetrics(quotaMetric("monitoring.channel.count", 0, 42))
                .build()));

        assertThat(getSolomon, equalTo(
            Quota.newBuilder()
                .setCloudId("solomon")
                .addMetrics(quotaMetric("monitoring.alert.count", 10, 100500))
                .addMetrics(quotaMetric("monitoring.subalert.count", 0, 5000))
                .build()));
    }

    @Test(expected = IllegalArgumentException.class)
    public void testBadMetric() throws Throwable {
        populate();

        try {
            syncUpdateMetric(UpdateQuotaMetricRequest.newBuilder()
                            .setCloudId("junk")
                            .setMetric(metricLimit("monitoring.puppies.count", 42))
                            .build(),
                    "uranix",
                    Instant.now());
        } catch (CompletionException e) {
            throw unwrapCompletionException(e);
        }
    }

    @Test
    public void testBatchUpdateMetric() {
        populate();

        syncBatchUpdateMetric(BatchUpdateQuotaMetricsRequest.newBuilder()
                        .setCloudId("solomon")
                        .addMetrics(metricLimit("monitoring.alert.count", 100500))
                        .addMetrics(metricLimit("monitoring.channel.count", 42))
                        .build(),
                "uranix",
                Instant.now());

        var getJunk = syncGet(GetQuotaRequest.newBuilder()
                .setCloudId("junk")
                .build());

        var getSolomon = syncGet(GetQuotaRequest.newBuilder()
                .setCloudId("solomon")
                .build());

        assertThat(getJunk, equalTo(
                Quota.newBuilder()
                        .setCloudId("junk")
                        .addMetrics(quotaMetric("monitoring.alert.count", 42, 200))
                        .addMetrics(quotaMetric("monitoring.subalert.count", 100500, 2000))
                        .build()));

        assertThat(getSolomon, equalTo(
                Quota.newBuilder()
                        .setCloudId("solomon")
                        .addMetrics(quotaMetric("monitoring.alert.count", 10, 100500))
                        .addMetrics(quotaMetric("monitoring.channel.count", 0, 42))
                        .addMetrics(quotaMetric("monitoring.subalert.count", 0, 5000))
                        .build()));
    }
}
