package ru.yandex.solomon.gateway.api.cloud.v1;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Month;
import java.time.ZoneOffset;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.DcMetricsClient;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.SolomonClientStub;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.util.time.Deadline;
import ru.yandex.solomon.util.time.Interval;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.alert.protobuf.TEvaluationStatus.ECode.ALARM;
import static ru.yandex.solomon.alert.protobuf.TEvaluationStatus.ECode.OK;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertingHistoryReaderTest {

    private SolomonClientStub solomon;
    private MetricsClient metricsClient;
    private AlertingHistoryReader reader;

    @Before
    public void setUp() {
        solomon = new SolomonClientStub();
        metricsClient = new DcMetricsClient("test", solomon.getMetabase(), solomon.getStockpile());
        reader = new AlertingHistoryReader(metricsClient, new ShardKey("solomon", "production", "alerting_statuses"));
    }

    @Test
    public void absent() {
        var result = reader.readEvaluationHistory(
            "absent",
            "my-alert",
            interval("00:00", "01:00"),
            60_000,
            deadline(),
            "subjectId",
            false, false)
            .join();

        assertEquals(0, result.timestamps.length);
        assertEquals(0, result.ok.length);
        assertEquals(0, result.warn.length);
        assertEquals(0, result.alarm.length);
        assertEquals(0, result.noData.length);
        assertEquals(0, result.error.length);
    }

    @Test
    public void absentChannel() {
        // No events -> no metric in solomon
        var result = reader.readChannelStatusesHistory(
            "absent",
            "deadbeef",
            interval("00:00", "01:00"),
            60_000,
            deadline(),
            "subjectId")
            .join();
        assertThat(result.statuses.size(), equalTo(0));
        assertThat(result.timestamps.length, equalTo(0));
    }

    @Test
    public void readOne() {
        solomon.addMetric(Labels.builder()
            .add("project", "solomon")
            .add("cluster", "production")
            .add("service", "alerting_statuses")
            .add("sensor", "alert.evaluation.status")
            .add("projectId", "my_cloud")
            .add("alertId", "my_alert")
            .build(),
            AggrGraphDataArrayList.of(
                point("00:10:00", ALARM),
                point("00:10:30", ALARM),
                point("00:11:00", ALARM),
                point("00:11:30", OK),
                point("00:12:00", OK)));

        solomon.addMetric(Labels.builder()
                .add("project", "solomon")
                .add("cluster", "production")
                .add("service", "alerting_statuses")
                .add("sensor", "alert.evaluation.status")
                .add("projectId", "my_cloud")
                .add("alertId", "noise")
                .build(),
            AggrGraphDataArrayList.of(point("00:12:00", OK)));

        {
            var result = reader.readEvaluationHistory(
                "my_cloud",
                "my_alert",
                interval("00:10", "00:12"),
                60_000,
                deadline(),
                "subjectId",
                false, false)
                .join();

            assertEquals(2, result.timestamps.length);

            assertEquals(1, result.alarm[0]);
            assertEquals(0, result.ok[0]);

            assertEquals(0, result.alarm[1]);
            assertEquals(1, result.ok[1]);
        }

        {
            var result = reader.readEvaluationHistory(
                "my_cloud",
                "my_alert",
                interval("00:10", "00:15"),
                60_000,
                deadline(),
                "subjectId",
                false, false)
                .join();

            assertEquals(3, result.timestamps.length);
            assertEquals(1, result.alarm[0]);
            assertEquals(1, result.ok[1]);
            assertEquals(1, result.ok[2]);
        }

        {
            var result = reader.readEvaluationHistory(
                "my_cloud",
                "my_alert",
                interval("00:15", "00:20"),
                60_000,
                deadline(),
                "subjectId",
                false, false)
                .join();

            assertEquals(0, result.timestamps.length);
        }
    }

    private Labels makeLabelsForStatus(String channelId, String status) {
        return Labels.builder()
            .add("project", "solomon")
            .add("cluster", "production")
            .add("service", "alerting_statuses")
            .add("sensor", "channel.notification.status")
            .add("projectId", "my_cloud")
            .add("channelId", channelId)
            .add("status", status)
            .build();
    }

    @Test
    public void readStatuses() {
        solomon.addMetric(makeLabelsForStatus("channel1", "SUCCESS"),
            AggrGraphDataArrayList.of(
                point("00:10:00", 0),
                point("00:10:30", 12),
                point("00:11:00", 13),
                point("00:11:30", 18),
                point("00:12:00", 18)));

        solomon.addMetric(makeLabelsForStatus("channel1", "ERROR"),
            AggrGraphDataArrayList.of(
                point("00:11:00", 5),
                point("00:12:00", 5)));

        solomon.addMetric(makeLabelsForStatus("channel1", "RESOURCE_EXHAUSTED"),
            AggrGraphDataArrayList.of(
                point("00:09:30", 8),
                point("00:10:00", 10),
                point("00:11:00", 1),
                point("00:11:30", 5)));

        solomon.addMetric(makeLabelsForStatus("spam", "SUCCESS"),
            AggrGraphDataArrayList.of(
                point("00:10:00", 100500),
                point("00:10:30", 100600),
                point("00:11:00", 100700),
                point("00:11:30", 100800),
                point("00:12:00", 100900)));

        {
            var result = reader.readChannelStatusesHistory(
                "my_cloud",
                "channel1",
                interval("00:10", "00:12"),
                30_000,
                deadline(),
                "subjectId")
                .join();

            assertEquals(4, result.timestamps.length);

            assertThat(result.statuses.get("SUCCESS"), equalTo(new long[] {0, 12, 1, 5}));
            assertThat(result.statuses.get("ERROR"), equalTo(new long[] {0, 0, 5, 0}));
            assertThat(result.statuses.get("RESOURCE_EXHAUSTED"), equalTo(new long[] {2, 0, 1, 4}));

        }

        {
            var result = reader.readChannelStatusesHistory(
                "my_cloud",
                "channel1",
                interval("00:10", "00:15"),
                60_000,
                deadline(),
                "subjectId")
                .join();

            assertEquals(5, result.timestamps.length);

            assertThat(result.statuses.get("SUCCESS"), equalTo(new long[] {12, 6, 0, 0, 0}));
            assertThat(result.statuses.get("ERROR"), equalTo(new long[] {0, 5, 0, 0, 0}));
            assertThat(result.statuses.get("RESOURCE_EXHAUSTED"), equalTo(new long[] {2, 5, 0, 0, 0}));
        }

        {
            var result = reader.readChannelStatusesHistory(
                "my_cloud",
                "channel1",
                interval("00:15", "00:20"),
                60_000,
                deadline(),
                "subjectId")
                .join();

            assertEquals(0, result.timestamps.length);

            assertEquals(0, result.statuses.get("SUCCESS").length);
            assertEquals(0, result.statuses.get("ERROR").length);
            assertEquals(0, result.statuses.get("RESOURCE_EXHAUSTED").length);
        }
    }

    private long deadline() {
        return System.currentTimeMillis() + Deadline.DEFAULT_TIMEOUT_MILLIS;
    }

    private static AggrPoint point(String time, TEvaluationStatus.ECode code) {
        return point(time, code.getNumber());
    }

    private static AggrPoint point(String time, long value) {
        return AggrPoint.builder()
            .time(ts(time))
            .longValue(value)
            .count(1)
            .build();
    }

    private Interval interval(String from, String to) {
        return Interval.millis(ts(from), ts(to));
    }

    private static long ts(String time) {
        return LocalDateTime.of(LocalDate.of(2000, Month.JANUARY, 1), LocalTime.parse(time))
            .toInstant(ZoneOffset.UTC)
            .toEpochMilli();
    }
}
