package ru.yandex.solomon.gateway.api.cloud.v1;

import java.time.Instant;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.discovery.cluster.ClusterMapperStub;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.fake.AnonymousAuthorizer;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolverStub;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.core.conf.SolomonConfStub;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.gateway.api.cloud.common.ProgramValidationException;
import ru.yandex.solomon.gateway.api.v2.dto.data.DataResultDto;
import ru.yandex.solomon.gateway.cloud.api.RequestProducerResolver;
import ru.yandex.solomon.gateway.data.DataClient;
import ru.yandex.solomon.gateway.data.DataClientMetrics;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.DcMetricsClient;
import ru.yandex.solomon.metrics.client.SolomonClientStub;
import ru.yandex.solomon.model.point.AggrPoints;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.misc.concurrent.CompletableFutures.unwrapCompletionException;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class CloudDataReadControllerTest {
    private SolomonClientStub solomon;

    private DataClient dataClient;
    private CloudAuthorizer authorizer;
    private RequestProducerResolver producerResolver;
    private CloudDataReadController controller;
    private SolomonConfHolder confHolder;
    private SolomonConfStub conf;

    private final static String folderId = "myFolder";
    private final static String cloudId = "myCloud";

    private static class ProducerResolverStub implements RequestProducerResolver {
        @Override
        public RequestProducer resolve(@Nullable String header) {
            return RequestProducer.REQUEST_PRODUCER_UNSPECIFIED;
        }
    }

    @Before
    public void setUp() {
        solomon = new SolomonClientStub();
        var metricsClient = new DcMetricsClient("test", solomon.getMetabase(), solomon.getStockpile());
        var mapper = new ClusterMapperStub();
        var registry = new MetricRegistry();
        confHolder = new SolomonConfHolder();
        conf = new SolomonConfStub();
        var featureFlagsHolder = new FeatureFlagHolderStub();
        var dataClientMetrics = new DataClientMetrics();
        dataClient = new DataClient(dataClientMetrics, metricsClient, mapper, registry, confHolder, featureFlagsHolder);
        var resolver = new CloudByFolderResolverStub();
        resolver.add(folderId, cloudId);
        authorizer = new CloudAuthorizer(new AnonymousAuthorizer(), Optional.of(resolver));
        producerResolver = new ProducerResolverStub();
        controller = new CloudDataReadController(dataClient, authorizer, producerResolver);

        solomon.addMetric(Labels.of("project", cloudId, "cluster", folderId, "service", "custom",
                "name", "cpu_usage", "host", "solomon"),
                MetricType.DGAUGE,
                AggrGraphDataArrayList.of(
                        AggrPoints.dpoint("2020-01-01T00:01:00Z", 42),
                        AggrPoints.dpoint("2020-01-01T00:02:00Z", 43),
                        AggrPoints.dpoint("2020-01-01T00:05:00Z", 48)
                ));

        solomon.addMetric(Labels.of("project", cloudId, "cluster", folderId, "service", "custom",
                "name", "cpu_usage", "host", "kikimr"),
                MetricType.DGAUGE,
                AggrGraphDataArrayList.of(
                        AggrPoints.dpoint("2020-01-01T00:01:00Z", 12),
                        AggrPoints.dpoint("2020-01-01T00:02:00Z", 13),
                        AggrPoints.dpoint("2020-01-01T00:05:00Z", 28)
                ));

        for (String bin : List.of("100", "200", "500", "1000")) {
            solomon.addMetric(Labels.of("project", cloudId, "cluster", folderId, "service", "custom",
                    "name", "responseMs", "bin", bin),
                    MetricType.DGAUGE,
                    AggrGraphDataArrayList.of(
                            AggrPoints.dpoint("2020-01-01T00:01:00Z", 12),
                            AggrPoints.dpoint("2020-01-01T00:02:00Z", 13),
                            AggrPoints.dpoint("2020-01-01T00:05:00Z", 28)
                    ));
        }

        addShard(ShardKey.create(cloudId, folderId, "custom"), 15, 60);
    }

    private void addShard(ShardKey key, int fetchIntervalSec, int gridSec) {
        String shardId = key.getProject() + "_" + key.getCluster() + "_" + key.getService();
        conf.addService(conf.service(key.getProject(), key.getService())
                .toBuilder()
                .setName(key.getService())
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                        null,
                        gridSec,
                        37,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new ServiceMetricConf.AggrRule[0], false),
                        fetchIntervalSec))
                .build());
        conf.addCluster(conf.cluster(key.getProject(), key.getCluster())
                .toBuilder()
                .setName(key.getCluster())
                .build());
        conf.addShard(conf.shard(key.getProject(), shardId)
                .toBuilder()
                .setClusterId(key.getCluster())
                .setClusterName(key.getCluster())
                .setServiceId(key.getService())
                .setServiceName(key.getService())
                .build());
        confHolder.onConfigurationLoad(conf.snapshot());
    }

    private DataResultDto syncReadDataByQuery(String query) {
        return join(controller.readDataFromText(AnonymousAuthSubject.INSTANCE, folderId,
                Instant.parse("2020-01-01T00:00:00Z"), Instant.parse("2020-01-01T00:10:00Z"),
                0, 0, Aggregation.DEFAULT_AGGREGATION, OperationDownsampling.FillOption.NULL,
                query, ""));
    }

    @Test
    public void validRequests() {
        List<String> queries = List.of(
                "cpu_usage{host=solomon}",
                "sqr(cpu_usage{host=solomon})",
                "responseMs{}",
                "asap(histogram_percentile([50, 90, 95], responseMs{}))",
                "shift(line{}, -1h)",
                "cpu_usage{host=solomon} * 1000",
                "cpu_usage{host=solomon} - cpu_usage{host=solomon}",
                "tail(cpu_usage{host=solomon}, size(cpu_usage{host=kikimr}))",
                "size(cpu_usage{host=solomon}) > 0"
        );

        for (var query : queries) {
            syncReadDataByQuery(query);
        }
    }

    @Test
    public void invalidRequests() {
        List<String> queries = List.of(
                "-cpu_usage{host=solomon}",
                "let foo = cpu_usage{host=solomon}; foo",
                "use {host=solomon}; cpu_usage{}"
        );

        for (var query : queries) {
            boolean thrown = false;
            try {
                syncReadDataByQuery(query);
            } catch (Exception e) {
                Throwable t = unwrapCompletionException(e);
                Assert.assertTrue("Wrong exception for query: " + query, t instanceof ProgramValidationException);
                System.err.println(query + ": " + t.getMessage());
                thrown = true;
            }
            Assert.assertTrue("Validation exception was not thrown for: " + query, thrown);
        }
    }

    @Test
    public void groupLinesReturnsVector() {
        syncReadDataByQuery("group_lines('sum', group_lines('sum', cpu_usage{}))");
    }
}
