package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Month;
import java.time.ZoneOffset;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.jetbrains.annotations.NotNull;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metrics.client.ReadResponse;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.util.time.Interval;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.alert.protobuf.TEvaluationStatus.ECode.ALARM;
import static ru.yandex.solomon.alert.protobuf.TEvaluationStatus.ECode.ERROR;
import static ru.yandex.solomon.alert.protobuf.TEvaluationStatus.ECode.NO_DATA;
import static ru.yandex.solomon.alert.protobuf.TEvaluationStatus.ECode.OK;
import static ru.yandex.solomon.alert.protobuf.TEvaluationStatus.ECode.WARN;
import static ru.yandex.solomon.expression.expr.func.util.SelFnAlertEvaluationHistory.ALERT_STATUS_LOWER_BITS;


/**
 * @author Vladimir Gordiychuk
 */
public class AlertEvaluationHistoryDtoTest {

    @Test
    public void empty() {
        var result = create(AggrGraphDataArrayList.empty(), 60_000);
        assertEquals(0, result.timestamps.length);
        assertEquals(0, result.ok.length);
        assertEquals(0, result.warn.length);
        assertEquals(0, result.alarm.length);
        assertEquals(0, result.noData.length);
        assertEquals(0, result.error.length);
        assertEquals(0, result.okMuted.length);
        assertEquals(0, result.warnMuted.length);
        assertEquals(0, result.alarmMuted.length);
        assertEquals(0, result.noDataMuted.length);
        assertEquals(0, result.errorMuted.length);
    }

    @Test
    public void oneAlarm() {
        var source = AggrGraphDataArrayList.of(point("00:01:30", ALARM));
        var result = create(source, 60_000);
        assertEquals(ts("00:01"), result.timestamps[0]);
        assertEquals(0, result.ok[0]);
        assertEquals(0, result.warn[0]);
        assertEquals(1, result.alarm[0]);
        assertEquals(0, result.noData[0]);
        assertEquals(0, result.error[0]);
        assertEquals(0, result.okMuted[0]);
        assertEquals(0, result.warnMuted[0]);
        assertEquals(0, result.alarmMuted[0]);
        assertEquals(0, result.noDataMuted[0]);
        assertEquals(0, result.errorMuted[0]);
    }

    @Test
    public void oneOk() {
        var source = AggrGraphDataArrayList.of(point("00:01:30", OK));
        var result = create(source, 60_000);
        assertEquals(ts("00:01"), result.timestamps[0]);
        assertEquals(1, result.ok[0]);
        assertEquals(0, result.warn[0]);
        assertEquals(0, result.alarm[0]);
        assertEquals(0, result.noData[0]);
        assertEquals(0, result.error[0]);
        assertEquals(0, result.okMuted[0]);
        assertEquals(0, result.warnMuted[0]);
        assertEquals(0, result.alarmMuted[0]);
        assertEquals(0, result.noDataMuted[0]);
        assertEquals(0, result.errorMuted[0]);
    }

    @Test
    public void oneWarn() {
        var source = AggrGraphDataArrayList.of(point("00:01:30", WARN));
        var result = create(source, 60_000);
        assertEquals(ts("00:01"), result.timestamps[0]);
        assertEquals(0, result.ok[0]);
        assertEquals(1, result.warn[0]);
        assertEquals(0, result.alarm[0]);
        assertEquals(0, result.noData[0]);
        assertEquals(0, result.error[0]);
        assertEquals(0, result.okMuted[0]);
        assertEquals(0, result.warnMuted[0]);
        assertEquals(0, result.alarmMuted[0]);
        assertEquals(0, result.noDataMuted[0]);
        assertEquals(0, result.errorMuted[0]);
    }

    @Test
    public void oneError() {
        var source = AggrGraphDataArrayList.of(point("00:01:30", ERROR));
        var result = create(source, 60_000);
        assertEquals(ts("00:01"), result.timestamps[0]);
        assertEquals(0, result.ok[0]);
        assertEquals(0, result.warn[0]);
        assertEquals(0, result.alarm[0]);
        assertEquals(0, result.noData[0]);
        assertEquals(1, result.error[0]);
        assertEquals(0, result.okMuted[0]);
        assertEquals(0, result.warnMuted[0]);
        assertEquals(0, result.alarmMuted[0]);
        assertEquals(0, result.noDataMuted[0]);
        assertEquals(0, result.errorMuted[0]);
    }

    @Test
    public void oneNoData() {
        var source = AggrGraphDataArrayList.of(point("00:01:30", NO_DATA));
        var result = create(source, 60_000);
        assertEquals(ts("00:01"), result.timestamps[0]);
        assertEquals(0, result.ok[0]);
        assertEquals(0, result.warn[0]);
        assertEquals(0, result.alarm[0]);
        assertEquals(1, result.noData[0]);
        assertEquals(0, result.error[0]);
        assertEquals(0, result.okMuted[0]);
        assertEquals(0, result.warnMuted[0]);
        assertEquals(0, result.alarmMuted[0]);
        assertEquals(0, result.noDataMuted[0]);
        assertEquals(0, result.errorMuted[0]);
    }

    @Test
    public void oneMutedNoData() {
        var source = AggrGraphDataArrayList.of(AggrPoint.builder()
                .time(ts("00:01:30"))
                .longValue((((long) 3) << ALERT_STATUS_LOWER_BITS) | NO_DATA.getNumber())
                .count(1)
                .build());
        var result = create(source, 60_000);
        assertEquals(ts("00:01"), result.timestamps[0]);
        assertEquals(0, result.ok[0]);
        assertEquals(0, result.warn[0]);
        assertEquals(0, result.alarm[0]);
        assertEquals(1, result.noData[0]);
        assertEquals(0, result.error[0]);
        assertEquals(0, result.okMuted[0]);
        assertEquals(0, result.warnMuted[0]);
        assertEquals(0, result.alarmMuted[0]);
        assertEquals(1, result.noDataMuted[0]);
        assertEquals(0, result.errorMuted[0]);
    }

    @Test
    public void countMultipleStatus() {
        var source = AggrGraphDataArrayList.of(
            point("00:01:00", ALARM),
            point("00:02:00", ALARM),
            point("00:03:00", WARN),
            point("00:04:00", OK),
            point("00:05:00", OK),
            point("00:06:00", OK),

            point("00:10:00", OK),
            point("00:11:00", OK),
            point("00:12:00", ALARM)
        );

        var result = create(source, TimeUnit.MINUTES.toMillis(10));
        assertEquals(2, result.timestamps.length);
        {
            assertEquals(ts("00:00"), result.timestamps[0]);
            assertEquals(3, result.ok[0]);
            assertEquals(1, result.warn[0]);
            assertEquals(2, result.alarm[0]);
            assertEquals(0, result.noData[0]);
            assertEquals(0, result.error[0]);
        }
        {
            assertEquals(ts("00:10"), result.timestamps[1]);
            assertEquals(2, result.ok[1]);
            assertEquals(0, result.warn[1]);
            assertEquals(1, result.alarm[1]);
            assertEquals(0, result.noData[1]);
            assertEquals(0, result.error[1]);
        }
    }

    @Test
    public void countMultipleStatusWithExtraStatus() {
        var source = AggrGraphDataArrayList.of(
                point("00:01:00", ALARM),
                point("00:02:00", ALARM.getNumber() | 128),
                point("00:03:00", WARN.getNumber() | 128),
                point("00:04:00", OK.getNumber() | 128),
                point("00:05:00", OK),
                point("00:06:00", OK),

                point("00:10:00", OK),
                point("00:11:00", OK),
                point("00:12:00", ALARM)
        );

        var result = create(source, TimeUnit.MINUTES.toMillis(10));
        assertEquals(2, result.timestamps.length);
        {
            assertEquals(ts("00:00"), result.timestamps[0]);
            assertEquals(3, result.ok[0]);
            assertEquals(1, result.warn[0]);
            assertEquals(2, result.alarm[0]);
            assertEquals(0, result.noData[0]);
            assertEquals(0, result.error[0]);
        }
        {
            assertEquals(ts("00:10"), result.timestamps[1]);
            assertEquals(2, result.ok[1]);
            assertEquals(0, result.warn[1]);
            assertEquals(1, result.alarm[1]);
            assertEquals(0, result.noData[1]);
            assertEquals(0, result.error[1]);
        }
    }

    @Test
    public void countMultipleStatusWithExtraStatus_downsamplingOff() {
        var source = AggrGraphDataArrayList.of(
                point("00:01:00", ALARM),
                point("00:01:30", ALARM),
                point("00:02:00", ALARM.getNumber() | 128),
                point("00:02:30", ALARM.getNumber() | 128),
                point("00:03:00", WARN.getNumber() | 128),
                point("00:03:30", WARN.getNumber() | 128),
                point("00:04:00", OK.getNumber() | 128),
                point("00:04:30", OK.getNumber() | 128),
                point("00:05:00", OK),
                point("00:05:30", OK),
                point("00:06:00", OK),
                point("00:06:30", OK),

                point("00:10:00", OK),
                point("00:10:30", OK),
                point("00:11:00", OK),
                point("00:11:30", OK),
                point("00:12:00", ALARM),
                point("00:12:30", ALARM)
        );

        var result = create(source, TimeUnit.MINUTES.toMillis(10), true);
        assertEquals(2, result.timestamps.length);
        {
            assertEquals(ts("00:00"), result.timestamps[0]);
            assertEquals(3, result.ok[0]);
            assertEquals(1, result.warn[0]);
            assertEquals(2, result.alarm[0]);
            assertEquals(0, result.noData[0]);
            assertEquals(0, result.error[0]);
        }
        {
            assertEquals(ts("00:10"), result.timestamps[1]);
            assertEquals(2, result.ok[1]);
            assertEquals(0, result.warn[1]);
            assertEquals(1, result.alarm[1]);
            assertEquals(0, result.noData[1]);
            assertEquals(0, result.error[1]);
        }
    }

    @Test
    public void gap() {
        var source = AggrGraphDataArrayList.of(
            point("00:06:00", OK),
            point("00:10:15", ALARM)
        );

        var result = create(source, TimeUnit.MINUTES.toMillis(1));
        assertEquals(2, result.timestamps.length);
        {
            assertEquals(ts("00:06"), result.timestamps[0]);
            assertEquals(1, result.ok[0]);
            assertEquals(0, result.warn[0]);
            assertEquals(0, result.alarm[0]);
            assertEquals(0, result.noData[0]);
            assertEquals(0, result.error[0]);
        }
        {
            assertEquals(ts("00:10"), result.timestamps[1]);
            assertEquals(0, result.ok[1]);
            assertEquals(0, result.warn[1]);
            assertEquals(1, result.alarm[1]);
            assertEquals(0, result.noData[1]);
            assertEquals(0, result.error[1]);
        }
    }

    @Test
    public void smallAmountOfPoint() {
        var source = AggrGraphDataArrayList.of(
            point("00:06:00", OK),
            point("00:06:15", ALARM),
            point("00:06:30", ALARM)
        );

        var interval = Interval.millis(ts("00:06:00"), ts("00:06:30"));
        var metric = new Metric<>(MetricKey.unknown(), StockpileColumns.typeByMask(source.columnSetMask()), source);
        ReadResponse readResponse = new ReadResponse(metric, StockpileStatus.OK, true);
        var result = AlertEvaluationHistoryDto.create(readResponse, interval, 15_000);
        assertEquals(3, result.timestamps.length);
    }

    @Test
    public void multiAlert() {
        var metrics = List.of(
                makeMetric("OK", AggrGraphDataArrayList.of(
                        point("00:00:00"),
                        point("00:00:00"),
                        point("00:01:00", OK),
                        point("00:01:30", OK),
                        point("00:02:00", OK),
                        point("00:02:30", OK),
                        point("00:03:00"),
                        point("00:03:30")
                )),
                makeMetric("ALARM", AggrGraphDataArrayList.of(
                        point("00:00:00"),
                        point("00:00:00"),
                        point("00:01:00", ALARM),
                        point("00:01:30", ALARM),
                        point("00:02:00"),
                        point("00:02:30"),
                        point("00:03:00", ALARM),
                        point("00:03:30", ALARM)
                )),
                makeMetric("WARN", AggrGraphDataArrayList.of(
                        point("00:00:00"),
                        point("00:00:00"),
                        point("00:01:00"),
                        point("00:01:30"),
                        point("00:02:00", WARN),
                        point("00:02:30", WARN),
                        point("00:03:00", WARN),
                        point("00:03:30", WARN)
                )),
                get("NO_DATA"),
                get("ERROR"),
                get("WARN:MUTED"),
                get("NO_DATA:MUTED"),
                get("ALARM:MUTED"),
                get("OK:MUTED"),
                get("ERROR:MUTED")
        );

        var interval = Interval.millis(ts("00:00:00"), ts("00:04:00"));
        var result = AlertEvaluationHistoryDto.create(metrics,interval, 60_000L);
        assertEquals(3, result.timestamps.length);
        {
            assertEquals(ts("00:01"), result.timestamps[0]);
            assertEquals(1, result.ok[0]);
            assertEquals(0, result.warn[0]);
            assertEquals(4, result.alarm[0]);
            assertEquals(0, result.noData[0]);
            assertEquals(0, result.error[0]);
        }
        {
            assertEquals(ts("00:02"), result.timestamps[1]);
            assertEquals(1, result.ok[1]);
            assertEquals(6, result.warn[1]);
            assertEquals(0, result.alarm[1]);
            assertEquals(0, result.noData[1]);
            assertEquals(0, result.error[1]);
        }
        {
            assertEquals(ts("00:03"), result.timestamps[2]);
            assertEquals(0, result.ok[2]);
            assertEquals(6, result.warn[2]);
            assertEquals(4, result.alarm[2]);
            assertEquals(0, result.noData[2]);
            assertEquals(0, result.error[2]);
        }
    }

    @NotNull
    private Metric<MetricKey> get(String status) {
        return makeMetric(status, AggrGraphDataArrayList.of(
                point("00:00:00"),
                point("00:00:00"),
                point("00:01:00"),
                point("00:01:30"),
                point("00:02:00"),
                point("00:02:30"),
                point("00:03:00"),
                point("00:03:30")
        ));
    }

    private static AggrPoint point(String time, TEvaluationStatus.ECode code) {
        return point(time, code.getNumber());
    }

    private static AggrPoint point(String time, long code) {
        return AggrPoint.builder()
                .time(ts(time))
                .longValue(code)
                .count(1)
                .build();
    }

    private static AggrPoint point(String time) {
        return AggrPoint.builder()
                .time(ts(time))
                .count(0)
                .build();
    }

    private AlertEvaluationHistoryDto create(AggrGraphDataIterable source, long gridMillis) {
        return create(source, gridMillis, false);
    }

    private AlertEvaluationHistoryDto create(AggrGraphDataIterable source, long gridMillis, boolean off) {
        var interval = Interval.millis(ts("00:00"), ts("23:59:59"));
        var metric = new Metric<>(MetricKey.unknown(), StockpileColumns.typeByMask(source.columnSetMask()), source);
        ReadResponse readResponse = new ReadResponse(metric, StockpileStatus.OK, true);
        return AlertEvaluationHistoryDto.create(readResponse, interval, gridMillis, off);
    }

    private static long ts(String time) {
        return LocalDateTime.of(LocalDate.of(2000, Month.JANUARY, 1), LocalTime.parse(time))
            .toInstant(ZoneOffset.UTC)
            .toEpochMilli();
    }

    private Metric<MetricKey> makeMetric(String status, AggrGraphDataArrayList graphData) {
        var key = new MetricKey(MetricType.COUNTER, "", Labels.of("status", status), 0, List.of());
        return new Metric<>(key, ru.yandex.solomon.model.protobuf.MetricType.COUNTER, graphData);
    }

}
