package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.time.Duration;
import java.util.List;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.node.ObjectNode;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.AstAssignment;
import ru.yandex.solomon.expression.ast.AstBinOp;
import ru.yandex.solomon.expression.ast.AstCall;
import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.ast.AstInterpolatedString;
import ru.yandex.solomon.expression.ast.AstOp;
import ru.yandex.solomon.expression.ast.AstSelector;
import ru.yandex.solomon.expression.ast.AstSelectors;
import ru.yandex.solomon.expression.ast.AstValueDouble;
import ru.yandex.solomon.expression.ast.AstValueDuration;
import ru.yandex.solomon.expression.ast.AstValueString;
import ru.yandex.solomon.expression.ast.serialization.AstMappingContext;
import ru.yandex.solomon.gateway.api.cloud.v1.ExpressionController;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.ParseRequestDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.ParseResponseDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.PositionRangeDto;
import ru.yandex.solomon.gateway.api.cloud.v1.dto.el.RenderRequestDto;
import ru.yandex.solomon.labels.InterpolatedString;
import ru.yandex.solomon.labels.query.SelectorType;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.expression.PositionRange.UNKNOWN;

/**
 * @author Ivan Tsybulin
 */
public class ExpressionControllerTest {
    private ExpressionController controller;
    private AuthSubject subject;

    @Before
    public void setUp() {
        controller = new ExpressionController();
        subject = AnonymousAuthSubject.INSTANCE;
    }

    private static ParseRequestDto query(@Nullable String program, boolean withRanges) {
        return new ParseRequestDto(program, withRanges);
    }

    private static RenderRequestDto render(ObjectNode json) {
        return new RenderRequestDto(json);
    }

    @Test
    public void emptyWithRanges() {
        ParseResponseDto result = controller.parseExpression(subject, query(null, true));

        assertNull(result.programAst);
        assertEquals(result.firstParsingError.position, new PositionRangeDto(PositionRange.of(1)));
    }

    @Test
    public void emptyWithoutRangesStillHasErrorInfo() {
        ParseResponseDto result = controller.parseExpression(subject, query(null, false));

        assertNull(result.programAst);
        assertEquals(result.firstParsingError.position, new PositionRangeDto(PositionRange.of(1)));
    }

    @Test
    public void simpleWithRanges() {
        ParseResponseDto result = controller.parseExpression(subject, query("42", true));

        assertNull(result.firstParsingError);
        assertEquals(List.of(), result.programAst.variables);
        assertEquals("num", result.programAst.expression.get("_t").textValue());
        assertEquals("[1,1,0,1,2,1]", result.programAst.expression.get("_r").toString());
        assertEquals(42d, result.programAst.expression.get("v").doubleValue(), 0);

        String program = controller.renderExpression(subject, render(result.programAst.expression)).program;
        assertEquals("42", program);
    }


    @Test
    public void simpleWithoutRanges() {
        ParseResponseDto result = controller.parseExpression(subject, query("42", false));

        assertNull(result.firstParsingError);
        assertEquals(List.of(), result.programAst.variables);
        assertEquals("num", result.programAst.expression.get("_t").textValue());
        assertNull(result.programAst.expression.get("_r"));
        assertEquals(42d, result.programAst.expression.get("v").doubleValue(), 0);

        String program = controller.renderExpression(subject, render(result.programAst.expression)).program;
        assertEquals("42", program);
    }

    @Test
    public void complex() {
        ParseResponseDto result = controller.parseExpression(subject,
                query("sum({host=\"{{host}}\"}) by 5m", false));

        AstCall expected = new AstCall(UNKNOWN,
                new AstIdent(UNKNOWN, "group_by_time"),
                List.of(
                        new AstValueDuration(UNKNOWN, Duration.ofMinutes(5)),
                        new AstValueString(UNKNOWN, "sum"),
                        new AstSelectors(
                                UNKNOWN,
                                "",
                                List.of(
                                        new AstSelector(
                                                UNKNOWN,
                                                new AstValueString(UNKNOWN, "host"),
                                                new AstInterpolatedString(UNKNOWN, InterpolatedString.parse("{{host}}")),
                                                SelectorType.GLOB
                                        )
                                )
                        )
                )
        );

        assertNull(result.firstParsingError);
        assertEquals(List.of(), result.programAst.variables);
        assertEquals(new AstMappingContext(false).render(expected), result.programAst.expression);

        String program = controller.renderExpression(subject, render(result.programAst.expression)).program;
        assertEquals("group_by_time(5m, 'sum', {'host'='{{host}}'})", program);
    }

    @Test
    public void multiline() {
        ParseResponseDto result = controller.parseExpression(subject,
                query("let a = 1;\nlet b = 2;\na + b", false));

        AstAssignment first = new AstAssignment(UNKNOWN, "a", new AstValueDouble(UNKNOWN, 1));
        AstAssignment second = new AstAssignment(UNKNOWN, "b", new AstValueDouble(UNKNOWN, 2));
        AstBinOp expr = new AstBinOp(UNKNOWN, new AstIdent(UNKNOWN, "a"), new AstIdent(UNKNOWN, "b"),
                new AstOp(UNKNOWN, "+"));

        assertNull(result.firstParsingError);
        assertEquals(new AstMappingContext(false).render(first), result.programAst.variables.get(0));
        assertEquals(new AstMappingContext(false).render(second), result.programAst.variables.get(1));
        assertEquals(new AstMappingContext(false).render(expr), result.programAst.expression);

        String program = controller.renderExpression(subject, render(result.programAst.expression)).program;
        assertEquals("a + b", program);
    }
}
