package ru.yandex.solomon.gateway.api.cloud.v1.dto;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.point.AggrPoints;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class StatusMergingSupportTest {

    @Test
    public void mergeNone() {
        var result = StatusMergingSupport.mergeFrom(List.of());

        Assert.assertEquals(0, result.timestamps.length);
    }

    private Metric<MetricKey> makeMetric(String status, AggrGraphDataArrayList graphData) {
        var key = new MetricKey(MetricType.COUNTER, "", Labels.of("status", status), 0, List.of());
        return new Metric<>(key, ru.yandex.solomon.model.protobuf.MetricType.COUNTER, graphData);
    }

    @Test
    public void mergeEmpty() {

        var result = StatusMergingSupport.mergeFrom(List.of(
                makeMetric("SUCCESS", AggrGraphDataArrayList.empty())
        ));

        Assert.assertEquals(0, result.timestamps.length);
    }

    @Test
    public void mergeSingle() {
        var result = StatusMergingSupport.mergeFrom(List.of(
                makeMetric("SUCCESS", AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(1000, 42),
                        AggrPoints.lpoint(1500, 43),
                        AggrPoints.lpoint(2500, 45)
                ))
        ));

        Assert.assertArrayEquals(new long[]{1500, 2500}, result.timestamps);
        Assert.assertArrayEquals(new long[]{1, 2}, result.statuses.get("SUCCESS"));
    }

    @Test
    public void mergeTwo() {
        var result = StatusMergingSupport.mergeFrom(List.of(
                makeMetric("SUCCESS", AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(1000, 42),
                        AggrPoints.lpoint(1500, 43),
                        AggrPoints.lpoint(2000, 45)
                )),
                makeMetric("INVALID_REQUEST", AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(1000, 0),
                        AggrPoints.lpoint(2500, 1)
                ))
        ));

        Assert.assertArrayEquals(new long[]{1500, 2000, 2500}, result.timestamps);
        Assert.assertArrayEquals(new long[]{1, 2, 0}, result.statuses.get("SUCCESS"));
        Assert.assertArrayEquals(new long[]{0, 0, 1}, result.statuses.get("INVALID_REQUEST"));
    }

    @Test
    public void mergeOneAndEmpty() {
        var result = StatusMergingSupport.mergeFrom(List.of(
                makeMetric("SUCCESS", AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(1000, 42),
                        AggrPoints.lpoint(1500, 43),
                        AggrPoints.lpoint(2500, 45)
                )),
                makeMetric("INVALID_REQUEST", AggrGraphDataArrayList.empty())
        ));

        Assert.assertArrayEquals(new long[]{1500, 2500}, result.timestamps);
        Assert.assertArrayEquals(new long[]{1, 2}, result.statuses.get("SUCCESS"));
        Assert.assertArrayEquals(new long[]{0, 0}, result.statuses.get("INVALID_REQUEST"));
    }

    @Test
    public void mergeWithHoles() {
        var result = StatusMergingSupport.mergeFrom(List.of(
                makeMetric("SUCCESS", AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(1000, 42),
                        AggrPoints.lpoint(2000, 43),
                        AggrPoints.lpoint(2500, 45)
                )),
                makeMetric("INVALID_REQUEST", AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(1000, 1),
                        AggrPoints.lpoint(1500, 1),
                        AggrPoints.lpoint(2500, 3)
                ))
        ));

        Assert.assertArrayEquals(new long[]{1500, 2000, 2500}, result.timestamps);
        Assert.assertArrayEquals(new long[]{1, 0, 2}, result.statuses.get("SUCCESS"));
        Assert.assertArrayEquals(new long[]{0, 0, 2}, result.statuses.get("INVALID_REQUEST"));
    }


}
