package ru.yandex.solomon.gateway.api.cloud.v2;

import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricDataDto;
import ru.yandex.solomon.model.point.AggrPoints;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class MetricDataDtoTest {
    @Test
    public void dgaugeEmpty() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("cpu_usage")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.DGAUGE)
                .setGraphData(AggrGraphDataArrayList.empty())
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.DGAUGE, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("cpu_usage", dto.getName());
        assertArrayEquals(new long[0], dto.getTimeseries().getTimestamps());
        assertNull(dto.getTimeseries().getInt64Values());
        assertArrayEquals(new double[0], dto.getTimeseries().getDoubleValues(), 0);
    }

    @Test
    public void dgaugeWithPoints() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("cpu_usage")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.DGAUGE)
                .setGraphData(AggrGraphDataArrayList.of(
                        AggrPoints.dpoint(1000, 42),
                        AggrPoints.dpoint(2000, 43)
                ))
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.DGAUGE, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("cpu_usage", dto.getName());
        assertArrayEquals(new long[] {1000, 2000}, dto.getTimeseries().getTimestamps());
        assertNull(dto.getTimeseries().getInt64Values());
        assertArrayEquals(new double[] {42d, 43d}, dto.getTimeseries().getDoubleValues(), 0);
    }

    @Test
    public void igaugeEmpty() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("cpu_usage_perc")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.IGAUGE)
                .setGraphData(AggrGraphDataArrayList.empty())
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.IGAUGE, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("cpu_usage_perc", dto.getName());
        assertArrayEquals(new long[0], dto.getTimeseries().getTimestamps());
        assertArrayEquals(new long[0], dto.getTimeseries().getInt64Values());
        assertNull(dto.getTimeseries().getDoubleValues());
    }

    @Test
    public void igaugeWithPoints() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("cpu_usage_perc")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.IGAUGE)
                .setGraphData(AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(1000, 42),
                        AggrPoints.lpoint(2000, 43)
                ))
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.IGAUGE, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("cpu_usage_perc", dto.getName());
        assertArrayEquals(new long[] {1000, 2000}, dto.getTimeseries().getTimestamps());
        assertArrayEquals(new long[] {42, 43}, dto.getTimeseries().getInt64Values());
        assertNull(dto.getTimeseries().getDoubleValues());
    }

    @Test
    public void igaugeStoredAsDgauge() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("cpu_usage_perc")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.IGAUGE)
                .setGraphData(AggrGraphDataArrayList.of(
                        AggrPoints.dpoint(1000, 42.1),
                        AggrPoints.dpoint(2000, 42.8)
                ))
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.IGAUGE, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("cpu_usage_perc", dto.getName());
        assertArrayEquals(new long[] {1000, 2000}, dto.getTimeseries().getTimestamps());
        assertArrayEquals(new long[] {42, 43}, dto.getTimeseries().getInt64Values());
        assertNull(dto.getTimeseries().getDoubleValues());
    }

    @Test
    public void counterStoredAsDgauge() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("cpu_usage_perc")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.COUNTER)
                .setGraphData(AggrGraphDataArrayList.of(
                        AggrPoints.dpoint(1000, 100),
                        AggrPoints.dpoint(2000, 200)
                ))
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.COUNTER, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("cpu_usage_perc", dto.getName());
        assertArrayEquals(new long[] {1000, 2000}, dto.getTimeseries().getTimestamps());
        assertArrayEquals(new long[] {100, 200}, dto.getTimeseries().getInt64Values());
        assertNull(dto.getTimeseries().getDoubleValues());
    }

    @Test
    public void dgaugeStoredAsInt() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("cpu_usage_perc")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.DGAUGE)
                .setGraphData(AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(1000, 100),
                        AggrPoints.lpoint(2000, 200)
                ))
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.DGAUGE, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("cpu_usage_perc", dto.getName());
        assertArrayEquals(new long[] {1000, 2000}, dto.getTimeseries().getTimestamps());
        assertNull(dto.getTimeseries().getInt64Values());
        assertArrayEquals(new double[] {100d, 200d}, dto.getTimeseries().getDoubleValues(), 0);
    }

    @Test
    public void rate() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("rps")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.RATE)
                .setGraphData(AggrGraphDataArrayList.of(
                        AggrPoints.dpoint(15000, 2d, 15000),
                        AggrPoints.dpoint(30000, 3d, 15000)
                ))
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.RATE, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("rps", dto.getName());
        assertArrayEquals(new long[] {15000, 30000}, dto.getTimeseries().getTimestamps());
        assertNull(dto.getTimeseries().getInt64Values());
        assertArrayEquals(new double[] {2d, 3d}, dto.getTimeseries().getDoubleValues(), 0);
    }

    @Ignore("RATEs in Stockpile are subject to change")
    @Test
    public void rateNative() {
        var dto = MetricDataDto.fromModel(NamedGraphData.newBuilder()
                .setMetricName("rps")
                .setLabels(Labels.of("host", "solomon-01", "cpu", "0"))
                .setType(MetricType.RATE)
                .setGraphData(ru.yandex.solomon.model.protobuf.MetricType.RATE, AggrGraphDataArrayList.of(
                        AggrPoints.lpoint(15000, 30, 15000),
                        AggrPoints.lpoint(30000, 45, 15000)
                ))
                .build());
        assertEquals(ru.yandex.solomon.gateway.api.cloud.v2.dto.MetricType.RATE, dto.getType());
        assertEquals(Map.of("host", "solomon-01", "cpu", "0"), dto.getLabels());
        assertEquals("rps", dto.getName());
        assertArrayEquals(new long[] {15000, 30000}, dto.getTimeseries().getTimestamps());
        assertNull(dto.getTimeseries().getInt64Values());
        assertArrayEquals(new double[] {2d, 3d}, dto.getTimeseries().getDoubleValues(), 0);
    }
}
