package ru.yandex.solomon.gateway.api.cloud.v2;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.gateway.cloud.billing.BillingLogStub;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.metrics.client.MetricsClients;
import ru.yandex.solomon.metrics.client.SolomonClientStub;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPoints;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class PrometheusMetricsClientTest {
    private static final String CLOUD_ID = "cloud";
    private static final String FOLDER_ID = "folder";
    private static final String SERVICE = "service";

    private SolomonClientStub solomon;
    private BillingLogStub billing;
    private PrometheusMetricsClient client;

    @Before
    public void setUp() {
        solomon = new SolomonClientStub();
        var metricsClient = MetricsClients.create("test", solomon.getMetabase(), solomon.getStockpile());
        billing = new BillingLogStub();
        client = new PrometheusMetricsClient(metricsClient, createConfHolder(), billing, new PrometheusMetricsClientMetrics());
    }

    private static SolomonConfHolder createConfHolder() {
        SolomonConfHolder confHolder = new SolomonConfHolder();

        Project project = Project.newBuilder()
                .setId(CLOUD_ID)
                .setName(CLOUD_ID)
                .setOwner("robot-solomon")
                .build();

        Cluster cluster = Cluster.newBuilder()
                .setProjectId(CLOUD_ID)
                .setId(FOLDER_ID)
                .setName(FOLDER_ID)
                .build();

        Service service = Service.newBuilder()
                .setProjectId(CLOUD_ID)
                .setId(SERVICE)
                .setName(SERVICE)
                .build();

        Shard shard = Shard.newBuilder()
                .setProjectId(project.getId())
                .setId("shard")
                .setNumId(0)
                .setClusterId(cluster.getId())
                .setClusterName(cluster.getName())
                .setServiceId(service.getId())
                .setServiceName(service.getName())
                .build();

        confHolder.onConfigurationLoad(SolomonConfWithContext.create(new SolomonRawConf(
                List.of(),
                List.of(project),
                List.of(cluster),
                List.of(service),
                List.of(shard)
        )));

        return confHolder;
    }

    @Test
    public void empty() {
        Instant now = Instant.parse("2020-01-01T01:00:00Z");
        String actual = loadPrometheusMetrics("", now);
        assertEquals(0, countReads(MetricType.DGAUGE));
        assertTrue(StringUtils.isBlank(actual));
    }

    @Test
    public void metricWithoutPoints() {
        Instant now = Instant.parse("2020-01-01T01:00:00Z");

        addMetric(MetricType.DGAUGE, "metric_without_points", Labels.of());

        String actual = loadPrometheusMetrics("", now);
        assertEquals(0, countReads(MetricType.DGAUGE));
        assertTrue(StringUtils.isBlank(actual));
    }

    @Test
    public void metricWithDotsInLabels() {
        Instant now = Instant.parse("2020-01-01T01:00:00Z");

        addMetric(MetricType.COUNTER, "metric.with.points",
                Labels.of("points.everywhere", "and.here", "one.more", "with.points"),
                AggrPoints.point("2020-01-01T00:54:00Z", 30),
                AggrPoints.point("2020-01-01T00:55:00Z", 32),
                AggrPoints.point("2020-01-01T00:56:00Z", 32),
                AggrPoints.point("2020-01-01T00:57:00Z", 0),
                AggrPoints.point("2020-01-01T00:58:00Z", 15),
                AggrPoints.point("2020-01-01T00:59:00Z", 28),
                AggrPoints.point("2020-01-01T01:00:00Z", 29));

        String actual = loadPrometheusMetrics("", now);

        String expectedMetric = "" +
                "# TYPE metric_with_points counter\n" +
                "metric_with_points{one_more=\"with.points\", points_everywhere=\"and.here\"} 28.0\n\n";

        assertContains(actual, expectedMetric);
        assertEquals(1, countReads(MetricType.COUNTER));
    }

    @Test
    public void simpleMetric() {
        Instant now = Instant.parse("2020-01-01T01:00:00Z");

        addMetric(MetricType.COUNTER, "simple_metric", Labels.of(),
                AggrPoints.point("2020-01-01T00:54:00Z", 30),
                AggrPoints.point("2020-01-01T00:55:00Z", 32),
                AggrPoints.point("2020-01-01T00:56:00Z", 32),
                AggrPoints.point("2020-01-01T00:57:00Z", 0),
                AggrPoints.point("2020-01-01T00:58:00Z", 15),
                AggrPoints.point("2020-01-01T00:59:00Z", 28),
                AggrPoints.point("2020-01-01T01:00:00Z", 29)
        );

        String actual = loadPrometheusMetrics("", now);

        String expectedMetric = "" +
                "# TYPE simple_metric counter\n" +
                "simple_metric 28.0\n\n";

        assertContains(actual, expectedMetric);
        assertEquals(1, countReads(MetricType.COUNTER));
    }

    @Test
    public void cachedResult() {
        Instant now = Instant.parse("2020-01-01T01:00:00Z");

        addMetric(MetricType.COUNTER, "simple_metric", Labels.of(),
                AggrPoints.point("2020-01-01T00:54:00Z", 30),
                AggrPoints.point("2020-01-01T00:55:00Z", 32),
                AggrPoints.point("2020-01-01T00:56:00Z", 32),
                AggrPoints.point("2020-01-01T00:57:00Z", 0),
                AggrPoints.point("2020-01-01T00:58:00Z", 15),
                AggrPoints.point("2020-01-01T00:59:00Z", 28),
                AggrPoints.point("2020-01-01T01:00:00Z", 29)
        );

        {
            String actual = loadPrometheusMetrics("", now);

            String expectedMetric = "" +
                    "# TYPE simple_metric counter\n" +
                    "simple_metric 28.0\n\n";

            assertContains(actual, expectedMetric);
            assertEquals(1, countReads(MetricType.COUNTER));
        }


        addMetric(MetricType.DGAUGE, "another_metric", Labels.of(),
                AggrPoints.point("2020-01-01T00:59:00Z", 42));

        {
            String actual = loadPrometheusMetrics("", now);

            String expectedMetric = "" +
                    "# TYPE simple_metric counter\n" +
                    "simple_metric 28.0\n\n";

            assertContains(actual, expectedMetric);
            assertEquals(2, countReads(MetricType.COUNTER));
            assertThat(actual, Matchers.not(Matchers.containsString("another_metric")));
        }
    }

    @Test
    public void histogramMetric() {
        Instant now = Instant.parse("2020-01-01T01:00:00Z");

        addMetric(MetricType.RATE, "histogram_metric", Labels.of("bin", "10"),
                AggrPoints.point("2020-01-01T00:59:00Z", 28));
        addMetric(MetricType.RATE, "histogram_metric", Labels.of("bin", "20"),
                AggrPoints.point("2020-01-01T00:59:00Z", 14));
        addMetric(MetricType.RATE, "histogram_metric", Labels.of("bin", "30"),
                AggrPoints.point("2020-01-01T00:59:00Z", 7));
        addMetric(MetricType.RATE, "histogram_metric", Labels.of("bin", "40.5"),
                AggrPoints.point("2020-01-01T00:59:00Z", 5));
        addMetric(MetricType.RATE, "histogram_metric", Labels.of("bin", "40ms"),
                AggrPoints.point("2020-01-01T00:59:00Z", 3));
        addMetric(MetricType.RATE, "histogram_metric", Labels.of("bin", "inf"),
                AggrPoints.point("2020-01-01T00:59:00Z", 1));

        String actual = loadPrometheusMetrics("", now);

        String expectedMetric = "" +
                "# TYPE histogram_metric histogram\n" +
                "histogram_metric_bucket{le=\"10.0\"} 28.0\n" +
                "histogram_metric_bucket{le=\"20.0\"} 42.0\n" +
                "histogram_metric_bucket{le=\"30.0\"} 49.0\n" +
                "histogram_metric_bucket{le=\"40.5\"} 54.0\n" +
                "histogram_metric_bucket{le=\"+Inf\"} 55.0\n" +
                "histogram_metric_count 55.0\n";

        assertContains(actual, expectedMetric);
        assertEquals(6, countReads(MetricType.RATE));
    }

    @Test
    public void metricsBySelectors() {
        Instant now = Instant.parse("2020-01-01T01:00:00Z");

        addMetric(MetricType.DGAUGE, "metric1", Labels.of("resource_id", "1"),
                AggrPoints.point("2020-01-01T00:59:00Z", 1)
        );
        addMetric(MetricType.DGAUGE, "metric1", Labels.of("resource_id", "2"),
                AggrPoints.point("2020-01-01T00:59:00Z", 2)
        );
        addMetric(MetricType.DGAUGE, "metric2", Labels.of("resource_id", "1"),
                AggrPoints.point("2020-01-01T00:59:00Z", 3)
        );

        String actual = loadPrometheusMetrics("{resource_id=\"1\"}", now);

        String expectedMetric1 = "" +
                "# TYPE metric1 gauge\n" +
                "metric1{resource_id=\"1\"} 1.0\n";

        String expectedMetric2 = "" +
                "# TYPE metric2 gauge\n" +
                "metric2{resource_id=\"1\"} 3.0\n";

        assertContains(actual, expectedMetric1, expectedMetric2);
        assertEquals(2, countReads(MetricType.DGAUGE));
    }

    @Test
    public void pagedMetrics() {
        Instant now = Instant.parse("2020-01-01T01:00:00Z");

        final int stepCount = 2000;
        List<String> expectedMetrics = new ArrayList<>();
        for (int i = 1; i <= stepCount; ++i) {
            addMetric(MetricType.DGAUGE, "gauge_metric" + i, Labels.of(),
                    AggrPoints.point("2020-01-01T00:59:00Z", 1));
            addMetric(MetricType.COUNTER, "counter_metric" + i, Labels.of(),
                    AggrPoints.point("2020-01-01T00:59:00Z", 1));
            addMetric(MetricType.RATE, "histogram_metric" + i, Labels.of("bin", "10"),
                    AggrPoints.point("2020-01-01T00:59:00Z", 5));
            addMetric(MetricType.RATE, "histogram_metric" + i, Labels.of("bin", "20"),
                    AggrPoints.point("2020-01-01T00:59:00Z", 3));
            addMetric(MetricType.RATE, "histogram_metric" + i, Labels.of("bin", "30"),
                    AggrPoints.point("2020-01-01T00:59:00Z", 1));
            addMetric(MetricType.RATE, "histogram_metric" + i, Labels.of("bin", "40"),
                    AggrPoints.point("2020-01-01T00:59:00Z", 0));
            addMetric(MetricType.RATE, "histogram_metric" + i, Labels.of("bin", "inf"),
                    AggrPoints.point("2020-01-01T00:59:00Z", 1));

            expectedMetrics.add("" +
                    "# TYPE gauge_metric" + i + " gauge\n" +
                    "gauge_metric" + i + " 1.0\n");

            expectedMetrics.add("" +
                    "# TYPE counter_metric" + i + " counter\n" +
                    "counter_metric" + i + " 1.0\n");

            expectedMetrics.add("" +
                    "# TYPE histogram_metric" + i + " histogram\n" +
                    "histogram_metric" + i + "_bucket{le=\"10.0\"} 5.0\n" +
                    "histogram_metric" + i + "_bucket{le=\"20.0\"} 8.0\n" +
                    "histogram_metric" + i + "_bucket{le=\"30.0\"} 9.0\n" +
                    "histogram_metric" + i + "_bucket{le=\"40.0\"} 9.0\n" +
                    "histogram_metric" + i + "_bucket{le=\"+Inf\"} 10.0\n" +
                    "histogram_metric" + i + "_count 10.0\n");
        }

        String actual = loadPrometheusMetrics("", now);

        assertContains(actual, expectedMetrics.toArray(String[]::new));
        assertEquals(stepCount, countReads(MetricType.DGAUGE));
        assertEquals(stepCount, countReads(MetricType.COUNTER));
        assertEquals(stepCount * 5, countReads(MetricType.RATE));
    }

    private void addMetric(MetricType type, String name, Labels labels, AggrPoint... points) {
        Labels fullLabels = Labels.builder()
                .add(LabelKeys.PROJECT, "cloud")
                .add(LabelKeys.CLUSTER, "folder")
                .add(LabelKeys.SERVICE, "service")
                .add(LabelKeys.SENSOR, name)
                .addAll(labels)
                .build();

        solomon.addMetric(fullLabels, type, AggrGraphDataArrayList.of(points));
    }

    private String loadPrometheusMetrics(String selectors, Instant now) {
        Instant deadline = now.plusSeconds(30);
        byte[] data = client.loadPrometheusMetrics(CLOUD_ID, FOLDER_ID, SERVICE, selectors, now, deadline, "subjectId").join();
        return new String(data);
    }

    private long countReads(MetricType type) {
        return billing.reads(CLOUD_ID, FOLDER_ID, type);
    }

    private void assertContains(String text, String... parts) {
        String curText = text;
        for (String part : parts) {
            int index = curText.indexOf(part);
            if (index >= 0) {
                curText = curText.substring(0, index) + curText.substring(index + part.length());
            } else {
                throw new AssertionError("part \n" + part + "\n is absent in \n" + text);
            }
        }

        if (!StringUtils.isBlank(curText)) {
            throw new AssertionError("text contains unknown parts:\n" + curText);
        }
    }


}
