package ru.yandex.solomon.gateway.api.utils;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Test;

import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ProjectAclEntry;
import ru.yandex.solomon.acl.db.model.ServiceProviderAclEntry;
import ru.yandex.solomon.acl.db.model.SystemAclEntry;
import ru.yandex.solomon.auth.authorizers.RoleAuthorizer;
import ru.yandex.solomon.auth.roles.Role;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class RolesStateTest {

    private static String PROJECT = "solomon";
    private static String SERVICE_PROVIDER = "service provider";
    private static String USER = "user 1";
    private static String USER2 = "user 2";
    private static String USER3 = "user 3";
    private static String GROUP = "group 1";
    private static String GROUP2 = "group 2";

    private RoleAuthorizer.RolesState state;

    @Test
    public void emptyState() {
        state = new RoleAuthorizer.RolesState(Map.of(), Map.of(), Map.of(), Map.of(), Map.of(), Map.of());

        assertNull(state.getProjectAclEntry(ProjectAclEntry.compositeId(PROJECT, USER, AclUidType.USER)));
        assertNull(state.getServiceProviderAclEntry(ServiceProviderAclEntry.compositeId(SERVICE_PROVIDER, USER, AclUidType.USER)));
        assertEquals(List.of(), state.getProjectAclEntryByLoginGroups(USER, PROJECT));
        assertFalse(state.isSystemLogin(USER, AclUidType.USER));
    }

    @Test
    public void fullState() {
        state = new RoleAuthorizer.RolesState(
                Map.of(
                        SystemAclEntry.compositeId(USER, AclUidType.USER), new SystemAclEntry(USER, AclUidType.USER, Set.of(Role.ADMIN.name()), 0),
                        SystemAclEntry.compositeId(USER2, AclUidType.USER), new SystemAclEntry(USER, AclUidType.USER, Set.of(Role.PROJECT_DELETER.name()), 0),
                        SystemAclEntry.compositeId(GROUP2, AclUidType.GROUP), new SystemAclEntry(GROUP2, AclUidType.GROUP, Set.of(Role.ADMIN.name()), 0)
                ),
                Map.of(
                        ProjectAclEntry.compositeId(PROJECT, USER, AclUidType.USER), new ProjectAclEntry(PROJECT, USER, AclUidType.USER, Set.of(Role.ADMIN.name()), 0),
                        ProjectAclEntry.compositeId(PROJECT, GROUP, AclUidType.GROUP), new ProjectAclEntry(PROJECT, GROUP, AclUidType.GROUP, Set.of(Role.ADMIN.name()), 0)
                ),
                Map.of(
                        ServiceProviderAclEntry.compositeId(SERVICE_PROVIDER, USER3, AclUidType.USER), new ServiceProviderAclEntry(SERVICE_PROVIDER, USER3, AclUidType.USER, Set.of(Role.ADMIN.name()), 0),
                        ServiceProviderAclEntry.compositeId(SERVICE_PROVIDER, GROUP2, AclUidType.GROUP), new ServiceProviderAclEntry(SERVICE_PROVIDER, GROUP2, AclUidType.GROUP, Set.of(Role.ADMIN.name()), 0)
                ),
                Map.of(
                        USER2, List.of(GROUP),
                        USER3, List.of(GROUP2)
                ),
                Map.of(
                        new RoleAuthorizer.SubjectId(USER2, AclUidType.USER), List.of(new RoleAuthorizer.ObjectRoles(PROJECT, Set.of(Role.ADMIN.name())), new RoleAuthorizer.ObjectRoles(PROJECT + 1, Set.of(Role.ADMIN.name()))),
                        new RoleAuthorizer.SubjectId(GROUP, AclUidType.GROUP), List.of(new RoleAuthorizer.ObjectRoles(PROJECT + 2, Set.of(Role.ADMIN.name())), new RoleAuthorizer.ObjectRoles(PROJECT + 3, Set.of(Role.ADMIN.name()))),
                        new RoleAuthorizer.SubjectId(USER, AclUidType.USER), List.of(new RoleAuthorizer.ObjectRoles(PROJECT + 4, Set.of(Role.ADMIN.name())))
                ),
                Map.of(
                        new RoleAuthorizer.SubjectId(USER2, AclUidType.USER), List.of(new RoleAuthorizer.ObjectRoles(SERVICE_PROVIDER, Set.of(Role.ADMIN.name())), new RoleAuthorizer.ObjectRoles(SERVICE_PROVIDER + 1, Set.of(Role.ADMIN.name()))),
                        new RoleAuthorizer.SubjectId(GROUP, AclUidType.GROUP), List.of(new RoleAuthorizer.ObjectRoles(SERVICE_PROVIDER + 2, Set.of(Role.ADMIN.name())), new RoleAuthorizer.ObjectRoles(SERVICE_PROVIDER + 3, Set.of(Role.ADMIN.name()))),
                        new RoleAuthorizer.SubjectId(USER, AclUidType.USER), List.of(new RoleAuthorizer.ObjectRoles(SERVICE_PROVIDER + 4, Set.of(Role.ADMIN.name())))
                ));

        assertNull(state.getProjectAclEntry(ProjectAclEntry.compositeId(PROJECT, USER2, AclUidType.USER)));
        assertEquals(new ProjectAclEntry(PROJECT, USER, AclUidType.USER, Set.of(Role.ADMIN.name()), 0),
                state.getProjectAclEntry(ProjectAclEntry.compositeId(PROJECT, USER, AclUidType.USER)));

        assertEquals(List.of(), state.getProjectAclEntryByLoginGroups(USER, PROJECT));
        assertEquals(List.of(new ProjectAclEntry(PROJECT, GROUP, AclUidType.GROUP, Set.of(Role.ADMIN.name()), 0)),
                state.getProjectAclEntryByLoginGroups(USER2, PROJECT));

        assertNull(state.getServiceProviderAclEntry(ServiceProviderAclEntry.compositeId(SERVICE_PROVIDER, USER2, AclUidType.USER)));
        assertEquals(new ServiceProviderAclEntry(SERVICE_PROVIDER, USER3, AclUidType.USER, Set.of(Role.ADMIN.name()), 0),
                state.getServiceProviderAclEntry(ServiceProviderAclEntry.compositeId(SERVICE_PROVIDER, USER3, AclUidType.USER)));

        assertEquals(List.of(), state.getServiceProviderAclEntryByLoginGroups(USER, SERVICE_PROVIDER));
        assertEquals(List.of(new ServiceProviderAclEntry(SERVICE_PROVIDER, GROUP2, AclUidType.GROUP, Set.of(Role.ADMIN.name()), 0)),
                state.getServiceProviderAclEntryByLoginGroups(USER3, SERVICE_PROVIDER));

        assertFalse(state.isSystemLogin(USER2, AclUidType.USER));
        assertTrue(state.isSystemLogin(USER, AclUidType.USER));
        assertTrue(state.isSystemLogin(USER3, AclUidType.USER));

        assertTrue(state.getProjectsBySubject("123", AclUidType.USER).isEmpty());

        var projects = state.getProjectsBySubject(USER, AclUidType.USER);
        assertEquals(1, projects.size());
        assertEquals(PROJECT + 4, projects.get(0).id());

        projects = state.getProjectsBySubject(USER2, AclUidType.USER);
        assertEquals(2, projects.size());
        assertEquals(1, projects.stream().filter(projectRoles -> projectRoles.id().equals(PROJECT)).count());
        assertEquals(1, projects.stream().filter(projectRoles -> projectRoles.id().equals(PROJECT + 1)).count());

        projects = state.getProjectsBySubject(GROUP, AclUidType.GROUP);
        assertEquals(2, projects.size());
        assertEquals(1, projects.stream().filter(projectRoles -> projectRoles.id().equals(PROJECT + 2)).count());
        assertEquals(1, projects.stream().filter(projectRoles -> projectRoles.id().equals(PROJECT + 3)).count());

        projects = state.getProjectsBySubjectGroup(USER2);
        assertEquals(2, projects.size());
        assertEquals(1, projects.stream().filter(projectRoles -> projectRoles.id().equals(PROJECT + 3)).count());
        assertEquals(1, projects.stream().filter(projectRoles -> projectRoles.id().equals(PROJECT + 2)).count());

        assertTrue(state.getProjectsBySubjectGroup("123").isEmpty());

        var sps = state.getServiceProvidersBySubject(USER, AclUidType.USER);
        assertEquals(1, sps.size());
        assertEquals(SERVICE_PROVIDER + 4, sps.get(0).id());

        sps = state.getServiceProvidersBySubject(USER2, AclUidType.USER);
        assertEquals(2, sps.size());
        assertEquals(1, sps.stream().filter(projectRoles -> projectRoles.id().equals(SERVICE_PROVIDER)).count());
        assertEquals(1, sps.stream().filter(projectRoles -> projectRoles.id().equals(SERVICE_PROVIDER + 1)).count());

        sps = state.getServiceProvidersBySubject(GROUP, AclUidType.GROUP);
        assertEquals(2, sps.size());
        assertEquals(1, sps.stream().filter(projectRoles -> projectRoles.id().equals(SERVICE_PROVIDER + 2)).count());
        assertEquals(1, sps.stream().filter(projectRoles -> projectRoles.id().equals(SERVICE_PROVIDER + 3)).count());

        sps = state.getServiceProvidersBySubjectGroup(USER2);
        assertEquals(2, sps.size());
        assertEquals(1, sps.stream().filter(projectRoles -> projectRoles.id().equals(SERVICE_PROVIDER + 3)).count());
        assertEquals(1, sps.stream().filter(projectRoles -> projectRoles.id().equals(SERVICE_PROVIDER + 2)).count());

        assertTrue(state.getServiceProvidersBySubjectGroup("123").isEmpty());
    }
}
