package ru.yandex.solomon.gateway.api.v2.dto;

import java.util.List;

import org.junit.Test;

import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.exceptions.BadRequestException;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

/**
 * @author Vladimir Gordiychuk
 */
public class ServiceDtoTest {

    @Test
    public void minValid() {
        ServiceDto dto = minValidService();
        dto.validate();
    }

    @Test(expected = BadRequestException.class)
    public void invalidGrid() {
        ServiceDto dto = minValidService();
        dto.setGridSec(7);
        dto.validate();
    }

    @Test
    public void gridUnknown() {
        ServiceDto dto = minValidService();
        dto.setGridSec(Service.GRID_UNKNOWN);
        dto.validate();
    }

    @Test
    public void gridAbsent() {
        ServiceDto dto = minValidService();
        dto.setGridSec(Service.GRID_ABSENT);
        dto.validate();
    }

    @Test
    public void gridValid() {
        ServiceDto dto = minValidService();
        for (int gridSec : List.of(1, 2, 3, 4, 5, 10, 15, 30, 300, 600)) {
            dto.setGridSec(gridSec);
            dto.validate();
        }
    }

    @Test(expected = BadRequestException.class)
    public void gridNegative() {
        ServiceDto dto = minValidService();
        dto.setGridSec(-15);
        dto.validate();
    }

    @Test
    public void pullServiceTest() {
        ServiceDto dto = pullService();
        Service service = ServiceDto.toModel(dto);
        assertEquals(ShardSettings.Type.PULL, service.getShardSettings().getType());
        assertEquals(dto.getPath(), service.getShardSettings().getPullSettings().getPath());
        assertEquals(dto.getPort(), service.getShardSettings().getPullSettings().getPort());
        assertEquals(dto.getGridSec(), service.getShardSettings().getGrid());
        assertEquals(dto.getInterval(), service.getShardSettings().getInterval());
        assertEquals(dto.isAddTsArgs(), service.getShardSettings().getPullSettings().isAddTsArgs());
        assertEquals(dto.getTvmDestId(), service.getShardSettings().getPullSettings().getTvmDestinationId());
        assertEquals(dto.getSensorsTtlDays(), Integer.valueOf(service.getShardSettings().getMetricsTtl()));
        assertEquals(DecimPolicy.UNDEFINED, service.getShardSettings().getRetentionPolicy());
    }

    @Test
    public void pushServiceTest() {
        ServiceDto dto = pushService();
        Service service = ServiceDto.toModel(dto);
        assertEquals(ShardSettings.Type.PUSH, service.getShardSettings().getType());
        assertNull(service.getShardSettings().getPullSettings());
        assertEquals(dto.getGridSec(), service.getShardSettings().getGrid());
        assertEquals(dto.getInterval(), service.getShardSettings().getInterval());
        assertEquals(dto.getSensorsTtlDays(), Integer.valueOf(service.getShardSettings().getMetricsTtl()));
        assertEquals(DecimPolicy.UNDEFINED, service.getShardSettings().getRetentionPolicy());
    }

    @Test
    public void pushServiceEmptyGridTest() {
        ServiceDto dto = pushService();
        dto.setGridSec(0);
        Service service = ServiceDto.toModel(dto);
        assertEquals(dto.getGridSec(), service.getShardSettings().getGrid());
    }

    private ServiceDto pullService() {
        ServiceDto dto = new ServiceDto();
        dto.setProjectId("myProjectId");
        dto.setId("myId");
        dto.setName("myName");
        dto.setPath("/path");
        dto.setPort(2);
        dto.setGridSec(5);
        dto.setInterval(11);
        dto.setAddTsArgs(true);
        dto.setTvmDestId("tvm");
        dto.setSensorsTtlDays(123);
        return dto;
    }

    private ServiceDto pushService() {
        ServiceDto dto = new ServiceDto();
        dto.setProjectId("myProjectId");
        dto.setId("myId");
        dto.setName("myName");
        dto.setGridSec(5);
        dto.setInterval(11);
        dto.setSensorsTtlDays(123);
        return dto;
    }

    private ServiceDto minValidService() {
        ServiceDto dto = new ServiceDto();
        dto.setProjectId("myProjectId");
        dto.setId("myId");
        dto.setName("myName");
        return dto;
    }
}
