package ru.yandex.solomon.gateway.api.v2.managers;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.protobuf.LabelValidationFilter;
import ru.yandex.solomon.metrics.client.LabelValuesResponse;
import ru.yandex.solomon.util.labelStats.LabelStats;

import static org.hamcrest.CoreMatchers.equalTo;

/**
 * @author Oleg Baryshnikov
 */
public class ShardLabelsManagerTest {

    private static final String PROJECT_ID = "project";

    private ShardLabelsManager shardLabelsManager;

    @Before
    public void setUp() {
        shardLabelsManager = new ShardLabelsManager(createConfHolder());
    }

    @Test
    public void findAllValues() {
        LabelValuesResponse result = shardLabelsManager.findShardLabelValues(
            Collections.emptyList(),
            Selectors.of(),
            "",
            LabelValidationFilter.ALL,
            0
        );

        Map<String, LabelStats> statsByKey = result.getLabelValuesStats().getStatsByLabelKey();

        Assert.assertThat(statsByKey.get("project"), equalTo(LabelStats.single(PROJECT_ID, 9)));
        Assert.assertThat(statsByKey.get("cluster"), equalTo(new LabelStats(Set.of("cluster1", "cluster2", "cluster3"), 9, false)));
        Assert.assertThat(statsByKey.get("service"), equalTo(new LabelStats(Set.of("service-1", "service-2", "service-3\n"), 9, false)));
    }

    @Test
    public void findValuesByKey() {
        LabelValuesResponse result = shardLabelsManager.findShardLabelValues(
            Collections.singletonList("cluster"),
            Selectors.of(),
            "",
            LabelValidationFilter.ALL,
            0
        );

        Map<String, LabelStats> statsByKey = result.getLabelValuesStats().getStatsByLabelKey();
        Assert.assertThat(statsByKey.get("cluster"), equalTo(new LabelStats(Set.of("cluster1", "cluster2", "cluster3"), 9, false)));
    }

    @Test
    public void findValuesBySelectors() {
        LabelValuesResponse result = shardLabelsManager.findShardLabelValues(
            Collections.emptyList(),
            Selectors.parse("cluster='cluster1'"),
            "",
            LabelValidationFilter.ALL,
            0
        );

        Map<String, LabelStats> statsByKey = result.getLabelValuesStats().getStatsByLabelKey();
        Assert.assertThat(statsByKey.get("project"), equalTo(LabelStats.single(PROJECT_ID, 3)));
        Assert.assertThat(statsByKey.get("cluster"), equalTo(new LabelStats(Set.of("cluster1"), 3, false)));
        Assert.assertThat(statsByKey.get("service"), equalTo(new LabelStats(Set.of("service-1", "service-2", "service-3\n"), 3, false)));
    }

    @Test
    public void findValuesByText() {
        LabelValuesResponse result = shardLabelsManager.findShardLabelValues(
            Collections.emptyList(),
            Selectors.of(),
            "cluster",
            LabelValidationFilter.ALL,
            0
        );

        Map<String, LabelStats> statsByKey = result.getLabelValuesStats().getStatsByLabelKey();
        Assert.assertThat(statsByKey.get("project"), equalTo(new LabelStats(Collections.emptySet(), 9, false)));
        Assert.assertThat(statsByKey.get("cluster"), equalTo(new LabelStats(Set.of("cluster1", "cluster2", "cluster3"), 9, false)));
        Assert.assertThat(statsByKey.get("service"), equalTo(new LabelStats(Collections.emptySet(), 9, false)));
    }

    @Test
    public void findValuesByValidationFilter() {
        LabelValuesResponse result = shardLabelsManager.findShardLabelValues(
            Collections.emptyList(),
            Selectors.of(),
            "",
            LabelValidationFilter.INVALID_ONLY,
            0
        );

        Map<String, LabelStats> statsByKey = result.getLabelValuesStats().getStatsByLabelKey();
        Assert.assertThat(statsByKey.get("project"), equalTo(new LabelStats(Collections.emptySet(), 9, false)));
        Assert.assertThat(statsByKey.get("cluster"), equalTo(new LabelStats(Collections.emptySet(), 9, false)));
        Assert.assertThat(statsByKey.get("service"), equalTo(new LabelStats(Collections.singleton("service-3\n"), 9, false)));
    }

    @Test
    public void findValuesWithLimit() {
        LabelValuesResponse result = shardLabelsManager.findShardLabelValues(
            Collections.emptyList(),
            Selectors.of(),
            "",
            LabelValidationFilter.ALL,
            1
        );

        Map<String, LabelStats> statsByKey = result.getLabelValuesStats().getStatsByLabelKey();
        Assert.assertThat(statsByKey.get("project"), equalTo(new LabelStats(Collections.singleton(PROJECT_ID), 9, false)));
        Assert.assertThat(statsByKey.get("cluster"), equalTo(new LabelStats(Collections.singleton("cluster1"), 9, true)));
        Assert.assertThat(statsByKey.get("service"), equalTo(new LabelStats(Collections.singleton("service-1"), 9, true)));
    }

    private static SolomonConfHolder createConfHolder() {
        Project project = Project.newBuilder()
            .setId(PROJECT_ID)
            .setName(PROJECT_ID)
            .setOwner("user")
            .build();

        SolomonRawConf solomonRawConf = new SolomonRawConf(
            List.of(),
            Collections.singletonList(project),
            Arrays.asList(
                createCluster("cluster1"),
                createCluster("cluster2"),
                createCluster("cluster3")
            ),
            Arrays.asList(
                createService("service-1"),
                createService("service-2"),
                createService("service-3\n")
            ),
            Arrays.asList(
                createShard("cluster1", "service-1"),
                createShard("cluster1", "service-2"),
                createShard("cluster1", "service-3\n"),
                createShard("cluster2", "service-1"),
                createShard("cluster2", "service-2"),
                createShard("cluster2", "service-3\n"),
                createShard("cluster3", "service-1"),
                createShard("cluster3", "service-2"),
                createShard("cluster3", "service-3\n")
            )
        );

        SolomonConfHolder solomonConfHolder = new SolomonConfHolder();
        solomonConfHolder.onConfigurationLoad(SolomonConfWithContext.create(solomonRawConf));
        return solomonConfHolder;
    }

    private static Cluster createCluster(String id) {
        return Cluster.newBuilder()
            .setProjectId(PROJECT_ID)
            .setId(id)
            .setName(id)
            .build();
    }

    private static Service createService(String id) {
        return Service.newBuilder()
            .setProjectId(PROJECT_ID)
            .setId(id)
            .setName(id)
            .build();
    }

    private static Shard createShard(String clusterId, String serviceId) {
        String shardId = PROJECT_ID + "_" + clusterId + "_" + serviceId;
        return Shard.newBuilder()
            .setProjectId(PROJECT_ID)
            .setId(shardId)
            .setNumId(shardId.hashCode())
            .setClusterId(clusterId)
            .setClusterName(clusterId)
            .setServiceId(serviceId)
            .setServiceName(serviceId)
            .build();
    }
}
