package ru.yandex.solomon.gateway.api.v3.cloud.priv.dto;

import java.util.List;

import org.junit.Test;

import ru.yandex.cloud.priv.monitoring.v3.AlertTemplateStatus;
import ru.yandex.cloud.priv.monitoring.v3.CreateAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.DeployAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.ListAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.ListAlertTemplateVersionsRequest;
import ru.yandex.cloud.priv.monitoring.v3.PublishAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.ReadAlertTemplateRequest;
import ru.yandex.cloud.priv.monitoring.v3.UnpublishAlertTemplateRequest;
import ru.yandex.monitoring.api.v3.ListAlertTemplateResponse;
import ru.yandex.monitoring.api.v3.ListAlertTemplateVersionsResponse;
import ru.yandex.solomon.alert.protobuf.AlertTemplate;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.AlertTemplateDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.AlertTemplateFactory;

import static org.junit.Assert.assertEquals;

/**
 * @author Alexey Trushkin
 */
public class CloudAlertTemplateDtoConverterTest {

    private CloudAlertTemplateDtoConverter converter = new CloudAlertTemplateDtoConverter();
    private AlertTemplateDtoConverter internalConverter = new AlertTemplateDtoConverter();

    @Test
    public void convertToCloud_expression() {
        var expression = AlertTemplateFactory.randomAlertTemplate(AlertTemplate.TypeCase.EXPRESSION);
        var proto = internalConverter.fromInternalProto(expression);
        var protoCloud = converter.convertToCloud(proto);
        assertTemplateCompatibility(proto, protoCloud);
    }

    @Test
    public void convertToCloud_threshold() {
        var expression = AlertTemplateFactory.randomAlertTemplate(AlertTemplate.TypeCase.THRESHOLD);
        var proto = internalConverter.fromInternalProto(expression);
        var protoCloud = converter.convertToCloud(proto);
        assertTemplateCompatibility(proto, protoCloud);
    }

    @Test
    public void testConvertToCloud_ListAlertTemplateVersionsResponse() {
        var response = ListAlertTemplateVersionsResponse.newBuilder()
                .addAllAlertTemplates(List.of(
                        internalConverter.fromInternalProto(AlertTemplateFactory.randomAlertTemplate(AlertTemplate.TypeCase.THRESHOLD)),
                        internalConverter.fromInternalProto(AlertTemplateFactory.randomAlertTemplate(AlertTemplate.TypeCase.EXPRESSION))
                ))
                .setNextPageToken("123")
                .build();
        var cloud = converter.convertToCloud(response);
        assertEquals(cloud.getNextPageToken(), response.getNextPageToken());
        assertTemplateCompatibility(response.getAlertTemplatesList().get(0), cloud.getAlertTemplatesList().get(0));
        assertTemplateCompatibility(response.getAlertTemplatesList().get(1), cloud.getAlertTemplatesList().get(1));
        assertEquals(cloud.getNextPageToken(), response.getNextPageToken());
    }

    @Test
    public void testConvertToCloud_ListAlertTemplateResponse() {
        var response = ListAlertTemplateResponse.newBuilder()
                .addAllAlertTemplates(List.of(
                        internalConverter.fromInternalProto(AlertTemplateFactory.randomAlertTemplate(AlertTemplate.TypeCase.THRESHOLD)),
                        internalConverter.fromInternalProto(AlertTemplateFactory.randomAlertTemplate(AlertTemplate.TypeCase.EXPRESSION))
                ))
                .setNextPageToken("123")
                .build();
        var cloud = converter.convertToCloud(response);
        assertEquals(cloud.getNextPageToken(), response.getNextPageToken());
        assertTemplateCompatibility(response.getAlertTemplatesList().get(0), cloud.getAlertTemplatesList().get(0));
        assertTemplateCompatibility(response.getAlertTemplatesList().get(1), cloud.getAlertTemplatesList().get(1));
        assertEquals(cloud.getNextPageToken(), response.getNextPageToken());
    }

    @Test
    public void convertToInternal_CreateAlertTemplateRequest() {
        var expression = AlertTemplateFactory.randomAlertTemplate(AlertTemplate.TypeCase.EXPRESSION);
        var proto = internalConverter.fromInternalProto(expression);
        var protoCloud = converter.convertToCloud(proto);
        var request = CreateAlertTemplateRequest.newBuilder()
                .setServiceProviderId("1234")
                .setAlertTemplate(protoCloud)
                .build();
        var requestInternal = converter.convertToInternal(request);
        assertEquals(requestInternal.getServiceProviderId(), request.getServiceProviderId());
        assertTemplateCompatibility(requestInternal.getAlertTemplate(), request.getAlertTemplate());
    }

    @Test
    public void testConvertToInternal_ReadAlertTemplateRequest() {
        var request = ReadAlertTemplateRequest.newBuilder()
                .setTemplateId("1234")
                .setTemplateVersionTag("version")
                .build();
        var requestInternal = converter.convertToInternal(request);
        assertEquals(requestInternal.getTemplateVersionTag(), request.getTemplateVersionTag());
        assertEquals(requestInternal.getTemplateId(), request.getTemplateId());
    }

    @Test
    public void testConvertToInternal_PublishAlertTemplateRequest() {
        var request = PublishAlertTemplateRequest.newBuilder()
                .setTemplateId("1234")
                .setTemplateVersionTag("version")
                .setServiceProviderId("sp")
                .build();
        var requestInternal = converter.convertToInternal(request);
        assertEquals(requestInternal.getTemplateVersionTag(), request.getTemplateVersionTag());
        assertEquals(requestInternal.getTemplateId(), request.getTemplateId());
        assertEquals(requestInternal.getServiceProviderId(), request.getServiceProviderId());
    }

    @Test
    public void testConvertToInternal_UnpublishAlertTemplateRequest() {
        var request = UnpublishAlertTemplateRequest.newBuilder()
                .setTemplateId("1234")
                .setServiceProviderId("sp")
                .build();
        var requestInternal = converter.convertToInternal(request);
        assertEquals(requestInternal.getTemplateId(), request.getTemplateId());
        assertEquals(requestInternal.getServiceProviderId(), request.getServiceProviderId());
    }

    @Test
    public void testConvertToInternal_DeployAlertTemplateRequest() {
        var request = DeployAlertTemplateRequest.newBuilder()
                .setTemplateId("1234")
                .setServiceProviderId("sp")
                .setTemplateDeployPolicy(DeployAlertTemplateRequest.TemplateDeployPolicy.MANUAL)
                .build();
        var requestInternal = converter.convertToInternal(request);
        assertEquals(requestInternal.getTemplateId(), request.getTemplateId());
        assertEquals(requestInternal.getServiceProviderId(), request.getServiceProviderId());
        assertEquals(requestInternal.getTemplateDeployPolicy().toString(), request.getTemplateDeployPolicy().toString());
    }

    @Test
    public void testConvertToInternal_ListAlertTemplateVersionsRequest() {
        var request = ListAlertTemplateVersionsRequest.newBuilder()
                .setTemplateId("1234")
                .setPageSize(11)
                .setPageToken("123")
                .build();
        var requestInternal = converter.convertToInternal(request);
        assertEquals(requestInternal.getTemplateId(), request.getTemplateId());
        assertEquals(requestInternal.getPageSize(), request.getPageSize());
        assertEquals(requestInternal.getPageToken(), request.getPageToken());
    }

    @Test
    public void testConvertToInternal_ListAlertTemplateRequest() {
        var request = ListAlertTemplateRequest.newBuilder()
                .addAllAlertTemplateStatusesFilter(List.of(AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED))
                .setPageSize(11)
                .setPageToken("123")
                .setServiceProviderId("1")
                .setNameFilter("name")
                .build();
        var requestInternal = converter.convertToInternal(request);
        assertEquals(requestInternal.getServiceProviderId(), request.getServiceProviderId());
        assertEquals(requestInternal.getNameFilter(), request.getNameFilter());
        assertEquals(requestInternal.getPageSize(), request.getPageSize());
        assertEquals(requestInternal.getPageToken(), request.getPageToken());
        assertEquals(requestInternal.getAlertTemplateStatusesFilterList().get(0).toString(), request.getAlertTemplateStatusesFilterList().get(0).toString());
    }

    private void assertTemplateCompatibility(ru.yandex.monitoring.api.v3.AlertTemplate proto, ru.yandex.cloud.priv.monitoring.v3.AlertTemplate protoCloud) {
        assertEquals(proto.getId(), protoCloud.getId());
        assertEquals(proto.getTemplateVersionTag(), protoCloud.getTemplateVersionTag());
        assertEquals(proto.getServiceProviderId(), protoCloud.getServiceProviderId());
        assertEquals(proto.getName(), protoCloud.getName());
        assertEquals(proto.getDescription(), protoCloud.getDescription());
        assertEquals(proto.getCreatedAt(), protoCloud.getCreatedAt());
        assertEquals(proto.getCreatedBy(), protoCloud.getCreatedBy());
        assertEquals(proto.getAnnotationsMap(), protoCloud.getAnnotationsMap());
        assertEquals(proto.getLabelsMap(), protoCloud.getLabelsMap());
        assertEquals(proto.getAlertTemplateStatus().toString(), protoCloud.getAlertTemplateStatus().toString());
        assertEquals(proto.getIsDefault(), protoCloud.getIsDefault());
        for (int i = 0; i < proto.getAlertTemplateParametersList().size(); i++) {
            var paramProto = proto.getAlertTemplateParametersList().get(i);
            var paramProtoCloud = protoCloud.getAlertTemplateParametersList().get(i);
            assertEquals(paramProto.getType().toString(), paramProtoCloud.getType().toString());
            assertEquals(paramProto.getParameter().getName(), paramProtoCloud.getParameter().getName());
            assertEquals(paramProto.getParameter().getDescription(), paramProtoCloud.getParameter().getDescription());
            assertEquals(paramProto.getParameter().getTitle(), paramProtoCloud.getParameter().getTitle());
            assertEquals(paramProto.getParameter().getHidden(), paramProtoCloud.getParameter().getHidden());
            assertEquals(paramProto.getParameter().getDataCase().toString(), paramProtoCloud.getParameter().getDataCase().toString());
            switch (paramProto.getParameter().getDataCase()) {
                case LABEL_VALUES -> {
                    assertEquals(paramProto.getParameter().getLabelValues().getFolderId(), paramProtoCloud.getParameter().getLabelValues().getFolderId());
                    assertEquals(paramProto.getParameter().getLabelValues().getLabelKey(), paramProtoCloud.getParameter().getLabelValues().getLabelKey());
                    assertEquals(paramProto.getParameter().getLabelValues().getDefaultValuesList(), paramProtoCloud.getParameter().getLabelValues().getDefaultValuesList());
                    assertEquals(paramProto.getParameter().getLabelValues().getMultiselectable(), paramProtoCloud.getParameter().getLabelValues().getMultiselectable());
                    assertEquals(paramProto.getParameter().getLabelValues().getSelectors(), paramProtoCloud.getParameter().getLabelValues().getSelectors());
                }
                case TEXT -> {
                    assertEquals(paramProto.getParameter().getText().getDefaultValue(), paramProtoCloud.getParameter().getText().getDefaultValue());
                }
                case INTEGER_PARAMETER -> {
                    assertEquals(paramProto.getParameter().getIntegerParameter().getDefaultValue(), paramProtoCloud.getParameter().getIntegerParameter().getDefaultValue());
                    assertEquals(paramProto.getParameter().getIntegerParameter().getUnitFormat().toString(), paramProtoCloud.getParameter().getIntegerParameter().getUnitFormat().toString());
                }
                case DOUBLE_PARAMETER -> {
                    assertEquals(paramProto.getParameter().getDoubleParameter().getDefaultValue(), paramProtoCloud.getParameter().getDoubleParameter().getDefaultValue(), 0.1);
                    assertEquals(paramProto.getParameter().getDoubleParameter().getUnitFormat().toString(), paramProtoCloud.getParameter().getDoubleParameter().getUnitFormat().toString());
                }
                case TEXT_VALUES -> {
                    assertEquals(paramProto.getParameter().getTextValues().getDefaultValuesList(), paramProtoCloud.getParameter().getTextValues().getDefaultValuesList());
                }
            }
        }
        switch (proto.getTypeCase()) {
            case THRESHOLD_TEMPLATE -> {
                assertEquals(proto.getThresholdTemplate().getDelay(), protoCloud.getThresholdTemplate().getDelay());
                assertEquals(proto.getThresholdTemplate().getGroupByLabelsList(), protoCloud.getThresholdTemplate().getGroupByLabelsList());
                assertEquals(proto.getThresholdTemplate().getWindow(), protoCloud.getThresholdTemplate().getWindow());
                assertEquals(proto.getThresholdTemplate().getNoPointsPolicy().toString(), protoCloud.getThresholdTemplate().getNoPointsPolicy().toString());
                assertEquals(proto.getThresholdTemplate().getNoMetricsPolicy().toString(), protoCloud.getThresholdTemplate().getNoMetricsPolicy().toString());
                assertEquals(proto.getThresholdTemplate().getQuery(), protoCloud.getThresholdTemplate().getQuery());
                for (int i = 0; i < proto.getThresholdTemplate().getTemplatePredicateRulesList().size(); i++) {
                    var ruleProto = proto.getThresholdTemplate().getTemplatePredicateRulesList().get(i);
                    var ruleProtoCloud = protoCloud.getThresholdTemplate().getTemplatePredicateRulesList().get(i);
                    assertEquals(ruleProto.getThresholdType().toString(), ruleProtoCloud.getThresholdType().toString());
                    assertEquals(ruleProto.getComparison().toString(), ruleProtoCloud.getComparison().toString());
                    assertEquals(ruleProto.getTargetStatus().toString(), ruleProtoCloud.getTargetStatus().toString());
                    assertEquals(ruleProto.getThresholdCase().toString(), ruleProtoCloud.getThresholdCase().toString());
                    switch (ruleProto.getThresholdCase()) {
                        case VALUE -> {
                            assertEquals(ruleProto.getValue(), ruleProtoCloud.getValue(), 0.1);
                        }
                        case THRESHOLD_PARAMETER_TEMPLATE -> {
                            assertEquals(ruleProto.getThresholdParameterTemplate(), ruleProtoCloud.getThresholdParameterTemplate());
                        }
                    }
                }
            }
            case EXPRESSION_TEMPLATE -> {
                assertEquals(proto.getExpressionTemplate().getDelay(), protoCloud.getExpressionTemplate().getDelay());
                assertEquals(proto.getExpressionTemplate().getGroupByLabelsList(), protoCloud.getExpressionTemplate().getGroupByLabelsList());
                assertEquals(proto.getExpressionTemplate().getWindow(), protoCloud.getExpressionTemplate().getWindow());
                assertEquals(proto.getExpressionTemplate().getNoPointsPolicy().toString(), protoCloud.getExpressionTemplate().getNoPointsPolicy().toString());
                assertEquals(proto.getExpressionTemplate().getNoMetricsPolicy().toString(), protoCloud.getExpressionTemplate().getNoMetricsPolicy().toString());
                assertEquals(proto.getExpressionTemplate().getProgram(), protoCloud.getExpressionTemplate().getProgram());
            }
        }
    }
}
