package ru.yandex.solomon.gateway.api.v3.intranet;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monitoring.api.v3.CreateDashboardRequest;
import ru.yandex.monitoring.api.v3.Dashboard;
import ru.yandex.monitoring.api.v3.DeleteDashboardRequest;
import ru.yandex.monitoring.api.v3.GetDashboardRequest;
import ru.yandex.monitoring.api.v3.ListDashboardsRequest;
import ru.yandex.monitoring.api.v3.UpdateDashboardRequest;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolverStub;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.conf.db3.memory.InMemoryMonitoringDashboardsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.gateway.api.v3.DashboardModelFactory;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.DashboardDtoConverter;
import ru.yandex.solomon.gateway.api.v3.intranet.impl.DashboardServiceImpl;
import ru.yandex.solomon.gateway.cloud.search.SearchEvent;
import ru.yandex.solomon.gateway.cloud.search.SearchEventSink;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class DashboardServiceTest {

    private DashboardService service;
    private MonitoringDashboardsDao dao;
    private CloudByFolderResolverStub cloudResolver;
    private StubSearchEventSink eventSink;

    @Before
    public void before() {
        service = new DashboardServiceImpl(
                Authorizer.anonymous(),
                dao = new InMemoryMonitoringDashboardsDao(),
                Optional.of(cloudResolver = new CloudByFolderResolverStub()),
                Optional.empty(),
                Optional.of(eventSink = new StubSearchEventSink())
        );
    }

    @Test(expected = BadRequestException.class)
    public void get_noId() {
        service.get(GetDashboardRequest.newBuilder().build(), AnonymousAuthSubject.INSTANCE).join();
    }

    @Test(expected = NotFoundException.class)
    public void get_notFoundById() {
        try {
            service.get(GetDashboardRequest.newBuilder()
                    .setDashboardId("dashboard")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException ex) {
            Throwables.propagate(Throwables.getRootCause(ex));
        }
    }

    @Test
    public void get_byFolder() {
        var dashboard = DashboardModelFactory.dashboard(true);
        dao.insert(DashboardDtoConverter.toEntity(dashboard, Integer.parseInt(dashboard.getEtag()))).join();
        cloudResolver.add(dashboard.getFolderId(), dashboard.getFolderId() + "cloud");

        var responseDashboard = service.get(GetDashboardRequest.newBuilder()
                .setDashboardId(dashboard.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();

        assertEquals(dashboard, responseDashboard);
    }

    @Test
    public void get_byProject() {
        var dashboard = DashboardModelFactory.dashboard(true).toBuilder()
                .clearFolderId()
                .setProjectId("projectId")
                .build();
        dao.insert(DashboardDtoConverter.toEntity(dashboard, Integer.parseInt(dashboard.getEtag()))).join();

        var responseDashboard = service.get(GetDashboardRequest.newBuilder()
                .setDashboardId(dashboard.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();

        assertEquals(dashboard, responseDashboard);
    }

    @Test(expected = BadRequestException.class)
    public void list_noContainer() {
        service.list(ListDashboardsRequest.newBuilder().build(), AnonymousAuthSubject.INSTANCE).join();
    }

    @Test(expected = BadRequestException.class)
    public void list_badFilter() {
        service.list(ListDashboardsRequest.newBuilder()
                .setProjectId("project")
                .setFilter("123")
                .build(), AnonymousAuthSubject.INSTANCE).join();
    }

    @Test
    public void list_byFolder() {
        List<Dashboard> dashboardList1 = new ArrayList<>();
        for (int i = 0; i < 10; i++) {
            var dashboard = DashboardModelFactory.dashboard(true).toBuilder()
                    .setFolderId(i % 2 + "")
                    .setId(UUID.randomUUID().toString())
                    .setName(UUID.randomUUID().toString())
                    .setEtag("0")
                    .build();
            dao.insert(DashboardDtoConverter.toEntity(dashboard, 0)).join();
            cloudResolver.add(dashboard.getFolderId(), dashboard.getFolderId() + "cloud");
            if (i % 2 != 0) {
                dashboardList1.add(dashboard);
            }
        }
        cloudResolver.add("folderId", "folderIdcloud");

        var response = service.list(ListDashboardsRequest.newBuilder()
                .setProjectId("project")
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(0, response.getDashboardsCount());
        assertEquals("", response.getNextPageToken());

        response = service.list(ListDashboardsRequest.newBuilder()
                .setFolderId("folderId")
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(0, response.getDashboardsCount());
        assertEquals("", response.getNextPageToken());

        response = service.list(ListDashboardsRequest.newBuilder()
                .setFolderId("1")
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(dashboardList1.stream().sorted(Comparator.comparing(Dashboard::getName)).collect(Collectors.toList()), response.getDashboardsList());
        assertEquals("", response.getNextPageToken());

        response = service.list(ListDashboardsRequest.newBuilder()
                .setFolderId("0")
                .setPageSize(2)
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(2, response.getDashboardsCount());
        assertEquals("2", response.getNextPageToken());

        response = service.list(ListDashboardsRequest.newBuilder()
                .setFolderId("0")
                .setPageToken(response.getNextPageToken())
                .setPageSize(100)
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(3, response.getDashboardsCount());
        assertEquals("", response.getNextPageToken());
    }

    @Test
    public void list_byProjectId() {
        List<Dashboard> dashboardList1 = new ArrayList<>();
        for (int i = 0; i < 10; i++) {
            var dashboard = DashboardModelFactory.dashboard(true).toBuilder()
                    .setProjectId(i % 2 + "")
                    .setId(UUID.randomUUID().toString())
                    .setName(UUID.randomUUID().toString())
                    .setEtag("0")
                    .build();
            dao.insert(DashboardDtoConverter.toEntity(dashboard, 0)).join();
            if (i % 2 != 0) {
                dashboardList1.add(dashboard);
            }
        }
        cloudResolver.add("folderId", "folderIdcloud");

        var response = service.list(ListDashboardsRequest.newBuilder()
                .setProjectId("project")
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(0, response.getDashboardsCount());
        assertEquals("", response.getNextPageToken());

        response = service.list(ListDashboardsRequest.newBuilder()
                .setFolderId("folderId")
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(0, response.getDashboardsCount());
        assertEquals("", response.getNextPageToken());

        response = service.list(ListDashboardsRequest.newBuilder()
                .setProjectId("1")
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(dashboardList1.stream().sorted(Comparator.comparing(Dashboard::getName)).collect(Collectors.toList()), response.getDashboardsList());
        assertEquals("", response.getNextPageToken());

        response = service.list(ListDashboardsRequest.newBuilder()
                .setProjectId("0")
                .setPageSize(2)
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(2, response.getDashboardsCount());
        assertEquals("2", response.getNextPageToken());

        response = service.list(ListDashboardsRequest.newBuilder()
                .setProjectId("0")
                .setPageToken(response.getNextPageToken())
                .setPageSize(100)
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals(3, response.getDashboardsCount());
        assertEquals("", response.getNextPageToken());
    }

    @Test(expected = BadRequestException.class)
    public void create_validation() {
        var request = CreateDashboardRequest.newBuilder()
                .setProjectId("dashboard.getFolderId()")
                .setName("Failed name")
                .build();
        service.create(request, AnonymousAuthSubject.INSTANCE).join();
    }

    @Test
    public void create_byFolder() {
        var dashboard = DashboardModelFactory.dashboard(true);
        cloudResolver.add(dashboard.getFolderId(), dashboard.getFolderId() + "cloud");
        var request = CreateDashboardRequest.newBuilder()
                .setFolderId(dashboard.getFolderId())
                .setName("my-dashboard")
                .setDescription(UUID.randomUUID().toString())
                .setTitle(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .addWidgets(dashboard.getWidgetsList().get(0))
                .setParametrization(dashboard.getParametrization())
                .build();

        var created = service.create(request, AnonymousAuthSubject.INSTANCE).join();
        var getDashboard = service.get(GetDashboardRequest.newBuilder()
                .setDashboardId(created.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();

        assertEquals(DashboardDtoConverter.toDashboard(request, getDashboard.getId(), "").toBuilder()
                .setCreatedAt(getDashboard.getCreatedAt())
                .setModifiedAt(getDashboard.getModifiedAt())
                .build(), getDashboard);
        assertSinkState(getDashboard);
    }

    @Test
    public void create_byProject() {
        var dashboard = DashboardModelFactory.dashboard(true);
        var request = CreateDashboardRequest.newBuilder()
                .setProjectId("project")
                .setName("my-dashboard")
                .setDescription(UUID.randomUUID().toString())
                .setTitle(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .addWidgets(dashboard.getWidgetsList().get(0))
                .setParametrization(dashboard.getParametrization())
                .build();

        var created = service.create(request, AnonymousAuthSubject.INSTANCE).join();
        var getDashboard = service.get(GetDashboardRequest.newBuilder()
                .setDashboardId(created.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();

        assertEquals(DashboardDtoConverter.toDashboard(request, getDashboard.getId(), "").toBuilder()
                .setCreatedAt(getDashboard.getCreatedAt())
                .setModifiedAt(getDashboard.getModifiedAt())
                .build(), getDashboard);
    }

    @Test
    public void update_byFolder() {
        var dashboard = DashboardModelFactory.dashboard(true);
        var requestCreate = CreateDashboardRequest.newBuilder()
                .setFolderId(dashboard.getFolderId())
                .setName("my-dashboard")
                .setDescription(UUID.randomUUID().toString())
                .setTitle(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .addWidgets(dashboard.getWidgetsList().get(0))
                .setParametrization(dashboard.getParametrization())
                .build();

        cloudResolver.add(dashboard.getFolderId(), dashboard.getFolderId() + "cloud");
        var created = service.create(requestCreate, AnonymousAuthSubject.INSTANCE).join();

        var request = UpdateDashboardRequest.newBuilder()
                .setDashboardId(created.getId())
                .setName("name")
                .setDescription(UUID.randomUUID().toString())
                .setTitle(UUID.randomUUID().toString())
                .putAllLabels(Map.of("1", "2"))
                .addWidgets(dashboard.getWidgetsList().get(0))
                .setParametrization(dashboard.getParametrization())
                .build();
        var responseDashboard = service.update(request, AnonymousAuthSubject.INSTANCE).join();
        var getDashboard = service.get(GetDashboardRequest.newBuilder()
                .setDashboardId(created.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();

        assertEquals(DashboardDtoConverter.toDashboard(request, "", dashboard, "1").toBuilder()
                .setCreatedAt(getDashboard.getCreatedAt())
                .setModifiedAt(getDashboard.getModifiedAt())
                .setCreatedBy("")
                .setId(getDashboard.getId())
                .build(), responseDashboard);
        assertSinkState(getDashboard);
    }

    @Test
    public void update_byProject() {
        var dashboard = DashboardModelFactory.dashboard(true).toBuilder()
                .clearFolderId()
                .setProjectId("prj")
                .build();
        var requestCreate = CreateDashboardRequest.newBuilder()
                .setProjectId("prj")
                .setName("my-dashboard")
                .setDescription(UUID.randomUUID().toString())
                .setTitle(UUID.randomUUID().toString())
                .putAllLabels(Map.of(UUID.randomUUID().toString(), UUID.randomUUID().toString()))
                .addWidgets(dashboard.getWidgetsList().get(0))
                .setParametrization(dashboard.getParametrization())
                .build();

        var created = service.create(requestCreate, AnonymousAuthSubject.INSTANCE).join();

        var request = UpdateDashboardRequest.newBuilder()
                .setDashboardId(created.getId())
                .setName("name")
                .setDescription(UUID.randomUUID().toString())
                .setTitle(UUID.randomUUID().toString())
                .putAllLabels(Map.of("1", "2"))
                .addWidgets(dashboard.getWidgetsList().get(0))
                .setParametrization(dashboard.getParametrization())
                .build();
        service.update(request, AnonymousAuthSubject.INSTANCE).join();
        var getDashboard = service.get(GetDashboardRequest.newBuilder()
                .setDashboardId(created.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();

        assertEquals(request.getName(), getDashboard.getName());
        assertEquals(request.getDescription(), getDashboard.getDescription());
        assertEquals(request.getTitle(), getDashboard.getTitle());
        assertEquals(request.getLabelsMap(), getDashboard.getLabelsMap());
        assertEquals("1", getDashboard.getEtag());

        service.update(request, AnonymousAuthSubject.INSTANCE).join();
        getDashboard = service.get(GetDashboardRequest.newBuilder()
                .setDashboardId(created.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals("2", getDashboard.getEtag());

        service.update(request.toBuilder().setEtag("2").build(), AnonymousAuthSubject.INSTANCE).join();
        getDashboard = service.get(GetDashboardRequest.newBuilder()
                .setDashboardId(created.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertEquals("3", getDashboard.getEtag());
    }

    @Test
    public void delete_byFolder() {
        var dashboard = DashboardModelFactory.dashboard(true);
        dao.insert(DashboardDtoConverter.toEntity(dashboard, 0)).join();
        cloudResolver.add(dashboard.getFolderId(), dashboard.getFolderId() + "cloud");

        service.delete(DeleteDashboardRequest.newBuilder()
                .setDashboardId(dashboard.getId())
                .build(), AnonymousAuthSubject.INSTANCE).join();
        assertTrue(dao.readAll().join().isEmpty());
        assertEquals(1, eventSink.state.size());
        assertEquals(dashboard.getId(), eventSink.state.get(dashboard.getId()).resourceId);
        assertTrue(eventSink.state.get(dashboard.getId()).deletedAt > 0);
    }

    @Test(expected = NotFoundException.class)
    public void delete_notFoundById() {
        try {
            service.delete(DeleteDashboardRequest.newBuilder()
                    .setDashboardId("dashboard")
                    .build(), AnonymousAuthSubject.INSTANCE).join();
        } catch (CompletionException ex) {
            Throwables.propagate(Throwables.getRootCause(ex));
        }
    }

    private void assertSinkState(Dashboard dashboard) {
        assertEquals(1, eventSink.state.size());
        assertEquals(dashboard.getId(), eventSink.state.get(dashboard.getId()).resourceId);
        assertEquals(dashboard.getFolderId() + "cloud", eventSink.state.get(dashboard.getId()).cloudId);
        assertEquals(dashboard.getFolderId(), eventSink.state.get(dashboard.getId()).folderId);
        assertEquals(dashboard.getTitle(), eventSink.state.get(dashboard.getId()).name);
        assertEquals(dashboard.getDescription(), eventSink.state.get(dashboard.getId()).description);
        assertEquals("dashboard", eventSink.state.get(dashboard.getId()).resourceType);
        assertEquals(0, eventSink.state.get(dashboard.getId()).deletedAt);
    }

    class StubSearchEventSink implements SearchEventSink {

        private Map<String, SearchEvent> state = new HashMap<>();

        @Override
        public void accept(SearchEvent event) {
            state.put(event.resourceId, event);
        }
    }
}
