package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;

import ru.yandex.monitoring.api.v3.UnitFormat;
import ru.yandex.solomon.alert.protobuf.AlertTemplate;
import ru.yandex.solomon.alert.protobuf.AlertTemplateParameter;
import ru.yandex.solomon.alert.protobuf.AlertTemplateStatus;
import ru.yandex.solomon.alert.protobuf.DoubleTemplateParameter;
import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.Expression;
import ru.yandex.solomon.alert.protobuf.IntegerTemplateParameter;
import ru.yandex.solomon.alert.protobuf.LabelListTemplateParameter;
import ru.yandex.solomon.alert.protobuf.NoPointsPolicy;
import ru.yandex.solomon.alert.protobuf.PredicateRule;
import ru.yandex.solomon.alert.protobuf.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TPredicateRule;
import ru.yandex.solomon.alert.protobuf.TextListTemplateParameter;
import ru.yandex.solomon.alert.protobuf.TextTemplateParameter;
import ru.yandex.solomon.alert.protobuf.Threshold;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class AlertTemplateFactory {

    public static AlertTemplate randomAlertTemplate(AlertTemplate.TypeCase type) {
        ThreadLocalRandom random = ThreadLocalRandom.current();

        AlertTemplate.Builder builder = AlertTemplate.newBuilder()
                .setId(UUID.randomUUID().toString())
                .setTemplateVersionTag(UUID.randomUUID().toString())
                .setServiceProviderId(UUID.randomUUID() + " service provider")
                .setName("Name with random: " + random.nextInt())
                .setDescription("Description with random: " + random.nextInt())
                .setCreatedBy("created by")
                .setCreatedAt(Timestamps.fromSeconds(1000))
                .setPeriodMillis(ThreadLocalRandom.current().nextLong(1000, 1_000_00))
                .setDelaySeconds(ThreadLocalRandom.current().nextInt(0, 100))
                .putAllAnnotations(random.nextBoolean() ? Map.of() : Map.of("key", "host"))
                .putAllLabels(random.nextBoolean() ? Map.of() : Map.of("key2", "host2"))
                .addAllGroupByLabels(random.nextBoolean() ? Collections.emptyList() : Collections.singleton("host"))
                .setNoPointsPolicy(NoPointsPolicy.NO_POINTS_NO_DATA)
                .setResolvedEmptyPolicy(ResolvedEmptyPolicy.RESOLVED_EMPTY_DEFAULT)
                .setSeverity(Severity.SEVERITY_CRITICAL)
                .addAllAlertTemplateParameters(getTemplateParameters())
                .setAlertTemplateStatus(random.nextBoolean()
                        ? AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT
                        : AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED)
                .setDefault(random.nextBoolean())
                .addAllAlertTemplateThresholds(getTemplateThresholds());

        switch (type) {
            case EXPRESSION:
                builder.setExpression(randomExpression(random));
                break;
            case THRESHOLD:
                builder.setThreshold(randomThreshold(random));
                break;
            default:
                throw new UnsupportedOperationException("Unsupported alert type: " + type);
        }

        return builder.build();
    }

    public static Threshold randomThreshold(ThreadLocalRandom random) {
        var selectors = "project=solomon, cluster=local, service=test, sensor=idleTime, host=solomon-"
                + random.nextInt(1, 100);

        return Threshold.newBuilder()
                .setSelectors(selectors)
                .addAllPredicateRules(IntStream.range(0, random.nextInt(2, 5))
                        .mapToObj((ignore) -> randomPredicateRule(random))
                        .collect(Collectors.toList()))
                .build();
    }

    private static PredicateRule randomPredicateRule(ThreadLocalRandom random) {
        var builder = PredicateRule.newBuilder()
                .setComparison(ECompare.values()[random.nextInt(0, ECompare.values().length - 1)])
                .setThresholdType(EThresholdType.values()[random.nextInt(0, EThresholdType.values().length - 1)])
                .setTargetStatus(TPredicateRule.ETargetStatus.values()[random.nextInt(1, TPredicateRule.ETargetStatus.values().length - 1)]);
        if (random.nextBoolean()) {
            builder.setThreshold(random.nextDouble());
        } else {
            builder.setThresholdParameterTemplate(random.nextInt() + "");
        }
        return builder.build();
    }

    private static Expression randomExpression(ThreadLocalRandom random) {
        return Expression.newBuilder()
                .setProgram("let rr = random01() < " + random.nextDouble(1, 10) + ";")
                .build();
    }

    private static List<AlertTemplateParameter> getTemplateParameters() {
        return List.of(
                AlertTemplateParameter.newBuilder()
                        .setDoubleParameter(DoubleTemplateParameter.newBuilder()
                                .setName("DoubleParameterValue param")
                                .setTitle("DoubleParameterValue param title")
                                .setDescription("DoubleParameterValue param title")
                                .setDefaultValue(2.1)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setDoubleParameter(DoubleTemplateParameter.newBuilder()
                                .setName("DoubleParameterValue param2")
                                .setTitle("DoubleParameterValue param title2")
                                .setDescription("DoubleParameterValue param descr2")
                                .setDefaultValue(2.12)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setIntegerParameter(IntegerTemplateParameter.newBuilder()
                                .setName("IntegerParameterValue param")
                                .setTitle("IntegerParameterValue param title")
                                .setDescription("IntegerParameterValue param descr")
                                .setDefaultValue(22)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setTextParameter(TextTemplateParameter.newBuilder()
                                .setName("TextParameterValue param")
                                .setTitle("TextParameterValue param title")
                                .setDescription("TextParameterValue param descr")
                                .setDefaultValue("text param")
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setTextListParameter(TextListTemplateParameter.newBuilder()
                                .setName("TextParameterValues param")
                                .setTitle("TextParameterValues param title")
                                .setDescription("TextParameterValues param descr")
                                .addAllDefaultValues(List.of("1", "2", "3"))
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setLabelListParameter(LabelListTemplateParameter.newBuilder()
                                .setName("LabelParameterValues param")
                                .setTitle("LabelParameterValues param title")
                                .setDescription("LabelParameterValues param descr")
                                .setSelectors("selector")
                                .setLabelKey("key")
                                .addAllDefaultValues(List.of("1l", "2l", "3L"))
                                .setProjectId("prj")
                                .setMultiselectable(true)
                                .build())
                        .build()
        );
    }

    private static List<AlertTemplateParameter> getTemplateThresholds() {
        return List.of(
                AlertTemplateParameter.newBuilder()
                        .setDoubleParameter(DoubleTemplateParameter.newBuilder()
                                .setName("DoubleParameterValue threshold")
                                .setTitle("DoubleParameterValue threshold title")
                                .setDescription("DoubleParameterValue threshold descr")
                                .setDefaultValue(12.1)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setDoubleParameter(DoubleTemplateParameter.newBuilder()
                                .setName("DoubleParameterValue threshold2")
                                .setTitle("DoubleParameterValue threshold title2")
                                .setDescription("DoubleParameterValue threshold descr2")
                                .setDefaultValue(12.12)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setIntegerParameter(IntegerTemplateParameter.newBuilder()
                                .setName("IntegerParameterValue threshold")
                                .setTitle("IntegerParameterValue threshold title")
                                .setDescription("IntegerParameterValue threshold descr")
                                .setDefaultValue(122)
                                .setUnitFormat(UnitFormat.values()[ThreadLocalRandom.current().nextInt(UnitFormat.values().length - 1)])
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setTextParameter(TextTemplateParameter.newBuilder()
                                .setName("TextParameterValue threshold")
                                .setTitle("TextParameterValue threshold title")
                                .setDescription("TextParameterValue threshold descr")
                                .setDefaultValue("1text threshold")
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setTextListParameter(TextListTemplateParameter.newBuilder()
                                .setName("TextParameterValues threshold")
                                .setTitle("TextParameterValues threshold title")
                                .setDescription("TextParameterValues threshold descr")
                                .addAllDefaultValues(List.of("0", "1", "2", "3"))
                                .build())
                        .build(),
                AlertTemplateParameter.newBuilder()
                        .setLabelListParameter(LabelListTemplateParameter.newBuilder()
                                .setName("LabelParameterValues threshold")
                                .setTitle("LabelParameterValues threshold title")
                                .setDescription("LabelParameterValues threshold descr")
                                .setSelectors("1selector")
                                .setLabelKey("1key")
                                .addAllDefaultValues(List.of("0L", "1l", "2l", "3L"))
                                .setProjectId("prj")
                                .setMultiselectable(true)
                                .build())
                        .build()
        );
    }
}
