package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Int64Value;
import com.google.protobuf.util.Timestamps;
import org.junit.Test;

import ru.yandex.monitoring.api.v3.ChartWidget;
import ru.yandex.monitoring.api.v3.CreateDashboardRequest;
import ru.yandex.monitoring.api.v3.Dashboard;
import ru.yandex.monitoring.api.v3.Downsampling;
import ru.yandex.monitoring.api.v3.Parametrization;
import ru.yandex.monitoring.api.v3.UpdateDashboardRequest;
import ru.yandex.monitoring.api.v3.Widget;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static ru.yandex.solomon.gateway.api.v3.DashboardModelFactory.dashboard;
import static ru.yandex.solomon.gateway.api.v3.DashboardModelFactory.entity;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DashboardDtoConverterTest {

    @Test
    public void notZeroPrecision() {
        Dashboard dashboard = Dashboard.newBuilder()
                .setId("dashboard")
                .setProjectId("projectId")
                .setEtag("1")
                .addWidgets(Widget.newBuilder()
                        .setPosition(position())
                        .setChart(ChartWidget.newBuilder()
                                .setId("00")
                                .setQueries(queries())
                                .setVisualizationSettings(ChartWidget.VisualizationSettings.newBuilder()
                                        .setType(ChartWidget.VisualizationSettings.VisualizationType.VISUALIZATION_TYPE_STACK)
                                        .setColorSchemeSettings(ChartWidget.VisualizationSettings.ColorSchemeSettings.newBuilder()
                                                .setAutomatic(ChartWidget.VisualizationSettings.ColorSchemeSettings.AutomaticColorScheme.getDefaultInstance())
                                                .build())
                                        .setAggregation(ChartWidget.VisualizationSettings.SeriesAggregation.SERIES_AGGREGATION_AVG)
                                        .setYaxisSettings(ChartWidget.VisualizationSettings.YaxisSettings.newBuilder()
                                                .setLeft(ChartWidget.VisualizationSettings.Yaxis.newBuilder()
                                                        .setPrecision(Int64Value.of(6))
                                                        .build()
                                                )
                                                .build())
                                        .build())
                                .build())
                        .build())
                .setParametrization(Parametrization.getDefaultInstance())
                .setCreatedAt(Timestamps.MIN_VALUE)
                .setModifiedAt(Timestamps.MIN_VALUE)
                .build();

        Entity entity = DashboardDtoConverter.toEntity(dashboard, 1);

        Dashboard decodedDashboard = DashboardDtoConverter.fromEntity(entity);

        assertEquals(dashboard, decodedDashboard);
    }

    @Test
    public void zeroPrecision() {
        Dashboard dashboard = Dashboard.newBuilder()
                .setId("dashboard")
                .setProjectId("projectId")
                .setEtag("0")
                .addWidgets(Widget.newBuilder()
                        .setPosition(position())
                        .setChart(ChartWidget.newBuilder()
                                .setId("00")
                                .setQueries(queries())
                                .setVisualizationSettings(ChartWidget.VisualizationSettings.newBuilder()
                                        .setType(ChartWidget.VisualizationSettings.VisualizationType.VISUALIZATION_TYPE_STACK)
                                        .setAggregation(ChartWidget.VisualizationSettings.SeriesAggregation.SERIES_AGGREGATION_AVG)
                                        .setYaxisSettings(ChartWidget.VisualizationSettings.YaxisSettings.newBuilder()
                                                .setLeft(ChartWidget.VisualizationSettings.Yaxis.newBuilder()
                                                        .setPrecision(Int64Value.of(0))
                                                        .build()
                                                )
                                                .build())
                                        .build())
                                .build())
                        .build())
                .setParametrization(Parametrization.getDefaultInstance())
                .setCreatedAt(Timestamps.MIN_VALUE)
                .setModifiedAt(Timestamps.MIN_VALUE)
                .build();

        Entity entity = DashboardDtoConverter.toEntity(dashboard, 0);

        Dashboard decodedDashboard = DashboardDtoConverter.fromEntity(entity);

        assertEquals(dashboard, decodedDashboard);
    }

    public static ChartWidget.Queries queries() {
        return ChartWidget.Queries.newBuilder()
                .addTargets(ChartWidget.Queries.Target.newBuilder()
                        .setQuery("abs({project=\"solomon\", ...})")
                        .build())
                .setDownsampling(Downsampling.newBuilder()
                        .setDisabled(true)
                        .build())
                .build();
    }

    @Test
    public void emptyPrecision() {
        Dashboard dashboard = Dashboard.newBuilder()
                .setId("dashboard")
                .setEtag("0")
                .setProjectId("projectId")
                .addWidgets(Widget.newBuilder()
                        .setPosition(position())
                        .setChart(ChartWidget.newBuilder()
                                .setId("00")
                                .setQueries(queries())
                                .setVisualizationSettings(ChartWidget.VisualizationSettings.newBuilder()
                                        .setType(ChartWidget.VisualizationSettings.VisualizationType.VISUALIZATION_TYPE_STACK)
                                        .setAggregation(ChartWidget.VisualizationSettings.SeriesAggregation.SERIES_AGGREGATION_AVG)
                                        .setYaxisSettings(ChartWidget.VisualizationSettings.YaxisSettings.newBuilder()
                                                .setLeft(ChartWidget.VisualizationSettings.Yaxis.newBuilder()
                                                        .clearPrecision()
                                                        .build()
                                                )
                                                .build())
                                        .build())
                                .build())
                        .build())
                .setParametrization(Parametrization.getDefaultInstance())
                .setCreatedAt(Timestamps.MIN_VALUE)
                .setModifiedAt(Timestamps.MIN_VALUE)
                .build();

        Entity entity = DashboardDtoConverter.toEntity(dashboard, 0);

        Dashboard decodedDashboard = DashboardDtoConverter.fromEntity(entity);

        assertEquals(dashboard, decodedDashboard);
    }

    @Test
    public void convertToAlphaAndBack_public() {
        var base = dashboard(true);
        var dashboard = base.toBuilder()
                .clearFolderId()
                .setProjectId("projectId")
                .setParametrization(base.getParametrization().toBuilder()
                        .clearParameters()
                        .addParameters(base.getParametrization().getParameters(0).toBuilder()
                                .clearDescription()
                                .setLabelValues(base.getParametrization().getParameters(0).getLabelValues().toBuilder()
                                        .clearFolderId()
                                        .setProjectId("projectId")
                                        .build())
                                .build())
                        .addParameters(base.getParametrization().getParameters(1))
                        .addParameters(base.getParametrization().getParameters(2))
                        .build())
                .build();
        var alphaDashboard = DashboardDtoConverter.toAlphaDashboardModel(dashboard);
        var dashboardConverted = DashboardDtoConverter.fromAlphaDashboardModel(alphaDashboard, dashboard.getName());
        assertEquals(dashboard, dashboardConverted);
    }

    @Test
    public void convertToAlphaAndBack_private() {
        var base = dashboard(false);
        var dashboard = base.toBuilder()
                .clearFolderId()
                .setProjectId("projectId")
                .setParametrization(base.getParametrization().toBuilder()
                        .clearParameters()
                        .addParameters(base.getParametrization().getParameters(0).toBuilder()
                                .clearDescription()
                                .setLabelValues(base.getParametrization().getParameters(0).getLabelValues().toBuilder()
                                        .clearFolderId()
                                        .setProjectId("projectId")
                                        .build())
                                .build())
                        .addParameters(base.getParametrization().getParameters(1))
                        .addParameters(base.getParametrization().getParameters(2))
                        .build())
                .clearWidgets()
                .addWidgets(base.getWidgets(0))
                .addWidgets(base.getWidgets(1))
                .addWidgets(base.getWidgets(2).toBuilder()
                        .setAlert(base.getWidgets(2).getAlert().toBuilder()
                                .clearFolderId()
                                .setProjectId("projectId")
                                .build())
                        .build())
                .addWidgets(base.getWidgets(3))
                .addWidgets(base.getWidgets(4))
                .addWidgets(base.getWidgets(5))
                .addWidgets(base.getWidgets(6))
                .addWidgets(base.getWidgets(7))
                .build();
        var alphaDashboard = DashboardDtoConverter.toAlphaDashboardModel(dashboard);
        var dashboardConverted = DashboardDtoConverter.fromAlphaDashboardModel(alphaDashboard, dashboard.getName());
        assertEquals(dashboard, dashboardConverted);
    }

    @Test
    public void fromEntity() {
        var dashboard = dashboard(true);
        var entity = entity(dashboard);

        var dashboardConverted = DashboardDtoConverter.fromEntity(entity);

        assertEquals(dashboard, dashboardConverted);
    }

    @Test
    public void fromEntityPage() {
        var dashboard = dashboard(true);
        var entity = entity(dashboard);
        var page = new TokenBasePage<>(List.of(entity), "123");

        var response = DashboardDtoConverter.fromEntity(page);

        assertEquals(response.getNextPageToken(), page.getNextPageToken());
        assertEquals(response.getDashboardsList(), List.of(dashboard));
    }

    @Test
    public void toEntity() {
        var dashboard = dashboard(true);
        var entity = DashboardDtoConverter.toEntity(dashboard, Integer.parseInt(dashboard.getEtag()));

        var dashboardConverted = DashboardDtoConverter.fromEntity(entity);
        assertEquals(dashboard, dashboardConverted);

        assertEquals(entity.getId(), dashboard.getId());
        assertEquals(entity.getParentId(), dashboard.getFolderId());
        assertEquals(entity.getName(), dashboard.getTitle());
        assertEquals(entity.getLocalId(), dashboard.getName());
        assertEquals(entity.getDescription(), dashboard.getDescription());
        assertEquals(entity.getCreatedBy(), dashboard.getCreatedBy());
        assertEquals(entity.getUpdatedBy(), dashboard.getModifiedBy());
        assertEquals(entity.getVersion(), Integer.parseInt(dashboard.getEtag()));
        assertEquals(entity.getContainerType(), ContainerType.FOLDER);
        assertEquals(entity.getCreatedAt(), Timestamps.toMillis(dashboard.getCreatedAt()));
        assertEquals(entity.getUpdatedAt(), Timestamps.toMillis(dashboard.getModifiedAt()));
        assertEquals(entity.getData(), "");
        assertNotNull(entity.getProto());
    }

    @Test
    public void toDashboard() {
        var previous = dashboard(true);
        var updateCmd = UpdateDashboardRequest.newBuilder()
                .setDashboardId(previous.getId())
                .setName(UUID.randomUUID().toString())
                .setDescription(UUID.randomUUID().toString())
                .setTitle(UUID.randomUUID().toString())
                .putAllLabels(Map.of("z1", "x1"))
                .addAllWidgets(previous.getWidgetsList())
                .setParametrization(previous.getParametrization())
                .build();
        var dashboard = DashboardDtoConverter.toDashboard(updateCmd, "user", previous, "111");

        assertEquals(previous.toBuilder()
                .setName(updateCmd.getName())
                .setDescription(updateCmd.getDescription())
                .setTitle(updateCmd.getTitle())
                .clearLabels()
                .putAllLabels(updateCmd.getLabelsMap())
                .setModifiedBy("user")
                .setEtag("111")
                .clearModifiedAt()
                .build(), dashboard.toBuilder().clearModifiedAt().build());

    }

    @Test
    public void toDashboard_create() {
        var previous = dashboard(true);
        var cmd = CreateDashboardRequest.newBuilder()
                .setFolderId(UUID.randomUUID().toString())
                .setName(UUID.randomUUID().toString())
                .setDescription(UUID.randomUUID().toString())
                .setTitle(UUID.randomUUID().toString())
                .putAllLabels(Map.of("z1", "x1"))
                .addAllWidgets(previous.getWidgetsList())
                .setParametrization(previous.getParametrization())
                .build();
        var dashboard = DashboardDtoConverter.toDashboard(cmd, "111", "user");

        assertEquals(previous.toBuilder()
                .setId("111")
                .setFolderId(cmd.getFolderId())
                .setName(cmd.getName())
                .setDescription(cmd.getDescription())
                .setTitle(cmd.getTitle())
                .clearLabels()
                .putAllLabels(cmd.getLabelsMap())
                .setModifiedBy("user")
                .setCreatedBy("user")
                .setEtag("0")
                .clearModifiedAt()
                .clearCreatedAt()
                .build(), dashboard.toBuilder().clearModifiedAt().clearCreatedAt().build());

    }

    private static Widget.LayoutPosition position() {
        return Widget.LayoutPosition.newBuilder()
                .setX(0)
                .setY(0)
                .setW(1)
                .setH(1)
                .build();
    }
}
