package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.time.Instant;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;
import org.junit.Test;

import ru.yandex.monitoring.api.v3.QuickLinks;
import ru.yandex.monitoring.api.v3.QuickLinksData;
import ru.yandex.solomon.conf.db3.QuickLinksRecord;

import static org.junit.Assert.assertEquals;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class QuickLinksDtoConverterTest {

    @Test
    public void encode() {
        var proto = QuickLinks.newBuilder()
                .setProjectId("solomon")
                .setData(QuickLinksData.newBuilder()
                        .addItems(simpleProtoItem("simple", "https://yandex.ru"))
                        .addItems(complexProtoItem("complex", simpleProtoItem("Sub item #1", "https://google.ru"), simpleProtoItem("Sub item #2", "https://solomon.yandex-team.ru")))
                        .addItems(complexProtoItem("empty folder"))
                        .build())
                .setCreatedAt(Timestamps.fromMillis(Instant.parse("2020-02-27T00:00:00Z").toEpochMilli()))
                .setModifiedAt(Timestamps.fromMillis(Instant.parse("2020-02-27T00:00:00Z").toEpochMilli()))
                .setCreatedBy("user")
                .setModifiedBy("user")
                .build();

        var actual = QuickLinksDtoConverter.encode(proto, 10);

        var expected = QuickLinksRecord.newBuilder()
                .setProjectId("solomon")
                .setItems(new QuickLinksRecord.LinkItem[] {
                        simpleItem("simple", "https://yandex.ru"),
                        complexItem("complex", simpleItem("Sub item #1", "https://google.ru"), simpleItem("Sub item #2", "https://solomon.yandex-team.ru")),
                        complexItem("empty folder")
                })
                .setCreatedAt(Instant.parse("2020-02-27T00:00:00Z"))
                .setUpdatedAt(Instant.parse("2020-02-27T00:00:00Z"))
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .setVersion(10)
                .build();

        assertEquals(expected, actual);
    }

    @Test
    public void decode() {
        var model = QuickLinksRecord.newBuilder()
                .setProjectId("solomon")
                .setItems(new QuickLinksRecord.LinkItem[] {
                        simpleItem("simple #2", "https://yandex.ru"),
                        complexItem("complex #2", simpleItem("Sub item #1", "https://google.ru"), simpleItem("Sub item #2", "https://solomon.yandex-team.ru")),
                        complexItem("empty folder")
                })
                .setCreatedAt(Instant.parse("2020-02-27T00:00:00Z"))
                .setUpdatedAt(Instant.parse("2020-02-27T00:00:00Z"))
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .setVersion(15)
                .build();

        var actual = QuickLinksDtoConverter.decode(model);

        var expected = QuickLinks.newBuilder()
                .setProjectId("solomon")
                .setData(QuickLinksData.newBuilder()
                        .addItems(simpleProtoItem("simple #2", "https://yandex.ru"))
                        .addItems(complexProtoItem("complex #2", simpleProtoItem("Sub item #1", "https://google.ru"), simpleProtoItem("Sub item #2", "https://solomon.yandex-team.ru")))
                        .addItems(complexProtoItem("empty folder"))
                        .build())
                .setCreatedAt(Timestamps.fromMillis(Instant.parse("2020-02-27T00:00:00Z").toEpochMilli()))
                .setModifiedAt(Timestamps.fromMillis(Instant.parse("2020-02-27T00:00:00Z").toEpochMilli()))
                .setCreatedBy("user")
                .setModifiedBy("user")
                .build();

        assertEquals(expected, actual);
    }

    private QuickLinksData.Item simpleProtoItem(String title, String url) {
        return QuickLinksData.Item.newBuilder()
                .setLink(QuickLinksData.LinkItem.newBuilder().setTitle(title).setUrl(url))
                .build();
    }

    private QuickLinksData.Item complexProtoItem(String title, QuickLinksData.Item ...subItems) {
        return QuickLinksData.Item.newBuilder()
                .setFolder(QuickLinksData.FolderItem.newBuilder()
                        .setTitle(title)
                        .addAllItems(List.of(subItems))
                        .build())
                .build();
    }

    private QuickLinksRecord.LinkItem simpleItem(String title, String url) {
        return new QuickLinksRecord.LinkItem(title, url, new QuickLinksRecord.LinkItem[0]);
    }

    private QuickLinksRecord.LinkItem complexItem(String title, QuickLinksRecord.LinkItem ...items) {
        return new QuickLinksRecord.LinkItem(title, "", items);
    }
}
