package ru.yandex.solomon.gateway.api.v3.intranet.dto;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import org.junit.Test;

import ru.yandex.monitoring.api.v3.AlertConfig;
import ru.yandex.monitoring.api.v3.AlertParameter;
import ru.yandex.monitoring.api.v3.MonitoringConfig;
import ru.yandex.monitoring.api.v3.NotificationChannel;
import ru.yandex.monitoring.api.v3.Resource;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.TNotificationChannelOptions;
import ru.yandex.solomon.gateway.stub.StubRequest;
import ru.yandex.solomon.util.Proto;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.gateway.api.v3.intranet.dto.StubRequestConverter.toCreateAlertsFromTemplateRequest;
import static ru.yandex.solomon.gateway.api.v3.utils.StubRequestIdempotency.CREATE_STUB_REQUEST_OPERATION;

/**
 * @author Nuradil Zhambyl
 */
@ParametersAreNonnullByDefault
public class StubRequestConverterTest {

    private final static String PROJECT_ID = "AdsMonitor";
    private final static String SERVICE_PROVIDER_ID = "alextrushkin_service_provider";
    private final static double EPSILON = 1e-6;

    @Test
    public void toCreateAlertsFromTemplateRequestTest() {
        StubRequest stubRequest = dummyMonitoringStubRequest("1");
        List<Resource> resources = dummyResources();
        String createdBy = "prudent";
        var templateRequest = toCreateAlertsFromTemplateRequest(stubRequest, resources, createdBy);
        MonitoringConfig config = Proto.unpack(stubRequest.stub(), MonitoringConfig.class);
        assertEquals(templateRequest.getTemplateIdsList(), alertIdsList(config));
        assertEquals(templateRequest.getServiceProviderId(), stubRequest.serviceProviderId());
        checkResourcesEqual(templateRequest.getResourcesList(), resources);
        assertTrue(templateRequest.getChannelsMap().isEmpty());
        assertEquals(templateRequest.getProjectId(), config.getProjectId());
        assertEquals(templateRequest.getCreatedBy(), createdBy);
        checkConfigsEqual(templateRequest.getTemplateConfigsList(), config.getAlertConfigsList());
        for (CreateAlertsFromTemplateRequest.TemplateConfigs templateConfigs : templateRequest.getTemplateConfigsList()) {
            assertEquals(templateConfigs.getEscalationsList(), List.of("e1"));
        }
    }

    private void checkConfigsEqual(List<CreateAlertsFromTemplateRequest.TemplateConfigs> templateConfigs,
                                   List<AlertConfig> alertConfigs) {
        assertEquals(templateConfigs.size(), alertConfigs.size());
        for (int i = 0; i < alertConfigs.size(); i++) {
            assertEquals(templateConfigs.get(i).getTemplateId(), alertConfigs.get(i).getAlertTemplateId());
            checkThresholds(templateConfigs.get(i).getAlertThresholdsList(), alertConfigs.get(i).getAlertThresholdsList());
            checkChannels(templateConfigs.get(i).getChannelsMap(), alertConfigs.get(i).getNotificationChannelsList());
        };
    }

    private void checkChannels(Map<String, TNotificationChannelOptions> templateChannels, List<NotificationChannel> alertChannels) {
        assertEquals(templateChannels.size(), alertChannels.size());
        checkEqualIds(templateChannels, alertChannels);
        for (var alertChannel : alertChannels) {
            var id = alertChannel.getNotificationChannelId();
            assertTrue(templateChannels.containsKey(id));
            var channelOptions = templateChannels.get(id);
            assertEquals(channelOptions.getChannelConfig().getRepeatNotifyDelayMillis(), alertChannel.getRepeatNotifyDelayMs());
            assertEquals(channelOptions.getChannelConfig().getNotifyAboutStatusesValueList(), alertChannel.getNotifyAboutStatusesValueList());
        }
    }

    private void checkEqualIds(Map<String, TNotificationChannelOptions> templateChannels, List<NotificationChannel> alertChannels) {
        List<String> alertIds = new ArrayList<>();
        for (var alertChannel : alertChannels) {
            alertIds.add(alertChannel.getNotificationChannelId());
        }
        var templateIds = new ArrayList<>(templateChannels.keySet());
        templateIds.sort(String::compareTo);
        alertIds.sort(String::compareTo);
        assertEquals(templateIds, alertIds);
    }

    private static void checkThresholds(List<ru.yandex.solomon.alert.protobuf.AlertParameter> templateThresholds,
                                        List<AlertParameter> alertThresholds) {
        assertEquals(templateThresholds.size(), alertThresholds.size());
        for (int i = 0; i < alertThresholds.size(); i++) {
            // checking only for double in this test because the type of the element is irrelevant
            var templateValue = templateThresholds.get(i).getDoubleParameterValue();
            var alertValue = alertThresholds.get(i).getDoubleParameterValue();
            assertEquals(templateValue.getName(), alertValue.getName());
            assertEquals(templateValue.getValue(), alertValue.getValue(), EPSILON);
        }
    }

    private List<String> alertIdsList(MonitoringConfig config) {
        List<String> alertIds = new ArrayList<>();
        for (var alertConfig : config.getAlertConfigsList()) {
            alertIds.add(alertConfig.getAlertTemplateId());
        }
        return alertIds;
    }

    private void checkResourcesEqual(List<CreateAlertsFromTemplateRequest.Resource> templateResources,
                                     List<Resource> resources) {
        assertEquals(templateResources.size(), resources.size());
        for (int i = 0; i < resources.size(); i++) {
            var map = new HashMap<>(resources.get(i).getResourceParametersMap());
            map.put("resourceType", "type");
            assertEquals(templateResources.get(i).getResourceParametersMap(), map);
        }
    }

    private static List<Resource> dummyResources() {
        Map<String, String> map = new HashMap<>();
        for (int index = 1; index <= 4; index++) {
            String suffix = Integer.toString(index);
            map.put("k" + suffix, "v" + suffix);
        }
        List<Resource> list = new ArrayList<>();
        list.add(Resource
                .newBuilder()
                .setResourceType("type")
                .putAllResourceParameters(map)
                .build());
        return list;
    }

    private StubRequest dummyMonitoringStubRequest(String id) {
        return new StubRequest(
                id,
                SERVICE_PROVIDER_ID,
                CREATE_STUB_REQUEST_OPERATION,
                Any.pack(dummyStub()),
                5);
    }

    private Map<String, String> dummyResourceParameters() {
        Map<String, String> map = new HashMap<>();
        map.put("k1", "v1");
        map.put("k2", "v2");
        map.put("k3", "v3");
        return map;
    }

    private static List<AlertParameter> dummyAlertThresholds(String id) {
        List<AlertParameter> alertThresholds = new ArrayList<>();
        alertThresholds.add(dummyAlertThreshold(id + "d1"));
        alertThresholds.add(dummyAlertThreshold(id + "d2"));
        alertThresholds.add(dummyAlertThreshold(id + "d3"));
        alertThresholds.add(dummyAlertThreshold(id + "d4"));
        return alertThresholds;
    }

    private static AlertParameter dummyAlertThreshold(String name) {
        return AlertParameter
                .newBuilder()
                .setDoubleParameterValue(dummyDoubleParameterValue(name))
                .build();
    }

    private static AlertParameter.DoubleParameterValue dummyDoubleParameterValue(String name) {
        return AlertParameter.DoubleParameterValue
                .newBuilder()
                .setName(name)
                .setValue(2.4)
                .build();
    }

    private static List<NotificationChannel> dummyNotificationChannels(String id) {
        List<NotificationChannel> notificationChannels = new ArrayList<>();
        notificationChannels.add(dummyNotificationChannel(id + "n1"));
        notificationChannels.add(dummyNotificationChannel(id + "n2"));
        notificationChannels.add(dummyNotificationChannel(id + "n3"));
        return notificationChannels;
    }

    private static NotificationChannel dummyNotificationChannel(String notificationId) {
        return NotificationChannel
                .newBuilder()
                .setNotificationChannelId(notificationId)
                .build();
    }

    private static List<AlertConfig> dummyAlertConfigs() {
        List<AlertConfig> alertConfigs = new ArrayList<>();
        alertConfigs.add(dummyAlertConfig("1"));
        alertConfigs.add(dummyAlertConfig("2"));
        alertConfigs.add(dummyAlertConfig("3"));
        alertConfigs.add(dummyAlertConfig("4"));
        alertConfigs.add(dummyAlertConfig("5"));
        return alertConfigs;
    }

    private static AlertConfig dummyAlertConfig(String id) {
        return AlertConfig
                .newBuilder()
                .setAlertTemplateId(id)
                .addAllNotificationChannels(dummyNotificationChannels(id))
                .addAllEscalations(List.of("e1"))
                .addAllAlertThresholds(dummyAlertThresholds(id))
                .build();
    }

    public static MonitoringConfig dummyStub() {
        return MonitoringConfig
                .newBuilder()
                .setProjectId(PROJECT_ID)
                .addAllAlertConfigs(dummyAlertConfigs())
                .setSeverityValue(3)
                .build();
    }
}
