package ru.yandex.solomon.gateway.api.v3.intranet.impl;

import java.time.Instant;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.monitoring.api.v3.AlertConfig;
import ru.yandex.monitoring.api.v3.AlertParameter;
import ru.yandex.monitoring.api.v3.CompleteStubRequest;
import ru.yandex.monitoring.api.v3.CompleteStubRequestResponse;
import ru.yandex.monitoring.api.v3.CreateStubRequest;
import ru.yandex.monitoring.api.v3.CreateStubRequestResponse;
import ru.yandex.monitoring.api.v3.GetStubRequest;
import ru.yandex.monitoring.api.v3.MonitoringConfig;
import ru.yandex.monitoring.api.v3.NotificationChannel;
import ru.yandex.monitoring.api.v3.Resource;
import ru.yandex.solomon.alert.client.stub.AlertApiStub;
import ru.yandex.solomon.auth.AnonymousAuthSubject;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;
import ru.yandex.solomon.gateway.inject.spring.GatewayIdempotentOperationContext;
import ru.yandex.solomon.gateway.stub.dao.ydb.YdbStubRequestDao;
import ru.yandex.solomon.idempotency.IdempotentOperationService;
import ru.yandex.solomon.idempotency.IdempotentOperationServiceImpl;
import ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationDao;
import ru.yandex.solomon.kikimr.LocalKikimr;
import ru.yandex.solomon.kikimr.YdbHelper;

import static org.junit.Assert.assertEquals;

/**
 * @author Nuradil Zhambyl
 */

@ParametersAreNonnullByDefault
public class AlertingStubRequestServiceTest {
    private AlertingStubRequestServiceImpl service;
    private YdbStubRequestDao dao;
    private YdbIdempotentOperationDao opDao;
    private IdempotentOperationService idempotentOperationService;
    private AuthSubject authSubject;

    private final static String PROJECT_ID = "prudent_project_id";
    private final static String SERVICE_PROVIDER_ID = "managed-postgresql";
    private final static String TEMPLATE_ID = "managed-postgresql-transaction-0.9";
    private final static String CHANNEL_ID = "prudent_channel_id";

    private static List<Project> dummyProjectsList() {
        List<Project> projects = new LinkedList<>();
        projects.add(dummyProject());
        return projects;
    }

    private static List<ServiceProvider> dummyServiceProvidersList() {
        List<ServiceProvider> serviceProviders = new LinkedList<>();
        serviceProviders.add(dummyServiceProvider());
        return serviceProviders;
    }

    private static SolomonConfHolder dummyConfHolder() {
        SolomonRawConf rawConf = new SolomonRawConf(
                dummyServiceProvidersList(),
                dummyProjectsList(),
                new LinkedList<>(),
                new LinkedList<>(),
                new LinkedList<>());
        SolomonConfHolder confHolder = new SolomonConfHolder();
        var confWithContext = SolomonConfWithContext.create(rawConf);
        confHolder.onConfigurationLoad(confWithContext);
        return confHolder;
    }

    private static Project dummyProject() {
        return Project.newBuilder()
                .setId(PROJECT_ID)
                .setName("other_name")
                .setOwner("other_user")
                .setOnlyAuthPush(true)
                .setAbcService("other_abc_service")
                .setCreatedBy("other-robot-solomon")
                .setUpdatedBy("other-robot-solomon")
                .build();
    }

    private static ServiceProvider dummyServiceProvider() {
        return ServiceProvider.newBuilder()
                .setId(SERVICE_PROVIDER_ID)
                .setDescription("description")
                .setShardSettings(new ServiceProviderShardSettings(
                        ServiceMetricConf.of(new ServiceMetricConf.AggrRule[]{ServiceMetricConf.AggrRule.of("host=*", "host=cluster", null)}, true),
                        30,
                        15,
                        15
                ))
                .setAbcService("solomon")
                .setCloudId("yc-solomon")
                .setTvmDestId("123")
                .setIamServiceAccountIds(List.of("456", "457"))
                .setCreatedAt(Instant.parse("2020-01-01T00:00:00Z"))
                .setUpdatedAt(Instant.parse("2020-01-02T00:00:00Z"))
                .setCreatedBy("user1")
                .setUpdatedBy("user2")
                .build();
    }

    @ClassRule
    public static LocalKikimr kikimr = new LocalKikimr();

    @Rule
    public TestName testName = new TestName();

    @Before
    public void setUp() throws Exception {
        var ydb = new YdbHelper(kikimr, this.getClass().getSimpleName() + "_" + testName.getMethodName());
        var root = ydb.getRootPath();

        dao = new YdbStubRequestDao(root, root + GatewayIdempotentOperationContext.PATH, ydb.getTableClient(), ydb.getSchemeClient());
        opDao = new YdbIdempotentOperationDao(root + GatewayIdempotentOperationContext.PATH, ydb.getTableClient(), ydb.getSchemeClient());
        opDao.createSchemaForTests().join();
        dao.createSchemaForTests().join();

        idempotentOperationService = new IdempotentOperationServiceImpl(opDao);
        service = new AlertingStubRequestServiceImpl(dao, idempotentOperationService, dummyConfHolder(), Authorizer.anonymous(), new AlertApiStub(), new ResourceServiceStub());
        authSubject = AnonymousAuthSubject.INSTANCE;
    }

    @Test(expected = Exception.class)
    public void get_noId() {
        service.get(dummyGetStubRequest("1"), authSubject).join();
    }

    @Test
    public void createNew() {
        assertEquals(service.create(dummyCreateStubRequest("1"), authSubject)
                .join(), dummyCreateStubRequestResponse("1"));
    }

    @Test
    public void createDuplicate() {
        service.create(dummyCreateStubRequest("1"), authSubject).join();
        assertEquals(service.create(dummyCreateStubRequest("1"), authSubject).join(),
                dummyCreateStubRequestResponse("1"));
    }

    @Test
    public void getId() {
        service.create(dummyCreateStubRequest("1"), authSubject).join();
        assertEquals(service.get(dummyGetStubRequest("1"), authSubject)
                .join().getId(), "1");
    }

    @Test
    public void deletePresent() {
        service.create(dummyCreateStubRequest("1"), authSubject).join();
        assertEquals(service.complete(dummyCompleteStubRequest("1"), authSubject).join(),
                dummyCompleteStubRequestResponse("1"));
    }

    @Test
    public void deleteIdempotent() {
        service.create(dummyCreateStubRequest("5"), authSubject).join();
        assertEquals(service.complete(dummyCompleteStubRequest("5"), authSubject).join(),
                dummyCompleteStubRequestResponse("5"));
        assertEquals(service.complete(dummyCompleteStubRequest("5"), authSubject).join(),
                dummyCompleteStubRequestResponse("5"));
    }

    @Test
    public void deleteDifferent() {
        service.create(dummyCreateStubRequest("4"), authSubject).join();
        assertEquals(service.complete(dummyCompleteStubRequest("4"), authSubject).join(),
                dummyCompleteStubRequestResponse("4"));
        service.create(dummyCreateStubRequest("5"), authSubject).join();
        assertEquals(service.complete(dummyCompleteStubRequest("5"), authSubject).join(),
                dummyCompleteStubRequestResponse("5"));
        assertEquals(service.complete(dummyCompleteStubRequest("4"), authSubject).join(),
                dummyCompleteStubRequestResponse("4"));
    }

    @Test(expected = Exception.class)
    public void deleteAbsent() {
        service.complete(dummyCompleteStubRequest("1"), authSubject).join();
    }

    private CreateStubRequestResponse dummyCreateStubRequestResponse(String id) {
        return CreateStubRequestResponse
                .newBuilder()
                .setId(id)
                .build();
    }

    private CompleteStubRequestResponse dummyCompleteStubRequestResponse(String id) {
        return CompleteStubRequestResponse
                .newBuilder()
                .setId(id)
                .build();
    }

    private static List<Resource> dummyResources() {
        Map<String, String> map = new HashMap<>();
        map.put("cluster", "1234");
        List<Resource> list = new LinkedList<>();
        list.add(Resource
                .newBuilder()
                .putAllResourceParameters(map)
                .build());
        return list;
    }

    private static CompleteStubRequest dummyCompleteStubRequest(String id) {
        return CompleteStubRequest
                .newBuilder()
                .setId(id)
                .addAllResources(dummyResources())
                .build();
    }

    private static GetStubRequest dummyGetStubRequest(String id) {
        return GetStubRequest
                .newBuilder()
                .setId(id)
                .build();
    }

    private static List<AlertParameter> dummyAlertThresholds(String id) {
        List<AlertParameter> alertThresholds = new LinkedList<>();
        alertThresholds.add(dummyAlertThreshold());
        return alertThresholds;
    }

    private static AlertParameter dummyAlertThreshold() {
        return AlertParameter
                .newBuilder()
                .setTextParameterValue(dummyTextParameterValue())
                .build();
    }

    private static AlertParameter.TextParameterValue dummyTextParameterValue() {
        return AlertParameter.TextParameterValue
                .newBuilder()
                .setValue("1234")
                .setName("cluster")
                .build();
    }

    private static List<NotificationChannel> dummyNotificationChannels() {
        List<NotificationChannel> notificationChannels = new LinkedList<>();
        notificationChannels.add(dummyNotificationChannel(CHANNEL_ID));
        return notificationChannels;
    }

    private static NotificationChannel dummyNotificationChannel(String notificationId) {
        return NotificationChannel
                .newBuilder()
                .setNotificationChannelId(notificationId)
                .build();
    }

    private static List<AlertConfig> dummyAlertConfigs() {
        List<AlertConfig> alertConfigs = new LinkedList<>();
        alertConfigs.add(dummyAlertConfig(TEMPLATE_ID));
        return alertConfigs;
    }

    private static AlertConfig dummyAlertConfig(String id) {
        return AlertConfig
                .newBuilder()
                .setAlertTemplateId(id)
                .addAllNotificationChannels(dummyNotificationChannels())
                .addAllAlertThresholds(dummyAlertThresholds(id))
                .build();
    }

    public static MonitoringConfig dummyStub() {
        return MonitoringConfig
                .newBuilder()
                .setProjectId(PROJECT_ID)
                .addAllAlertConfigs(dummyAlertConfigs())
                .setSeverityValue(3)
                .build();
    }

    private static CreateStubRequest dummyCreateStubRequest(String id) {
        return CreateStubRequest
                .newBuilder()
                .setId(id)
                .setServiceProviderId(SERVICE_PROVIDER_ID)
                .setStub(dummyStub())
                .build();
    }
}
