package ru.yandex.solomon.gateway.dao;

import java.util.concurrent.CompletionException;

import com.google.common.base.Throwables;
import com.google.protobuf.Any;
import com.google.protobuf.Int32Value;
import org.junit.Test;

import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.gateway.stub.StubRequest;
import ru.yandex.solomon.gateway.stub.dao.StubRequestDao;
import ru.yandex.solomon.idempotency.IdempotentOperation;
import ru.yandex.solomon.idempotency.IdempotentOperationExistException;
import ru.yandex.solomon.idempotency.dao.IdempotentOperationDao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.gateway.api.v3.utils.StubRequestIdempotency.CREATE_STUB_REQUEST_OPERATION;
import static ru.yandex.solomon.gateway.api.v3.utils.StubRequestIdempotency.DELETE_STUB_REQUEST_OPERATION;
import static ru.yandex.solomon.idempotency.IdempotentOperation.NO_OPERATION;

/**
 * @author Nuradil Zhambyl
 */
public abstract class StubRequestDaoTest {
    protected abstract StubRequestDao getDao();
    protected abstract IdempotentOperationDao getOperationsDao();

    @Test
    public void getStubRequest() {
        var sr = dummyStubRequest("1");
        var dao  = getDao();
        dao.insert(sr, NO_OPERATION).join();
        assertEquals(get(sr, dao), sr);
    }

    @Test
    public void deleteOneStubRequest() {
        var sr = dummyStubRequest("1");
        var dao = getDao();
        dao.insert(sr, NO_OPERATION).join();
        assertTrue(dao.deleteOne(sr.id(), NO_OPERATION).join());
        assertNull(get(sr, dao));
    }

    @Test // works on default which is with NO_OPERATION
    public void insertStubRequest() {
        StubRequest sr = dummyStubRequest("1");
        var dao = getDao();
        dao.insert(sr, NO_OPERATION).join();
        assertEquals(get(sr, dao), sr);
    }

    @Test // idempotency works
    public void insertStubRequest_IdempotentExist() {
        try {
            var sr1 = dummyStubRequest("1");
            var sr2 = dummyStubRequest("2");
            var sr3 = dummyStubRequest("3");
            var op1 = dummyOperation("1", CREATE_STUB_REQUEST_OPERATION);
            var op2 = dummyOperation("2", CREATE_STUB_REQUEST_OPERATION);
            var dao = getDao();
            dao.insert(sr1, op1).join();
            dao.insert(sr2, op2).join();
            dao.insert(sr3, op1).join();

        } catch (CompletionException e) {
            assertTrue(Throwables.getRootCause(e) instanceof IdempotentOperationExistException);
        }
    }

    @Test // no conflict
    public void insertStubRequest_Idempotent() {
        var sr1 = dummyStubRequest("1");
        var sr2 = dummyStubRequest("2");
        var op1 = dummyOperation("1", CREATE_STUB_REQUEST_OPERATION);
        var op2 = dummyOperation("2", CREATE_STUB_REQUEST_OPERATION);
        var dao = getDao();
        dao.insert(sr1, op1).join();
        dao.insert(sr2, op2).join();
        assertEquals(get(sr1, dao), sr1);
        assertEquals(get(sr2, dao), sr2);
        var opDao = getOperationsDao();
        assertEquals(opDao.get(op1.id(), op1.containerId(), op1.containerType(), op1.operationType()).join().orElse(null), op1);
        assertEquals(opDao.get(op2.id(), op2.containerId(), op2.containerType(), op2.operationType()).join().orElse(null), op2);
    }

    @Test // NO_OPERATION isn't added in OperationsDao
    public void insertStubRequest_noIdempotent() {
        var sr = dummyStubRequest("1");
        var dao = getDao();
        dao.insert(sr, NO_OPERATION).join();
        assertEquals(get(sr, dao), sr);
        var op = getOperationsDao().get(NO_OPERATION.id(), NO_OPERATION.containerId(), NO_OPERATION.containerType(), NO_OPERATION.operationType()).join();
        assertTrue(op.isEmpty());
    }

    @Test
    public void deleteOperation() {
        var sr = dummyStubRequest("7");
        var dao = getDao();
        assertTrue(dao.insert(sr, dummyOperation("7", CREATE_STUB_REQUEST_OPERATION)).join());
        assertTrue(dao.deleteOne(sr.id(), dummyOperation("7", DELETE_STUB_REQUEST_OPERATION)).join());
    }

    @Test(expected = Exception.class)
    public void deleteIdempotent() {
        var sr = dummyStubRequest("7");
        var dao = getDao();
        assertTrue(dao.insert(sr, dummyOperation("7", CREATE_STUB_REQUEST_OPERATION)).join());
        assertTrue(dao.deleteOne(sr.id(), dummyOperation("7", DELETE_STUB_REQUEST_OPERATION)).join());
        dao.deleteOne(sr.id(), dummyOperation("7", DELETE_STUB_REQUEST_OPERATION)).join();
    }

    @Test
    public void deleteNonexistent() {
        assertFalse(getDao().deleteOne("7", NO_OPERATION).join());
        assertFalse(getDao().deleteOne("7", dummyOperation("7", DELETE_STUB_REQUEST_OPERATION)).join());
    }

    private StubRequest dummyStubRequest(String id) {
        return new StubRequest(
                id,
                "service",
                CREATE_STUB_REQUEST_OPERATION,
                Any.pack(Int32Value.of(42)),
                5);
    }

    private IdempotentOperation dummyOperation(String id, String type) {
        return new IdempotentOperation(
                id,
                id,
                ContainerType.SERVICE_PROVIDER,
                type,
                id,
                Any.pack(Int32Value.of(42)),
                1);
    }

    private StubRequest get(StubRequest sr, StubRequestDao dao) {
        return dao.get(sr.id()).join().orElse(null);
    }
}
