package ru.yandex.solomon.gateway.data;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.tuple.Pair;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.core.conf.SolomonConfStub;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.model.timeseries.decim.DecimPoliciesPredefined;
import ru.yandex.solomon.util.time.Interval;

import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.util.time.DurationUtils.formatDurationMillis;

/**
 * @author Vladimir Gordiychuk
 */
public class DataClientRequestCustomizerTest {

    private SolomonConfStub conf;
    private SolomonConfHolder confHolder;
    private FeatureFlagHolderStub featureFlagHolder;
    private DataClientRequestCustomizer customizer;

    @Before
    public void setUp() {
        conf = new SolomonConfStub();
        featureFlagHolder = new FeatureFlagHolderStub();
        confHolder = new SolomonConfHolder();
        customizer = new DataClientRequestCustomizer(confHolder, featureFlagHolder);
        configUpdated();
    }

    @Test
    public void skipOverrideDefinedManually() {
        addShard("solomon", "test", "gateway");
        var selectors = List.of(
                Selectors.parse("project=solomon, cluster=test, service=gateway"));
        var interval = interval(1, ChronoUnit.DAYS);
        var opts = DownsamplingOptions.newBuilder()
                .setGridMillis(TimeUnit.SECONDS.toMillis(2L))
                .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                .setShardGridMillis(15000)
                .build();

        var result = customize(opts, interval, selectors);
        assertEquals(opts, result);
    }

    @Test
    public void skipOverrideShardUnknown() {
        var selectors = List.of(
                Selectors.parse("project=solomon, cluster=test, service=gateway"));
        var interval = interval(1, ChronoUnit.DAYS);
        var opts = DownsamplingOptions.newBuilder()
                .setPoints(100)
                .setDownsamplingType(DownsamplingType.BY_POINTS)
                .setShardGridMillis(15000)
                .build();

        var result = customize(opts, interval, selectors);
        assertEquals(opts, result);
    }

    @Test
    public void autoGridCanNotBeLessThenShardGrid() {
        var shard = addShard("solomon", "prod", "gateway");
        int shardGridSec = 10;
        setGrid(shard, shardGridSec);

        var selectors = List.of(
                Selectors.parse("project=solomon, cluster=prod, service=gateway"));
        var interval = interval(10, ChronoUnit.MINUTES);
        var opts = DownsamplingOptions.newBuilder()
                .setDownsamplingType(DownsamplingType.BY_POINTS)
                .setPoints(100)
                .setShardGridMillis(15000)
                .build();
        var result = customize(opts, interval, selectors);
        assertThat(result.getGridMillis(), greaterThanOrEqualTo(TimeUnit.SECONDS.toMillis(shardGridSec)));
        assertEquals(DownsamplingType.BY_INTERVAL, result.getDownsamplingType());
    }

    @Test
    public void autoGridAlignedByShardGrid() {
        var shard = addShard("one", "two", "tree");
        int shardGridSec = 15;
        setGrid(shard, shardGridSec);
        enableGridAlight(shard);
        Set<Long> possibleGridMillis = Set.of(15_000L, 30_000L, 45_000L, 60_000L);

        var selectors = List.of(Selectors.parse("project=one, cluster=two, service=tree"));
        for (int min = 1; min <= 60; min++) {
            var interval = interval(min, ChronoUnit.MINUTES);
            var opts = DownsamplingOptions.newBuilder()
                    .setDownsamplingType(DownsamplingType.BY_POINTS)
                    .setPoints(100)
                    .build();

            var result = customize(opts, interval, selectors);
            assertTrue(formatDurationMillis(result.getGridMillis()), possibleGridMillis.contains(result.getGridMillis()));
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setGridMillis(result.getGridMillis())
                    .setShardGridMillis(TimeUnit.SECONDS.toMillis(shardGridSec))
                    .build();
            assertEquals(expectedOpts, result);
        }
    }

    @Test
    public void autoGridCeiledByShardGrid() {
        var shard = addShard("one", "two", "tree", DecimPolicy.POLICY_KEEP_FOREVER);
        int shardGridSec = 15;
        setGrid(shard, shardGridSec);
        enableGridAlight(shard);

        List<Pair<Duration, Long>> intervalAndGridPairs = List.of(
            Pair.of(Duration.ofMinutes(1), 15000L),
            Pair.of(Duration.ofMinutes(2), 15000L),
            Pair.of(Duration.ofMinutes(5), 15000L),
            Pair.of(Duration.ofMinutes(10), 15000L),
            Pair.of(Duration.ofMinutes(15), 15000L),
            Pair.of(Duration.ofMinutes(30), 30000L),
            Pair.of(Duration.ofMinutes(45), 30000L),
            Pair.of(Duration.ofHours(1), 45000L),
            Pair.of(Duration.ofHours(2), 75000L),
            Pair.of(Duration.ofHours(3), 120000L),
            Pair.of(Duration.ofHours(6), 225000L),
            Pair.of(Duration.ofHours(12), 450000L),
            Pair.of(Duration.ofDays(1), 885000L),
            Pair.of(Duration.ofDays(2), 1755000L),
            Pair.of(Duration.ofDays(3), 2625000L),
            Pair.of(Duration.ofDays(7), 6120000L),
            Pair.of(Duration.ofDays(14), 12225000L),
            Pair.of(Duration.ofDays(31), 27060000L)
        );

        var selectors = List.of(Selectors.parse("project=one, cluster=two, service=tree"));
        for (var intervalAndGridPair : intervalAndGridPairs) {
            long min = intervalAndGridPair.getLeft().toMinutes();
            long grid = intervalAndGridPair.getRight();
            var interval = interval(min, ChronoUnit.MINUTES);
            var opts = DownsamplingOptions.newBuilder()
                    .setDownsamplingType(DownsamplingType.BY_POINTS)
                    .setPoints(100)
                    .build();

            var result = customize(opts, interval, selectors);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setGridMillis(grid)
                    .setShardGridMillis(TimeUnit.SECONDS.toMillis(shardGridSec))
                    .build();
            assertEquals(intervalAndGridPair.toString(), expectedOpts, result);
        }
    }

    @Test
    public void multiShardRequestUseCommonGrid() {
        var alice = addShard("solomon", "test", "alice");
        setGrid(alice, 20);
        var bob = addShard("solomon", "test", "bob");
        setGrid(bob, 15);
        var eva = addShard("solomon", "test", "eva");
        setGrid(eva, 10);
        var mark = addShard("solomon", "test", "mark");
        setGrid(mark, 5);
        enableGridAlight(alice, bob, eva, mark);

        var selectors = List.of(Selectors.parse("project=solomon, cluster=test, service=*"));
        for (int min = 1; min <= 60; min++) {
            var interval = interval(60, ChronoUnit.MINUTES);
            var opts = DownsamplingOptions.newBuilder()
                    .setDownsamplingType(DownsamplingType.BY_POINTS)
                    .setPoints(100)
                    .build();

            var result = customize(opts, interval, selectors);
            var expected = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setGridMillis(TimeUnit.MINUTES.toMillis(1L))
                    .setShardGridMillis(60000) // lcm of (20, 15, 10, 5)
                    .build();
            assertEquals("", expected, result);
        }
    }

    @Test
    public void useDecimPolicyGridWhenIntersectsWithAge() {
        var definedDecimPolicies = DecimPolicy.values();
        var shardGridSec = 15;
        var shardGridMillis = TimeUnit.SECONDS.toMillis(shardGridSec);
        var now = Instant.now();

        var intervals = List.of(
                intervalBefore(now, Duration.ofMinutes(1)),
                intervalBefore(now, Duration.ofMinutes(2)),
                intervalBefore(now, Duration.ofMinutes(5)),
                intervalBefore(now, Duration.ofMinutes(10)),
                intervalBefore(now, Duration.ofMinutes(15)),
                intervalBefore(now, Duration.ofMinutes(30)),
                intervalBefore(now, Duration.ofMinutes(45)),
                intervalBefore(now, Duration.ofHours(1)),
                intervalBefore(now, Duration.ofHours(2)),
                intervalBefore(now, Duration.ofHours(3)),
                intervalBefore(now, Duration.ofHours(6)),
                intervalBefore(now, Duration.ofHours(12)),
                intervalBefore(now, Duration.ofDays(1)),
                intervalBefore(now, Duration.ofDays(2)),
                intervalBefore(now, Duration.ofDays(3)),
                intervalBefore(now, Duration.ofDays(7)),
                intervalBefore(now, Duration.ofDays(8)),
                intervalBefore(now, Duration.ofDays(14)),
                intervalBefore(now, Duration.ofDays(30)),
                intervalBefore(now, Duration.ofDays(31)),
                intervalBefore(now, Duration.ofDays(62)),
                intervalBefore(now, Duration.ofDays(93)),
                intervalBefore(now, Duration.ofDays(100))
        );

        var opts = DownsamplingOptions.newBuilder()
                .setDownsamplingType(DownsamplingType.BY_POINTS)
                .setPoints(100)
                .build();

        for (DecimPolicy decimPolicy : definedDecimPolicies) {
            var shard = addShard("solomon", "useDecimPolicyGridWhenIntersectsWithAge", decimPolicy.name(), decimPolicy);
            var selectors = List.of(Selectors.parse("project=solomon, cluster=useDecimPolicyGridWhenIntersectsWithAge, service=" + decimPolicy.name()));
            setGrid(shard, shardGridSec);
            enableGridAlight(shard);
            var policy = DecimPoliciesPredefined.policyFromProto(decimPolicy.toProto());
            assertNotNull(policy);
            for (Interval interval : intervals) {
                var item = policy.findOldest(interval.getBeginMillis(), now.toEpochMilli());
                boolean itemPolicyOnScreen = null != item;

                var result = customize(opts, interval, selectors);
                String errorMessage = "Failed interval=" + interval + " with policyItem=" + item;

                if (itemPolicyOnScreen) {
                    assertEquals(errorMessage, 0, result.getGridMillis() % item.getStepMillis());
                    assertEquals(errorMessage, item.getStepMillis(), result.getShardGridMillis());
                }
                assertEquals(errorMessage, 0, result.getGridMillis() % shardGridMillis);
                assertThat(errorMessage, opts.getPoints() + 0l, greaterThanOrEqualTo((interval.getEndMillis() - interval.getBeginMillis()) / result.getGridMillis()));
            }
        }
    }

    @Test
    public void decimPolicyMerging() {
        var shardGridSec = 15;
        var shardGridMillis = TimeUnit.SECONDS.toMillis(shardGridSec);
        var now = Instant.now();

        var shard = addShard("solomon", "decimPolicyMerging", "mergeDecim", DecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS);
        setGrid(shard, shardGridSec);

        setGrid(addShard("solomon", "decimPolicyMerging", "mergeDecim2", DecimPolicy.POLICY_5_MIN_AFTER_7_DAYS), shardGridSec);
        setGrid(addShard("solomon", "decimPolicyMerging", "mergeDecim3", DecimPolicy.POLICY_KEEP_FOREVER), shardGridSec);
        setGrid(addShard("solomon", "decimPolicyMerging", "mergeDecim4", DecimPolicy.POLICY_5_MIN_AFTER_8_DAYS), shardGridSec);

        var policiesLcmMillis = TimeUnit.MINUTES.toMillis(5);

        var opts = DownsamplingOptions.newBuilder()
                .setDownsamplingType(DownsamplingType.BY_POINTS)
                .setPoints(100)
                .build();
        var selectors = List.of(Selectors.parse("project=solomon, cluster=decimPolicyMerging, service=mergeD*"));

        {
            var interval = intervalBefore(now, Duration.ofMinutes(3));
            var result = customize(opts, interval, selectors);
            var errorMessage = "Failed interval=" + interval;
            assertTrue(errorMessage, result.getGridMillis() > 0);
            assertEquals(errorMessage, result.getGridMillis(), shardGridMillis);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(result.getGridMillis())
                    .setShardGridMillis(shardGridMillis)
                    .build();
            assertEquals(errorMessage, expectedOpts, result);
        }

        {
            var interval = intervalBefore(now.minus(Duration.ofDays(7)), Duration.ofMinutes(10));
            var result = customize(opts, interval, selectors);
            var errorMessage = "Failed interval=" + interval;
            assertTrue(errorMessage, result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(policiesLcmMillis)
                    .setShardGridMillis(policiesLcmMillis)
                    .build();
            assertEquals(errorMessage, expectedOpts, result);
        }

        {
            var interval = intervalBefore(now.minus(Duration.ofDays(10)), Duration.ofMinutes(10));
            var result = customize(opts, interval, selectors);
            var errorMessage = "Failed interval=" + interval;
            assertTrue(errorMessage, result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(policiesLcmMillis)
                    .setShardGridMillis(policiesLcmMillis)
                    .build();
            assertEquals(errorMessage, expectedOpts, result);
        }
    }

    @Test
    public void fiveMinutesAfter7Days() {
        var shardGridSec = 15;
        var now = Instant.now();

        var shard = addShard("solomon", "fiveMinutesAfter7Days", DecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.name(), DecimPolicy.POLICY_5_MIN_AFTER_7_DAYS);
        setGrid(shard, shardGridSec);

        var policy = DecimPoliciesPredefined.policyFromProto(DecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.toProto());
        assertNotNull(policy);

        var opts = DownsamplingOptions.newBuilder()
                .setDownsamplingType(DownsamplingType.BY_POINTS)
                .setPoints(100)
                .build();
        var selectors = List.of(Selectors.parse("project=solomon, cluster=fiveMinutesAfter7Days, service=" + DecimPolicy.POLICY_5_MIN_AFTER_7_DAYS.name()));

        {
            var interval = intervalBefore(now, Duration.ofMinutes(3));
            var result = customize(opts, interval, selectors);
            assertTrue(result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(result.getGridMillis())
                    .setShardGridMillis(TimeUnit.SECONDS.toMillis(shardGridSec))
                    .build();
            assertEquals(expectedOpts, result);
        }

        {
            var interval = intervalBefore(now, Duration.ofMinutes(10));
            var result = customize(opts, interval, selectors, interval.getBegin().minus(Duration.ofDays(7)).toEpochMilli());
            assertTrue(result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(policy.getItems()[0].getStepMillis())
                    .setShardGridMillis(policy.getItems()[0].getStepMillis())
                    .build();
            assertEquals(expectedOpts, result);
        }

        {
            var interval = intervalBefore(now.minus(Duration.ofDays(7)), Duration.ofMinutes(10));
            var result = customize(opts, interval, selectors);
            assertTrue(result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(policy.getItems()[0].getStepMillis())
                    .setShardGridMillis(policy.getItems()[0].getStepMillis())
                    .build();
            assertEquals(expectedOpts, result);
        }

        {
            var interval = intervalBefore(now.minus(Duration.ofDays(10)), Duration.ofMinutes(10));
            var result = customize(opts, interval, selectors);
            assertTrue(result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(policy.getItems()[0].getStepMillis())
                    .setShardGridMillis(policy.getItems()[0].getStepMillis())
                    .build();
            assertEquals(expectedOpts, result);
        }
    }

    @Test
    public void yasmDefaultDecimPolicy() {
        var shardGridSec = 15;
        var now = Instant.now();

        var shard = addShard("yasm_solomon", "yasmDefaultDecimPolicy", DecimPolicy.UNDEFINED.name(), DecimPolicy.UNDEFINED);
        setGrid(shard, shardGridSec);

        var policy = DecimPoliciesPredefined.policyFromProto(DecimPolicy.POLICY_5_MIN_AFTER_8_DAYS.toProto());
        assertNotNull(policy);

        var opts = DownsamplingOptions.newBuilder()
                .setDownsamplingType(DownsamplingType.BY_POINTS)
                .setPoints(100)
                .build();
        var selectors = List.of(Selectors.parse("project=yasm_solomon, cluster=yasmDefaultDecimPolicy, service=" + DecimPolicy.UNDEFINED.name()));

        {
            var interval = intervalBefore(now, Duration.ofMinutes(3));
            var result = customize(opts, interval, selectors);
            assertTrue(result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(result.getGridMillis())
                    .setShardGridMillis(TimeUnit.SECONDS.toMillis(shardGridSec))
                    .build();
            assertEquals(expectedOpts, result);
        }

        {
            var interval = intervalBefore(now.minus(Duration.ofDays(7)), Duration.ofMinutes(10));
            var result = customize(opts, interval, selectors);
            assertTrue(result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(result.getGridMillis())
                    .setShardGridMillis(TimeUnit.SECONDS.toMillis(shardGridSec))
                    .build();
            assertEquals(expectedOpts, result);
        }

        {
            var interval = intervalBefore(now.minus(Duration.ofDays(8)), Duration.ofMinutes(10));
            var result = customize(opts, interval, selectors);
            assertTrue(result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(policy.getItems()[0].getStepMillis())
                    .setShardGridMillis(policy.getItems()[0].getStepMillis())
                    .build();
            assertEquals(expectedOpts, result);
        }

        {
            var interval = intervalBefore(now.minus(Duration.ofDays(10)), Duration.ofMinutes(10));
            var result = customize(opts, interval, selectors);
            assertTrue(result.getGridMillis() > 0);
            var expectedOpts = opts.toBuilder()
                    .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                    .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                    .setGridMillis(policy.getItems()[0].getStepMillis())
                    .setShardGridMillis(policy.getItems()[0].getStepMillis())
                    .build();
            assertEquals(expectedOpts, result);
        }
    }

    private Interval interval(long value, ChronoUnit unit) {
        return Interval.before(Instant.now(), Duration.of(value, unit));
    }

    private Interval intervalBefore(Instant time, Duration duration) {
        return Interval.before(time, duration);
    }

    private DownsamplingOptions customize(DownsamplingOptions opts, Interval interval, List<Selectors> selectors) {
        return customizer.customizeDownsamplingOpts(opts, interval, selectors, interval.getBeginMillis());
    }

    private DownsamplingOptions customize(DownsamplingOptions opts, Interval interval, List<Selectors> selectors, long minBeginMillis) {
        return customizer.customizeDownsamplingOpts(opts, interval, selectors, minBeginMillis);
    }

    private Shard addShard(String project, String cluster, String service) {
        return addShard(project, cluster, service, DecimPolicy.UNDEFINED);
    }

    private Shard addShard(String project, String cluster, String service, DecimPolicy decimPolicy) {
        var shard = conf.shard(project, project + "_" + cluster + "_" + service)
                .toBuilder()
                .setClusterId(cluster)
                .setClusterName(cluster)
                .setServiceId(service)
                .setServiceName(service)
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        decimPolicy,
                        ShardSettings.AggregationSettings.EMPTY,
                        0))
                .build();
        conf.addShard(shard);
        configUpdated();
        return shard;
    }

    private void setGrid(Shard shard, int gridSec) {
        conf.addService(conf.service(shard.getProjectId(), shard.getServiceId()).toBuilder()
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                       null,
                        gridSec,
                        37,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.EMPTY,
                        3))
                .build());
        configUpdated();
    }

    private void enableGridAlight(Shard... shards) {
        for (var shard : shards) {
            featureFlagHolder.setFlag(shard.getNumId(), FeatureFlag.GRID_DOWNSAMPLING, true);
        }
    }

    private void configUpdated() {
        confHolder.onConfigurationLoad(conf.snapshot());
    }
}
