package ru.yandex.solomon.gateway.data;

import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import org.apache.commons.lang3.mutable.MutableInt;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.discovery.cluster.ClusterMapperStub;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.core.conf.SolomonConfStub;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.exceptions.CompilerException;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.expr.SelFunctions;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelFuncCategory;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueDouble;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.expression.value.SelValueString;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.DcMetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.metrics.client.SolomonClientStub;
import ru.yandex.solomon.metrics.client.combined.DataLimits;
import ru.yandex.solomon.metrics.client.exceptions.MemoryLimitForGraphIsReached;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.aggregation.DoubleSummary;
import ru.yandex.solomon.util.time.Interval;
import ru.yandex.stockpile.client.StockpileClient;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.model.point.AggrPoints.point;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class DataClientTest {
    private DataClient dataClient;
    private SolomonClientStub solomon;
    private MetricRegistry registry;
    private MetricsClientWrapper metricsClient;
    private SolomonConfHolder confHolder;
    private SolomonConfStub conf;
    private FeatureFlagHolderStub featureFlagsHolder;

    private static final int SERIES_COUNT = 300;
    private static final long ts1 = Instant.parse("2020-12-01T00:00:10Z").toEpochMilli();
    private static final long ts2 = Instant.parse("2020-12-01T00:00:20Z").toEpochMilli();
    private static final Interval someInterval = interval("2020-12-01T00:00:00Z", "2020-12-01T00:01:00Z");

    private static class MetricsClientWrapper extends DcMetricsClient {
        private final MutableInt lastReadSize = new MutableInt();

        int getLastReadSize() {
            return lastReadSize.intValue();
        }

        MetricsClientWrapper(String destination, MetabaseClient metabase, StockpileClient stockpile) {
            super(destination, metabase, stockpile);
        }

        @Override
        public CompletableFuture<ReadManyResponse> readMany(ReadManyRequest request) {
            return super.readMany(request)
                    .whenComplete((response, e) -> {
                        if (e != null) {
                            e.printStackTrace();
                        } else {
                            lastReadSize.setValue(response.getMetrics().size());
                        }
                    });
        }
    }

    @BeforeClass
    public static void classSetUp() {
        SelFunctions.REGISTRY.add(SelFunc.newBuilder()
                .name("fresh")
                .help("Fresh function, appeared in new version")
                .category(SelFuncCategory.INTERNAL)
                .supportedVersions(SelVersion.MAX::since)
                .returnType(SelTypes.DOUBLE)
                .handler(args -> new SelValueDouble(42))
                .build());

        SelFunctions.REGISTRY.add(SelFunc.newBuilder()
                .name("legacy")
                .help("Legacy function, dropped in new version")
                .category(SelFuncCategory.INTERNAL)
                .supportedVersions(SelVersion.MAX::before)
                .returnType(SelTypes.GRAPH_DATA)
                .handler(args -> new SelValueGraphData(GraphData.empty))
                .build());

        SelFunctions.REGISTRY.add(SelFunc.newBuilder()
                .name("__version")
                .help("Get expression language version as string")
                .category(SelFuncCategory.INTERNAL)
                .returnType(SelTypes.STRING)
                .handler((ctx, args) -> new SelValueString(ctx.getVersion().name()))
                .build());
    }

    @Before
    public void setUp() {
        solomon = new SolomonClientStub(false);
        metricsClient = new MetricsClientWrapper("test", solomon.getMetabase(), solomon.getStockpile());
        ClusterMapper mapper = new ClusterMapperStub();
        registry = new MetricRegistry();
        confHolder = new SolomonConfHolder();
        conf = new SolomonConfStub();
        featureFlagsHolder = new FeatureFlagHolderStub();
        DataClientMetrics dataClientMetrics = new DataClientMetrics();
        dataClient = new DataClient(dataClientMetrics, metricsClient, mapper, registry, confHolder, featureFlagsHolder);
    }

    @After
    public void tearDown() {
        solomon.close();
    }

    private DataResponse get(DataRequest request) {
        DataRequest newRequest = request.toBuilder()
                .setProducer(RequestProducer.SYSTEM)
                .setDeadline(Instant.now().plusSeconds(15))
                .build();

        return join(dataClient.readData(newRequest));
    }

    @Test
    public void topLoadOptimized() {
        Labels shard = addShard("");

        for (int i = 0; i < SERIES_COUNT; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)),
                    AggrGraphDataArrayList.listShort(ts1, 42 + i, ts2, 43 - i));
        }
        get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("top(5, 'max', {project=project, cluster=cluster, service=service})")
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .setInterval(Interval.millis(ts1, ts2))
                .build());
        assertEquals(5, metricsClient.getLastReadSize());
    }

    @Test
    public void bottomIgnoresEmptySeries() {
        Labels shard = addShard("");

        for (int i = 0; i < SERIES_COUNT; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)), (i % 2) == 0
                    ? AggrGraphDataArrayList.listShort(ts1, 42 + i, ts2, 43 - i)
                    : AggrGraphDataArrayList.empty());
        }
        var response = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("bottom(5, 'max', {project=project, cluster=cluster, service=service})")
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .setInterval(someInterval)
                .build());
        var result = response.getEvalResult().castToVector().valueArray();
        for (SelValue value : result) {
            NamedGraphData ngd = value.castToGraphData().getNamedGraphData();
            assertFalse(ngd.getAggrGraphDataArrayList().isEmpty());
        }
        assertEquals(5, metricsClient.getLastReadSize());
    }

    @Test
    public void topIgnoresEmptySeries() {
        Labels shard = addShard("");

        for (int i = 0; i < SERIES_COUNT; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)), (i % 2) == 0
                    ? AggrGraphDataArrayList.listShort(ts1, 42 + i, ts2, -43 - i)
                    : AggrGraphDataArrayList.empty());
        }
        var response = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("top(5, 'min', {project=project, cluster=cluster, service=service})")
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .setInterval(someInterval)
                .build());
        var result = response.getEvalResult().castToVector().valueArray();
        for (SelValue value : result) {
            NamedGraphData ngd = value.castToGraphData().getNamedGraphData();
            assertFalse(ngd.getAggrGraphDataArrayList().isEmpty());
        }
        assertEquals(5, metricsClient.getLastReadSize());
    }

    @Test
    public void topOptimizedCorrect() {
        Labels shard = addShard("");

        Random random = new Random(42);
        for (int i = 0; i < SERIES_COUNT; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)),
                    AggrGraphDataArrayList.listShort(ts1, random.nextDouble(), ts2, random.nextDouble()));
        }
        var optimized = Arrays.stream(get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("top(5, 'avg', {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build())
                .getEvalResult()
                .castToVector()
                .valueArray())
            .collect(Collectors.toSet());

        assertEquals(5, metricsClient.getLastReadSize());

        var notOmptimized = Arrays.stream(get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("top(5, 'avg', 0 + {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build())
                .getEvalResult()
                .castToVector()
                .valueArray())
            .collect(Collectors.toSet());
        assertEquals(SERIES_COUNT, metricsClient.getLastReadSize());

        assertEquals(optimized, notOmptimized);
    }

    @Test
    public void bottomOptimizedCorrect() {
        Labels shard = addShard("");

        Random random = new Random(42);
        for (int i = 0; i < SERIES_COUNT; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)),
                    AggrGraphDataArrayList.listShort(ts1, random.nextDouble(), ts2, random.nextDouble()));
        }
        var optimized = Arrays.stream(get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("bottom(10, 'sum', {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build())
                .getEvalResult()
                .castToVector()
                .valueArray())
            .collect(Collectors.toSet());
        assertEquals(10, metricsClient.getLastReadSize());

        var notOmptimized = Arrays.stream(get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("bottom(10, 'sum', 0 + {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build())
                .getEvalResult()
                .castToVector()
                .valueArray())
            .collect(Collectors.toSet());
        assertEquals(SERIES_COUNT, metricsClient.getLastReadSize());

        assertEquals(optimized, notOmptimized);
    }

    @Test
    public void topOptimizedCorrectAlias() {
        Labels shard = addShard("");

        Random random = new Random(42);
        for (int i = 0; i < SERIES_COUNT; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)),
                    AggrGraphDataArrayList.listShort(ts1, random.nextDouble(), ts2, random.nextDouble()));
        }
        var optimized = Arrays.stream(get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("top_avg(5, {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build())
                .getEvalResult()
                .castToVector()
                .valueArray())
                .collect(Collectors.toSet());

        assertEquals(5, metricsClient.getLastReadSize());

        var notOmptimized = Arrays.stream(get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("top_avg(5, 0 + {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build())
                .getEvalResult()
                .castToVector()
                .valueArray())
                .collect(Collectors.toSet());
        assertEquals(SERIES_COUNT, metricsClient.getLastReadSize());

        assertEquals(optimized, notOmptimized);
    }

    @Test
    public void bottomOptimizedCorrectAlias() {
        Labels shard = addShard("");

        Random random = new Random(42);
        for (int i = 0; i < SERIES_COUNT; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)),
                    AggrGraphDataArrayList.listShort(ts1, random.nextDouble(), ts2, random.nextDouble()));
        }
        var optimized = Arrays.stream(get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("bottom_sum(10, {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build())
                .getEvalResult()
                .castToVector()
                .valueArray())
                .collect(Collectors.toSet());
        assertEquals(10, metricsClient.getLastReadSize());

        var notOmptimized = Arrays.stream(get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("bottom_sum(10, 0 + {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build())
                .getEvalResult()
                .castToVector()
                .valueArray())
                .collect(Collectors.toSet());
        assertEquals(SERIES_COUNT, metricsClient.getLastReadSize());

        assertEquals(optimized, notOmptimized);
    }

    @Test
    public void downsamplingByPoints() {
        Labels shard = addShard("", 120, 60);

        AggrGraphDataArrayList source = AggrGraphDataArrayList.of(
                point("2020-02-17T00:01:00Z", 1),
                point("2020-02-17T00:02:00Z", 2),
                point("2020-02-17T00:03:00Z", 3),
                point("2020-02-17T00:04:00Z", 4),
                point("2020-02-17T00:05:00Z", 5),
                point("2020-02-17T00:06:00Z", 6)
        );

        solomon.addMetric(shard.add("sensor", "memoryUsage"), source);

        var result = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("{project=project, cluster=cluster, service=service, sensor=memoryUsage}")
                .setInterval(interval("2020-02-17T00:00:00Z", "2020-02-17T00:26:00Z"))
                .setDownsampling(DownsamplingOptions.newBuilder()
                        .setDownsamplingType(DownsamplingType.BY_POINTS)
                        .setDownsamplingAggr(Aggregation.LAST)
                        .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                        .setPoints(11)
                        .build())
                .build())
                .getEvalResult()
                .castToVector()
                .item(0)
                .castToGraphData()
                .getNamedGraphData()
                .getAggrGraphDataArrayList();

        var expected = AggrGraphDataArrayList.of(
                point("2020-02-17T00:00:00Z", 1),
                point("2020-02-17T00:02:00Z", 3),
                point("2020-02-17T00:04:00Z", 5),
                point("2020-02-17T00:06:00Z", 6)
        );

        Assert.assertNotEquals(source, AggrGraphDataArrayList.of(result).cloneWithMask(expected.columnSetMask()));
    }

    @Test
    public void overrideDownsamplingByProgram() {
        Labels shard = addShard("", 120, 60);

        AggrGraphDataArrayList source = AggrGraphDataArrayList.of(
                point("2020-02-17T00:01:00Z", 1),
                point("2020-02-17T00:02:00Z", 2),
                point("2020-02-17T00:03:00Z", 3),
                point("2020-02-17T00:04:00Z", 4),
                point("2020-02-17T00:05:00Z", 5),
                point("2020-02-17T00:06:00Z", 6)
        );

        solomon.addMetric(shard.add("sensor", "memoryUsage"), source);

        var result = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("last({project=project, cluster=cluster, service=service, sensor=memoryUsage}) by 2m")
                .setInterval(interval("2020-02-17T00:00:00Z", "2020-02-17T00:26:00Z"))
                .setDownsampling(DownsamplingOptions.newBuilder()
                        .setDownsamplingType(DownsamplingType.BY_INTERVAL)
                        .setDownsamplingAggr(Aggregation.SUM)
                        .setDownsamplingFill(OperationDownsampling.FillOption.NONE)
                        .setGridMillis(TimeUnit.MINUTES.toMillis(4L))
                        .build())
                .build())
                .getEvalResult()
                .castToVector()
                .item(0)
                .castToGraphData()
                .getNamedGraphData()
                .getAggrGraphDataArrayList();

        var expected = AggrGraphDataArrayList.of(
                point("2020-02-17T00:00:00Z", 1),
                point("2020-02-17T00:02:00Z", 3),
                point("2020-02-17T00:04:00Z", 5),
                point("2020-02-17T00:06:00Z", 6)
        );

        assertEquals(expected, AggrGraphDataArrayList.of(result).cloneWithMask(expected.columnSetMask()));
    }

    @Test(expected = CompilerException.class)
    public void multiVersionedRequestSameFail() throws Throwable {
        try {
            get(DataRequest.newBuilder()
                    .setProjectId("project")
                    .setProgram("junk")
                    .setInterval(someInterval)
                    .build());
        } catch (Exception e) {
            e.printStackTrace();

            assertRateIs(1L, "dataClient.versionedRequests.failed", SelVersion.MIN);
            assertRateIs(1L, "dataClient.versionedRequests.failed", SelVersion.MAX);
            assertRateIs(0L, "dataClient.versionedRequests.completed", SelVersion.MIN);
            assertRateIs(0L, "dataClient.versionedRequests.completed", SelVersion.MAX);
            assertRateIs(0L, "dataClient.versionedRequests.inconsistent", Labels.of());
            assertRateIs(1L, "dataClient.versionedRequests.consistent", Labels.of());

            throw CompletableFutures.unwrapCompletionException(e);
        }
    }

    @Test
    public void multiVersionedRequestDifferent() {
        DataResponse result = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("'Hello from expression language v' + __version()")
                .setInterval(someInterval)
                .build());

        assertEquals("Hello from expression language v" + SelVersion.MIN.name(),
                result.getEvalResult().castToString().getValue());

        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MIN);
        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MAX);
        assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MIN);
        assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MAX);
        assertRateIs(1L, "dataClient.versionedRequests.inconsistent", Labels.of());
        assertRateIs(0L, "dataClient.versionedRequests.consistent", Labels.of());
    }

    @Test(expected = EvaluationException.class)
    public void multiVersionedRequestCrashMin() throws Throwable {
        Labels shard = addShard("");

        for (int i = 0; i < 10; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)),
                    AggrGraphDataArrayList.listShort(ts2, 42 + i));
        }

        try {
            get(DataRequest.newBuilder()
                    .setProjectId("project")
                    .setProgram("group_lines((__version() == '" + SelVersion.MIN.name() + "') ? 'foo' : 'sum'" +
                            ", {project=project, cluster=cluster, service=service})")
                    .setInterval(someInterval)
                    .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                    .build());
        } catch (Throwable t) {
            assertRateIs(1L, "dataClient.versionedRequests.failed", SelVersion.MIN);
            assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MAX);
            assertRateIs(0L, "dataClient.versionedRequests.completed", SelVersion.MIN);
            assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MAX);
            assertRateIs(1L, "dataClient.versionedRequests.inconsistent", Labels.of());
            assertRateIs(0L, "dataClient.versionedRequests.consistent", Labels.of());

            throw CompletableFutures.unwrapCompletionException(t);
        }
    }

    @Test(expected = CompilerException.class)
    public void multiVersionedRequestNotCompilesMin() throws Throwable {
        try {
            get(DataRequest.newBuilder()
                    .setProjectId("project")
                    .setProgram("fresh()")
                    .setInterval(someInterval)
                    .build());
        } catch (Throwable t) {
            assertRateIs(1L, "dataClient.versionedRequests.failed", SelVersion.MIN);
            assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MAX);
            assertRateIs(0L, "dataClient.versionedRequests.completed", SelVersion.MIN);
            assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MAX);
            assertRateIs(1L, "dataClient.versionedRequests.inconsistent", Labels.of());
            assertRateIs(0L, "dataClient.versionedRequests.consistent", Labels.of());

            throw CompletableFutures.unwrapCompletionException(t);
        }
    }

    @Test
    public void multiVersionedRequestCrashMax() {
        Labels shard = addShard("");

        for (int i = 0; i < 10; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)),
                    AggrGraphDataArrayList.listShort(ts2, 42 + i));
        }

        DataResponse response = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .setProgram("group_lines((__version() == '" + SelVersion.MIN.name() + "') ? 'max' : 'xam'" +
                        ", {project=project, cluster=cluster, service=service})")
                .setInterval(someInterval)
                .build());

        assertEquals(GraphData.graphData(ts2, 51),
                response.getEvalResult().castToGraphData().getGraphData());

        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MIN);
        assertRateIs(1L, "dataClient.versionedRequests.failed", SelVersion.MAX);
        assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MIN);
        assertRateIs(0L, "dataClient.versionedRequests.completed", SelVersion.MAX);
        assertRateIs(1L, "dataClient.versionedRequests.inconsistent", Labels.of());
        assertRateIs(0L, "dataClient.versionedRequests.consistent", Labels.of());
    }

    @Test
    public void multiVersionedRequestNotCompilesMax() {
        DataResponse response = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("legacy()")
                .setInterval(someInterval)
                .build());

        assertEquals(GraphData.empty, response.getEvalResult().castToGraphData().getGraphData());

        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MIN);
        assertRateIs(1L, "dataClient.versionedRequests.failed", SelVersion.MAX);
        assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MIN);
        assertRateIs(0L, "dataClient.versionedRequests.completed", SelVersion.MAX);
        assertRateIs(1L, "dataClient.versionedRequests.inconsistent", Labels.of());
        assertRateIs(0L, "dataClient.versionedRequests.consistent", Labels.of());
    }

    @Test
    public void requestWithExplicitVersion() {
        featureFlagsHolder.setFlag("project", FeatureFlag.EXPRESSION_LAST_VERSION, true);

        DataResponse response = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("legacy()")
                .setVersion(SelVersion.MIN)
                .setInterval(someInterval)
                .build());

        assertEquals(GraphData.empty, response.getEvalResult().castToGraphData().getGraphData());

        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MIN);
        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MAX);
        assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MIN);
        assertRateIs(0L, "dataClient.versionedRequests.completed", SelVersion.MAX);
        assertRateIs(0L, "dataClient.versionedRequests.inconsistent", Labels.of());
        assertRateIs(0L, "dataClient.versionedRequests.consistent", Labels.of());
    }

    @Test
    public void requestWithFlag() {
        featureFlagsHolder.setFlag("project", FeatureFlag.EXPRESSION_LAST_VERSION, true);

        get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("fresh()")
                .setInterval(someInterval)
                .build());

        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MIN);
        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MAX);
        assertRateIs(0L, "dataClient.versionedRequests.completed", SelVersion.MIN);
        assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MAX);
        assertRateIs(0L, "dataClient.versionedRequests.inconsistent", Labels.of());
        assertRateIs(0L, "dataClient.versionedRequests.consistent", Labels.of());
    }

    @Test
    public void multiVersionedRequestSame() {
        Labels shard = addShard("");

        for (int i = 0; i < 10; i++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(i)),
                    AggrGraphDataArrayList.listShort(ts1, 42 + i, ts2, 43 - i));
        }
        get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("{project=project, cluster=cluster, service=service}")
                .setInterval(someInterval)
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build());

        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MIN);
        assertRateIs(0L, "dataClient.versionedRequests.failed", SelVersion.MAX);
        assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MIN);
        assertRateIs(1L, "dataClient.versionedRequests.completed", SelVersion.MAX);
        assertRateIs(0L, "dataClient.versionedRequests.inconsistent", Labels.of());
        assertRateIs(1L, "dataClient.versionedRequests.consistent", Labels.of());
    }

    @Test
    public void shiftLoadsFromPast() {
        Labels shard = addShard("");

        Instant ts0 = Instant.parse("2020-12-01T00:00:00Z");

        solomon.addMetric(shard.add("sensor", "old"),
                AggrGraphDataArrayList.listShort(
                        ts0.plusSeconds(   1).toEpochMilli(), 42,
                        ts0.plusSeconds(3601).toEpochMilli(), 43));

        DataResponse response = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("[" +
                        "{project=project, cluster=cluster, service=service}, " +
                        "shift({project=project, cluster=cluster, service=service}, 1h)]")
                .setInterval(new Interval(ts0.plusSeconds(3600), ts0.plusSeconds(3602)))
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.OFF).build())
                .build());

        List<GraphData> result = Arrays.stream(response.getEvalResult().castToVector().valueArray())
                .map(SelValue::castToGraphData)
                .map(SelValueGraphData::getGraphData)
                .collect(Collectors.toList());

        assertEquals(GraphData.graphData(ts0.plusSeconds(3601).toEpochMilli(), 43), result.get(0));
        assertEquals(GraphData.graphData(ts0.plusSeconds(3601).toEpochMilli(), 42), result.get(1));
    }

    @Test
    public void overrideAutoDownsamplingByProgram() {
        Labels shard = addShard("", 120, 60);

        AggrGraphDataArrayList source = AggrGraphDataArrayList.of(
                point("2020-02-17T00:01:00Z", 1),
                point("2020-02-17T00:02:00Z", 2),
                point("2020-02-17T00:03:00Z", 3),
                point("2020-02-17T00:04:00Z", 4),
                point("2020-02-17T00:05:00Z", 5),
                point("2020-02-17T00:06:00Z", 6)
        );

        solomon.addMetric(shard.add("sensor", "memoryUsage"), source);

        var result = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("last({project=project, cluster=cluster, service=service, sensor=memoryUsage}) by 2m")
                .setInterval(interval("2020-02-17T00:00:00Z", "2020-02-17T00:26:00Z"))
                .setDownsampling(DownsamplingOptions.newBuilder()
                        .setDownsamplingType(DownsamplingType.BY_POINTS)
                        .setPoints(500)
                        .setDownsamplingAggr(Aggregation.SUM)
                        .build())
                .build())
                .getEvalResult()
                .castToVector()
                .item(0)
                .castToGraphData()
                .getNamedGraphData()
                .getAggrGraphDataArrayList();

        var expected = AggrGraphDataArrayList.of(
                point("2020-02-17T00:00:00Z", 1),
                point("2020-02-17T00:02:00Z", 3),
                point("2020-02-17T00:04:00Z", 5),
                point("2020-02-17T00:06:00Z", 6)
        );

        assertEquals(expected, AggrGraphDataArrayList.of(result).cloneWithMask(expected.columnSetMask()));
    }

    @Test
    public void responseBytesLimit() {
        Labels shard = addShard("", 120, 1);

        for (int index = 0; index < 5_000; index++) {
            solomon.addMetric(shard.add("metric", Integer.toString(index)), AggrGraphDataArrayList.empty());
        }

        try {
            get(DataRequest.newBuilder()
                    .setProjectId("project")
                    .setProgram("last({project=project, cluster=cluster, service=service, metric='*'}) by 25m")
                    .setInterval(interval("2020-01-01T00:00:00Z", "2020-06-01T00:00:00Z"))
                    .build());
            fail("Request should fail with memory limit");
        } catch (Throwable e) {
            var cause = Throwables.getRootCause(e);
            assertThat(cause, Matchers.instanceOf(MemoryLimitForGraphIsReached.class));
        }
    }

    @Test
    public void metricsLimitInOldMode() {
        Labels shard = addShard("", 120, 60);

        for (int index = 0; index < 12_000; index++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(index)), AggrGraphDataArrayList.empty());
        }

        var result = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("{cluster=cluster, service=service, sensor='*'}")
                .setInterval(interval("2020-02-17T00:00:00Z", "2020-02-17T00:26:00Z"))
                .setOldMode(true)
                .build());

        var vector = result.getEvalResult().castToVector().valueArray();
        var oldModeResult = result.getOldModeResult();
        assertEquals(DataLimits.MAX_METRICS_FOR_AGGR_COUNT, vector.length);
        assertTrue(oldModeResult.isTruncated());
        assertTrue(oldModeResult.isSummary());
    }

    @Test
    public void topForTooManyMetricsInOldMode() {
        Labels shard = addShard("", 120, 60);

        for (int index = 0; index < 22_000; index++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(index)), AggrGraphDataArrayList.empty());
        }

        var result = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("top(50, 'avg', {cluster=cluster, service=service, sensor='*'})")
                .setInterval(interval("2020-02-17T00:00:00Z", "2020-02-17T00:26:00Z"))
                .setOldMode(true)
                .build());

        var vector = result.getEvalResult().castToVector().valueArray();
        var oldModeResult = result.getOldModeResult();
        assertEquals(50, vector.length);
        assertTrue(oldModeResult.isTruncated());
        assertFalse(oldModeResult.isSummary());
    }

    @Test
    public void onlySummary() {
        Labels shard = addShard("", 120, 60);

        for (int index = 0; index < 10; index++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(index)), AggrGraphDataArrayList.listShort(ts1, Math.cos(index)));
        }

        var result = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("{cluster=cluster, service=service, sensor='*'}")
                .setInterval(someInterval)
                .setSummaryOnly(true)
                .build());

        var vector = result.getEvalResult().castToVector().valueArray();
        var oldModeResult = result.getOldModeResult();
        assertEquals(10, vector.length);
        for (var v : vector) {
            // XXX this should indicate //absent// timeseries, not be just an empty list
            assertTrue(v.castToGraphData().getNamedGraphData().getAggrGraphDataArrayList().isEmpty());
            assertTrue(v.castToGraphData().getNamedGraphData().getSummary().has(Aggregation.AVG));
        }
        assertFalse(oldModeResult.isTruncated());
        assertTrue(oldModeResult.isSummary());
    }

    @Test
    public void onlySummaryTop() {
        Labels shard = addShard("", 120, 60);

        for (int index = 0; index < 1_000; index++) {
            solomon.addMetric(shard.add("sensor", Integer.toString(index)),
                    AggrGraphDataArrayList.listShort(ts1, Math.sin(index), ts2, Math.cos(index)));
        }

        var result = get(DataRequest.newBuilder()
                .setProjectId("project")
                .setProgram("top_avg(5, {cluster=cluster, service=service, sensor='*'})")
                .setInterval(someInterval)
                .setSummaryOnly(true)
                .build());

        var vector = result.getEvalResult().castToVector().valueArray();
        for (var v : vector) {
            // XXX this should indicate //absent// timeseries, not be just an empty list
            NamedGraphData ngd = v.castToGraphData().getNamedGraphData();
            assertTrue(ngd.getAggrGraphDataArrayList().isEmpty());
            assertTrue(ngd.getSummary().has(Aggregation.AVG));
            var summary = ngd.getSummary();
            assertTrue(summary instanceof DoubleSummary);
            assertTrue(((DoubleSummary) summary).getAvg() > 0.7);
        }
        var oldModeResult = result.getOldModeResult();
        assertEquals(5, vector.length);
        assertFalse(oldModeResult.isTruncated());
        assertTrue(oldModeResult.isSummary());
    }

    private static Interval interval(String from, String to) {
        return Interval.millis(timeToMillis(from), timeToMillis(to));
    }

    private static long timeToMillis(String time) {
        return Instant.parse(time).toEpochMilli();
    }

    private Labels addShard(String id) {
        return addShard(id, 15, 0);
    }

    private Labels addShard(String id, int fetchIntervalSec, int gridSec) {
        ShardKey key = new ShardKey("project" + id, "cluster" + id, "service" + id);
        addShard(key, fetchIntervalSec, gridSec);
        return key.toLabels();
    }

    private void addShard(ShardKey key, int fetchIntervalSec, int gridSec) {
        String shardId = key.getProject() + "_" + key.getCluster() + "_" + key.getService();
        conf.addService(conf.service(key.getProject(), key.getService())
                .toBuilder()
                .setName(key.getService())
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                        null,
                        gridSec,
                        37,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new ServiceMetricConf.AggrRule[0], false),
                        fetchIntervalSec))
                .build());
        conf.addCluster(conf.cluster(key.getProject(), key.getCluster())
                .toBuilder()
                .setName(key.getCluster())
                .build());
        conf.addShard(conf.shard(key.getProject(), shardId)
                .toBuilder()
                .setClusterId(key.getCluster())
                .setClusterName(key.getCluster())
                .setServiceId(key.getService())
                .setServiceName(key.getService())
                .build());
        confHolder.onConfigurationLoad(conf.snapshot());
    }

    private void assertRateIs(long expected, String name, Labels labels) {
        Rate metric = (Rate) registry.getMetric(new MetricId(name, labels));
        assertEquals(metric.toString(), expected, metric.get());
    }

    private void assertRateIs(long expected, String name, SelVersion version) {
        assertRateIs(expected, name, Labels.of("version", version.name()));
    }
}
