package ru.yandex.solomon.gateway.data;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.mutable.MutableInt;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.iterator.AggrPointCursor;
import ru.yandex.solomon.model.timeseries.iterator.GenericCombineIterator;
import ru.yandex.solomon.model.timeseries.iterator.GenericPointCollector;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class RandomSeriesGeneratorTest {

    private RandomSeriesGenerator validate(String key, long from, long to, long grid) {
        var iterator = new RandomSeriesGenerator(key, from, to, grid);

        AggrPoint point = new AggrPoint();

        int count = 0;
        for (long ts = from; ts < to; ts += grid) {
            iterator.next(point);
            Assert.assertEquals(ts, point.getTsMillis());
            count++;
        }
        Assert.assertEquals(count, iterator.estimatePointsCount());

        Assert.assertFalse(iterator.next(point));

        return new RandomSeriesGenerator(key, from, to, grid);
    }

    @Test
    public void isStableRound() {
        AggrGraphDataArrayList list1 = AggrGraphDataArrayList.of(validate("foo", 1000, 2000, 50));
        AggrGraphDataArrayList list2 = AggrGraphDataArrayList.of(validate("foo", 1000, 2000, 50));

        Assert.assertEquals(list1, list2);
    }

    @Test
    public void isStable() {
        AggrGraphDataArrayList list1 = AggrGraphDataArrayList.of(validate("foo", 1000, 2000, 30));
        AggrGraphDataArrayList list2 = AggrGraphDataArrayList.of(validate("foo", 1000, 2000, 30));

        Assert.assertEquals(list1, list2);
    }

    @Test
    public void differentKeys() {
        AggrGraphDataArrayList list1 = AggrGraphDataArrayList.of(validate("foo", 1000, 2000, 30));
        AggrGraphDataArrayList list2 = AggrGraphDataArrayList.of(validate("bar", 1000, 2000, 30));

        Assert.assertNotEquals(list1, list2);
    }

    @Test
    public void shiftingStable() {
        var iter1 = validate("foo", 1000, 3000, 30);
        var iter2 = validate("foo", 1900, 5000, 30);

        var cursors = List.of(iter1, iter2).stream().map(AggrPointCursor::new).collect(Collectors.toList());
        MutableInt countTested = new MutableInt();
        var combined = new GenericCombineIterator<>(cursors, new GenericPointCollector<AggrPointCursor, Void>() {
            private final AggrPoint[] points = new AggrPoint[2];

            @Override
            public void reset() {
                points[0] = points[1] = null;
            }

            @Override
            public void append(int cursorIndex, AggrPointCursor cursor) {
                points[cursorIndex] = cursor.getPoint();
            }

            @Override
            public void compute(long timestamp, Void target) {
                if (points[0] != null && points[1] != null) {
                    Assert.assertEquals(points[0], points[1]);
                    countTested.increment();
                }
            }
        });

        while (combined.next(null)) {
            // nop
        }

        Assert.assertTrue(countTested.intValue() > 0);
    }

    @Test
    public void downsamplingStable() {
        var iter1 = validate("foo", 1000, 3000, 50);
        var iter2 = validate("foo", 1000, 3000, 100);

        var cursors = List.of(iter1, iter2).stream().map(AggrPointCursor::new).collect(Collectors.toList());
        MutableInt countTested = new MutableInt();
        var combined = new GenericCombineIterator<>(cursors, new GenericPointCollector<AggrPointCursor, Void>() {
            private final AggrPoint[] points = new AggrPoint[2];

            @Override
            public void reset() {
                points[0] = points[1] = null;
            }

            @Override
            public void append(int cursorIndex, AggrPointCursor cursor) {
                points[cursorIndex] = cursor.getPoint();
            }

            @Override
            public void compute(long timestamp, Void target) {
                if (points[0] != null && points[1] != null) {
                    Assert.assertEquals(points[0], points[1]);
                    countTested.increment();
                }
            }
        });

        while (combined.next(null)) {
            // nop
        }

        Assert.assertTrue(countTested.intValue() > 0);
    }
}
