package ru.yandex.solomon.gateway.data.exceptions;

import java.io.IOException;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.Assert;
import org.junit.Test;
import org.springframework.mock.http.server.reactive.MockServerHttpRequest;
import org.springframework.mock.web.server.MockServerWebExchange;

import ru.yandex.solomon.exception.handlers.HttpApiExceptionHandler;
import ru.yandex.solomon.metrics.client.exceptions.DataClientException;
import ru.yandex.solomon.metrics.client.exceptions.MemoryLimitForGraphIsReached;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class DataClientExceptionRegistryTest {
    @Test
    public void population() {
        ObjectMapper mapper = new ObjectMapper();
        DataClientExceptionRegistry.PER_TYPE_SCHEMA.forEach((key, value) -> {
            try {
                System.out.println("===== " + key + " =====");
                System.out.println(mapper.writerWithDefaultPrettyPrinter().writeValueAsString(value));
                System.out.println();
            } catch (JsonProcessingException ex) {
                throw new RuntimeException(ex);
            }
        });
        Assert.assertTrue(DataClientExceptionRegistry.PER_TYPE_SCHEMA.size() > 0);

        Assert.assertTrue(DataClientExceptionRegistry.PER_TYPE_SCHEMA.containsKey("MEMORY_LIMIT_FOR_GRAPH_IS_REACHED"));
        Assert.assertTrue(DataClientExceptionRegistry.PER_TYPE_SCHEMA.containsKey("COMPILER_EXCEPTION"));
    }

    @Test
    public void schemaIsCorrect() {
        try {
            throw new MemoryLimitForGraphIsReached(10, 20, 30, 40, 50);
        } catch (DataClientException e) {
            final String type = "MEMORY_LIMIT_FOR_GRAPH_IS_REACHED";
            Assert.assertEquals(type, e.getType());
            Map<String, Object> details = e.getDetails();
            var props = DataClientExceptionRegistry.PER_TYPE_SCHEMA.get(type).get("properties");
            assertFieldIsDeclaredAndHasValue(props, details, "metric_count", 10);
            assertFieldIsDeclaredAndHasValue(props, details, "interval_millis", 20L);
            assertFieldIsDeclaredAndHasValue(props, details, "grid_millis", 30L);
            assertFieldIsDeclaredAndHasValue(props, details, "response_size_bytes", 40L);
            assertFieldIsDeclaredAndHasValue(props, details, "limit_bytes", 50L);
        }
    }

    @Test
    public void apiExceptionResolved() throws IOException {
        try {
            throw new MemoryLimitForGraphIsReached(10, 20, 30, 40, 50);
        } catch (DataClientException e) {
            ObjectMapper mapper = new ObjectMapper();
            var exHandler = new HttpApiExceptionHandler(mapper);
            var exch = MockServerWebExchange.builder(
                    MockServerHttpRequest.post("foo").body("bar"))
                    .build();
            exHandler.handle(exch, e).block();
            var json = mapper.readTree(exch.getResponse().getBodyAsString().block());
            Assert.assertEquals("MEMORY_LIMIT_FOR_GRAPH_IS_REACHED", json.get("type").textValue());
            Assert.assertEquals(400, json.get("code").intValue());
            Assert.assertEquals(e.getMessage(), json.get("message").textValue());
            var details = json.get("details");
            Assert.assertEquals(10, details.get("metric_count").intValue());
            Assert.assertEquals(50, details.get("limit_bytes").intValue());
        }
    }

    private static void assertFieldIsDeclaredAndHasValue(JsonNode properties, Map<String, Object> details, String field, Object expected) {
        Assert.assertTrue(properties.has(field));
        Assert.assertEquals(expected, details.get(field));
    }
}
