package ru.yandex.solomon.gateway.entityConverter;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Test;

import static org.junit.Assert.assertEquals;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DashSchemeTest {

    @Test
    public void reduceSizes() {
        DashScheme<String> dashScheme = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 10, 10, "1"),
                new DashScheme.WidgetScheme<>(10, 0, 10, 10, "2"),
                new DashScheme.WidgetScheme<>(0, 10, 10, 10, "3"),
                new DashScheme.WidgetScheme<>(10, 10, 10, 10, "4"),
                new DashScheme.WidgetScheme<>(0, 25, 5, 10, "5")
        ));

        DashScheme<String> actual = dashScheme.reduceSizes();

        DashScheme<String> expected = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 2, 2, "1"),
                new DashScheme.WidgetScheme<>(2, 0, 2, 2, "2"),
                new DashScheme.WidgetScheme<>(0, 2, 2, 2, "3"),
                new DashScheme.WidgetScheme<>(2, 2, 2, 2, "4"),
                new DashScheme.WidgetScheme<>(0, 5, 1, 2, "5")
        ));

        assertEquals(expected, actual);
    }

    @Test
    public void scaleByHeight() {
        DashScheme<String> dashScheme = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 1, 1, "1"),
                new DashScheme.WidgetScheme<>(1, 0, 1, 1, "2"),
                new DashScheme.WidgetScheme<>(0, 1, 2, 2, "3"),
                new DashScheme.WidgetScheme<>(0, 3, 1, 1, "4")
        ));

        DashScheme<String> actual = dashScheme.scaleByHeight(8);

        DashScheme<String> expected = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 1, 8, "1"),
                new DashScheme.WidgetScheme<>(1, 0, 1, 8, "2"),
                new DashScheme.WidgetScheme<>(0, 8, 2, 16, "3"),
                new DashScheme.WidgetScheme<>(0, 24, 1, 8, "4")
        ));

        assertEquals(expected, actual);
    }

    @Test
    public void scaleWidthFrom2To36() {
        DashScheme<String> dashScheme = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 1, 1, "1"),
                new DashScheme.WidgetScheme<>(1, 0, 1, 1, "2"),
                new DashScheme.WidgetScheme<>(0, 1, 1, 1, "3"),
                new DashScheme.WidgetScheme<>(1, 1, 1, 1, "4")
        ));

        DashScheme<String> actual = dashScheme.scaleByFixedWidth(36);

        DashScheme<String> expected = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 18, 1, "1"),
                new DashScheme.WidgetScheme<>(18, 0, 18, 1, "2"),
                new DashScheme.WidgetScheme<>(0, 1, 18, 1, "3"),
                new DashScheme.WidgetScheme<>(18, 1, 18, 1, "4")
        ));

        assertEquals(expected, actual);
    }

    @Test
    public void scaleWidthFrom5To36() {
        DashScheme<String> dashScheme = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 3, 1, "1"),
                new DashScheme.WidgetScheme<>(3, 0, 2, 1, "2"),
                new DashScheme.WidgetScheme<>(0, 1, 2, 1, "3"),
                new DashScheme.WidgetScheme<>(2, 1, 3, 1, "4")
        ));

        DashScheme<String> actual = dashScheme.scaleByFixedWidth(36);

        DashScheme<String> expected = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 22, 1, "1"),
                new DashScheme.WidgetScheme<>(22, 0, 14, 1, "2"),
                new DashScheme.WidgetScheme<>(0, 1, 15, 1, "3"),
                new DashScheme.WidgetScheme<>(15, 1, 21, 1, "4")
        ));

        assertEquals(expected, actual);
    }

    @Test
    public void scaleWidthDisproportionately() {
        DashScheme<String> dashScheme = new DashScheme<>(List.of(
                // blocks in row: 20.
                new DashScheme.WidgetScheme<>(0, 0, 20, 1, "1.1"),
                // blocks in row: 19, 1.
                new DashScheme.WidgetScheme<>(0, 1, 19, 1, "2.1"),
                new DashScheme.WidgetScheme<>(19, 1, 1, 1, "2.1"),
                // blocks in row: 1, 19.
                new DashScheme.WidgetScheme<>(0, 2, 1, 1, "3.1"),
                new DashScheme.WidgetScheme<>(1, 2, 19, 1, "3.1"),
                // blocks in row: 17, 3.
                new DashScheme.WidgetScheme<>(0, 3, 17, 1, "4.1"),
                new DashScheme.WidgetScheme<>(17, 3, 3, 1, "4.2"),
                // blocks in row: 3, 17.
                new DashScheme.WidgetScheme<>(0, 4, 3, 1, "5.1"),
                new DashScheme.WidgetScheme<>(3, 4, 17, 1, "5.2"),
                // blocks in row: 5, 15.
                new DashScheme.WidgetScheme<>(0, 5, 5, 1, "6.1"),
                new DashScheme.WidgetScheme<>(5, 5, 15, 1, "6.2"),
                // blocks in row: 15, 5.
                new DashScheme.WidgetScheme<>(0, 6, 15, 1, "7.1"),
                new DashScheme.WidgetScheme<>(15, 6, 5, 1, "7.2"),
                // blocks in row: 10, 10.
                new DashScheme.WidgetScheme<>(0, 7, 10, 1, "8.1"),
                new DashScheme.WidgetScheme<>(10, 7, 10, 1, "8.2"),
                // blocks in row: 7, 7, 6.
                new DashScheme.WidgetScheme<>(0, 8, 7, 1, "9.1"),
                new DashScheme.WidgetScheme<>(7, 8, 7, 1, "9.2"),
                new DashScheme.WidgetScheme<>(14, 8, 6, 1, "9.2")
        ));

        DashScheme<String> actual = dashScheme.scaleByFixedWidth(36);

        DashScheme<String> expected = new DashScheme<>(List.of(
                // blocks in row: 20 -> 36.
                new DashScheme.WidgetScheme<>(0, 0, 36, 1, "1.1"),
                // blocks in row: 19, 1 -> 35, 1.
                new DashScheme.WidgetScheme<>(0, 1, 35, 1, "2.1"),
                new DashScheme.WidgetScheme<>(35, 1, 1, 1, "2.1"),
                // blocks in row: 1, 19 -> 2, 34.
                new DashScheme.WidgetScheme<>(0, 2, 2, 1, "3.1"),
                new DashScheme.WidgetScheme<>(2, 2, 34, 1, "3.1"),
                // blocks in row: 17, 3 -> 33, 3.
                new DashScheme.WidgetScheme<>(0, 3, 33, 1, "4.1"),
                new DashScheme.WidgetScheme<>(33, 3, 3, 1, "4.2"),
                // blocks in row: 3, 17 -> 6, 30.
                new DashScheme.WidgetScheme<>(0, 4, 6, 1, "5.1"),
                new DashScheme.WidgetScheme<>(6, 4, 30, 1, "5.2"),
                // blocks in row: 5, 15 -> 10, 36.
                new DashScheme.WidgetScheme<>(0, 5, 10, 1, "6.1"),
                new DashScheme.WidgetScheme<>(10, 5, 26, 1, "6.2"),
                // blocks in row: 15, 5 -> 30, 6.
                new DashScheme.WidgetScheme<>(0, 6, 30, 1, "7.1"),
                new DashScheme.WidgetScheme<>(30, 6, 6, 1, "7.2"),
                // blocks in row: 10, 10 -> 20, 16.
                new DashScheme.WidgetScheme<>(0, 7, 20, 1, "8.1"),
                new DashScheme.WidgetScheme<>(20, 7, 16, 1, "8.2"),
                // blocks in row: 7, 7, 6 -> 14, 14, 8.
                new DashScheme.WidgetScheme<>(0, 8, 14, 1, "9.1"),
                new DashScheme.WidgetScheme<>(14, 8, 14, 1, "9.2"),
                new DashScheme.WidgetScheme<>(28, 8, 8, 1, "9.2")
        ));

        assertEquals(expected, actual);
    }

    @Test
    public void insertDash() {
        DashScheme<String> dashScheme = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 3, 1, "1"),
                new DashScheme.WidgetScheme<>(3, 0, 2, 1, "2"),
                new DashScheme.WidgetScheme<>(0, 1, 2, 1, "3"),
                new DashScheme.WidgetScheme<>(2, 1, 3, 1, "4")
        ));

        DashScheme<String> insertDashScheme = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 1, 1, "sub 1"),
                new DashScheme.WidgetScheme<>(1, 0, 1, 1, "sub 2"),
                new DashScheme.WidgetScheme<>(0, 1, 3, 1, "sub 3"),
                new DashScheme.WidgetScheme<>(0, 2, 2, 2, "sub 4")
        ));

        DashScheme<String> actual = dashScheme.replace(dashScheme.getWidgets().get(1), insertDashScheme);

        DashScheme<String> expected = new DashScheme<>(List.of(
                new DashScheme.WidgetScheme<>(0, 0, 9, 4, "1"),
                new DashScheme.WidgetScheme<>(9, 0, 2, 1, "sub 1"),
                new DashScheme.WidgetScheme<>(11, 0, 2, 1, "sub 2"),
                new DashScheme.WidgetScheme<>(9, 1, 6, 1, "sub 3"),
                new DashScheme.WidgetScheme<>(9, 2, 4, 2, "sub 4"),
                new DashScheme.WidgetScheme<>(0, 4, 6, 4, "3"),
                new DashScheme.WidgetScheme<>(6, 4, 9, 4, "4")
        ));

        assertEquals(expected, actual);
    }
}
