package ru.yandex.solomon.gateway.entityConverter;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Test;

import static org.junit.Assert.assertEquals;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class SelectorsReplacerTest {

    @Test
    public void empty() {
        var actual = SelectorsReplacer.replaceSelectorsInCode("", Map.of());
        var expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void blank() {
        var actual = SelectorsReplacer.replaceSelectorsInCode("  ", Map.of());
        var expected = "  ";
        assertEquals(expected, actual);
    }

    @Test
    public void text() {
        var actual = SelectorsReplacer.divideCodeBySelectors("2 + 2");
        var expected = List.of(SelectorsReplacer.Part.text("2 + 2"));
        assertEquals(expected, actual);
    }

    @Test
    public void singleSelectors() {
        var actual = SelectorsReplacer.divideCodeBySelectors("{a=b, c=d, e='f', \"g\"=\"h\"}");
        var expected = List.of(SelectorsReplacer.Part.selectors("{a=b, c=d, e='f', \"g\"=\"h\"}"));
        assertEquals(expected, actual);
    }

    @Test
    public void selectorsAtFirst() {
        var actual = SelectorsReplacer.divideCodeBySelectors("{sensor=\"value\"} + 1");
        var expected = List.of(
                SelectorsReplacer.Part.selectors("{sensor=\"value\"}"),
                SelectorsReplacer.Part.text(" + 1")
        );
        assertEquals(expected, actual);
    }

    @Test
    public void selectorsAtLast() {
        var actual = SelectorsReplacer.divideCodeBySelectors("1 + {sensor=\"value\"}");
        var expected = List.of(
                SelectorsReplacer.Part.text("1 + "),
                SelectorsReplacer.Part.selectors("{sensor=\"value\"}")
        );
        assertEquals(expected, actual);
    }

    @Test
    public void selectorsAtMiddle() {
        var actual = SelectorsReplacer.divideCodeBySelectors("1 + {sensor=\"value\"};");
        var expected = List.of(
                SelectorsReplacer.Part.text("1 + "),
                SelectorsReplacer.Part.selectors("{sensor=\"value\"}"),
                SelectorsReplacer.Part.text(";")
        );
        assertEquals(expected, actual);
    }

    @Test
    public void selectorsWithName() {
        var actual = SelectorsReplacer.divideCodeBySelectors("part.subpart.other_subpart {sensor=\"value\"}");
        var expected = List.of(SelectorsReplacer.Part.selectors("part.subpart.other_subpart {sensor=\"value\"}"));
        assertEquals(expected, actual);
    }

    @Test
    public void selectorsWithReturn() {
        var actual = SelectorsReplacer.divideCodeBySelectors("return {sensor=\"value\"};");
        var expected = List.of(
                SelectorsReplacer.Part.text("return "),
                SelectorsReplacer.Part.selectors("{sensor=\"value\"}"),
                SelectorsReplacer.Part.text(";")
        );
        assertEquals(expected, actual);
    }

    @Test
    public void oldLoadSelectors() {
        var actual = SelectorsReplacer.divideCodeBySelectors("load(\"a=b&c=d\")");
        var expected = List.of(SelectorsReplacer.Part.load("load(\"a=b&c=d\")", "a=b&c=d"));
        assertEquals(expected, actual);
    }

    @Test
    public void oldLoad1Selectors() {
        var actual = SelectorsReplacer.divideCodeBySelectors("load1(\"a=b&c=d\")");
        var expected = List.of(SelectorsReplacer.Part.load1("load1(\"a=b&c=d\")", "a=b&c=d"));
        assertEquals(expected, actual);
    }

    @Test
    public void oldAndNewSelectorsTogether() {
        var code = "load1(\"a=b&c=d\") + {a=b, c=d}";
        var actual = SelectorsReplacer.divideCodeBySelectors(code);
        var expected = List.of(
                SelectorsReplacer.Part.load1("load1(\"a=b&c=d\")", "a=b&c=d"),
                SelectorsReplacer.Part.text(" + "),
                SelectorsReplacer.Part.selectors("{a=b, c=d}")
        );
        assertEquals(expected, actual);
    }

    @Test
    public void complex() {
        var code = "alias({sensor='jvm.classes.loaded|jvm.classes.unloaded',host!='cluster',host!='per_dc'},'{{host}}.{{sensor}}')";
        var actual = SelectorsReplacer.divideCodeBySelectors(code);
        var expected = List.of(
                SelectorsReplacer.Part.text("alias("),
                SelectorsReplacer.Part.selectors("{sensor='jvm.classes.loaded|jvm.classes.unloaded',host!='cluster',host!='per_dc'}"),
                SelectorsReplacer.Part.text(",'{{host}}.{{sensor}}')")
        );
        assertEquals(expected, actual);
    }
}
