package ru.yandex.solomon.gateway.operations.db;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Consumer;
import java.util.stream.Stream;

import com.google.common.collect.Streams;
import com.google.protobuf.Any;
import com.google.protobuf.StringValue;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.LongRunningOperationType;
import ru.yandex.solomon.util.async.InFlightLimiter;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static java.util.Collections.shuffle;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Stanislav Kashirin
 */
public abstract class LongRunningOperationDaoTest {

    private static final Comparator<LongRunningOperation> expectedListComparator =
        Comparator.comparing(LongRunningOperation::createdAt).reversed()
            .thenComparing(LongRunningOperation::operationId);

    private static final long now = System.currentTimeMillis();

    protected abstract LongRunningOperationDao getDao();

    @Test
    public void insertOne() {
        var dao = getDao();
        var operation = operation();

        var inserted = join(dao.insert(operation));
        var result = join(dao.findOne(operation.operationId()));
        var list = list(dao, operation.containerId());

        assertTrue(inserted);
        assertEquals(operation, result.orElse(null));
        assertEquals(List.of(operation), list);
    }

    @Test
    public void insertTwice() {
        var dao = getDao();
        var operation = operation();
        assumeTrue(join(dao.insert(operation)));

        var inserted = join(dao.insert(operation));
        var result = join(dao.findOne(operation.operationId()));
        var list = list(dao, operation.containerId());

        assertFalse(inserted);
        assertEquals(operation, result.orElse(null));
        assertEquals(List.of(operation), list);
    }

    @Test
    public void insertMany() {
        var dao = getDao();
        var operation1 = operation();
        var operation2 = operation();
        var operation3 = operation();

        var inserted1 = join(dao.insert(operation1));
        var inserted2 = join(dao.insert(operation2));
        var inserted3 = join(dao.insert(operation3));
        var result1 = join(dao.findOne(operation1.operationId()));
        var result2 = join(dao.findOne(operation2.operationId()));
        var result3 = join(dao.findOne(operation3.operationId()));

        assertTrue(inserted1 && inserted2 && inserted3);
        assertEquals(operation1, result1.orElse(null));
        assertEquals(operation2, result2.orElse(null));
        assertEquals(operation3, result3.orElse(null));
    }

    @Test
    public void insertWhenAbsent() {
        var dao = getDao();
        var operation = operation();

        var old = join(dao.insertIfAbsent(operation));
        var result = join(dao.findOne(operation.operationId()));
        var list = list(dao, operation.containerId());

        assertEquals(Optional.empty(), old);
        assertEquals(operation, result.orElse(null));
        assertEquals(List.of(operation), list);
    }

    @Test
    public void insertWhenPresent() {
        var dao = getDao();
        var operation = operation();
        var operationAgain = operation.toBuilder().setDescription("new desc").build();

        var inserted = join(dao.insert(operation));
        var old = join(dao.insertIfAbsent(operationAgain));
        var result = join(dao.findOne(operation.operationId()));
        var list = list(dao, operation.containerId());

        assertTrue(inserted);
        assertEquals(operation, old.orElse(null));
        assertEquals(operation, result.orElse(null));
        assertEquals(List.of(operation), list);
    }

    @Test
    public void insertIfAbsentTwice() {
        var dao = getDao();
        var operation = operation();
        var operationAgain = operation.toBuilder().setDescription("new desc").build();

        var afterFirst = join(dao.insertIfAbsent(operation));
        var afterSecond = join(dao.insertIfAbsent(operationAgain));
        var result = join(dao.findOne(operation.operationId()));
        var list = list(dao, operation.containerId());

        assertEquals(Optional.empty(), afterFirst);
        assertEquals(operation, afterSecond.orElse(null));
        assertEquals(operation, result.orElse(null));
        assertEquals(List.of(operation), list);
    }

    @Test
    public void insertIfAbsentMany() {
        var dao = getDao();
        var operation1 = operation();
        var operation2 = operation();
        var operation3 = operation();

        var inserted1 = join(dao.insertIfAbsent(operation1));
        var inserted2 = join(dao.insertIfAbsent(operation2));
        var inserted3 = join(dao.insertIfAbsent(operation3));
        var result1 = join(dao.findOne(operation1.operationId()));
        var result2 = join(dao.findOne(operation2.operationId()));
        var result3 = join(dao.findOne(operation3.operationId()));

        assertTrue(inserted1.isEmpty() && inserted2.isEmpty() && inserted3.isEmpty());
        assertEquals(operation1, result1.orElse(null));
        assertEquals(operation2, result2.orElse(null));
        assertEquals(operation3, result3.orElse(null));
    }

    @Test
    public void findZero() {
        var dao = getDao();
        var operationId = operationId();

        var result = join(dao.findOne(operationId));

        assertEquals(Optional.empty(), result);
    }

    @Test
    public void updateNotFound() {
        var dao = getDao();
        var originalOperation = operation();
        var updatedOperation = originalOperation.toBuilder()
            .setOperationId(operationId())
            .setDescription(originalOperation.description() + "_updated")
            .build();
        assumeTrue(join(dao.insert(originalOperation)));

        var updated = join(dao.update(updatedOperation));
        var result = join(dao.findOne(originalOperation.operationId()));

        assertTrue(updated.isEmpty());
        assertEquals(originalOperation, result.orElse(null));
    }

    @Test
    public void updateOutOfDate() {
        var dao = getDao();
        var originalOperation = operation();
        var updatedOperation = originalOperation.toBuilder()
            .setDescription(originalOperation.description() + "_updated")
            .setVersion(originalOperation.version() - 1)
            .build();
        assumeTrue(join(dao.insert(originalOperation)));

        var updated = join(dao.update(updatedOperation));
        var result = join(dao.findOne(originalOperation.operationId()));

        assertTrue(updated.isEmpty());
        assertEquals(originalOperation, result.orElse(null));
    }

    @Test
    public void updateExisting() {
        var dao = getDao();
        var originalOperation = operation();
        var updatedOperation = originalOperation.toBuilder()
            .setDescription(originalOperation.description() + "_updated")
            .setCreatedAt(originalOperation.createdAt() + 666)
            .setCreatedBy(originalOperation.createdBy() + 666)
            .setUpdatedAt(originalOperation.updatedAt() + 11)
            .setStatus(originalOperation.status() + 33)
            .setData(Any.pack(StringValue.of(randomAlphanumeric(8) + "_updated")))
            .build();
        var expectedOperation = updatedOperation.toBuilder()
            .setCreatedAt(originalOperation.createdAt())
            .setCreatedBy(originalOperation.createdBy())
            .setVersion(originalOperation.version() + 1)
            .build();
        assumeTrue(join(dao.insert(originalOperation)));

        var updated = join(dao.update(updatedOperation));
        var result = join(dao.findOne(originalOperation.operationId()));

        assertTrue(updated.isPresent());
        assertEquals(expectedOperation, result.orElse(null));
    }

    @Test
    public void emptyList() {
        var dao = getDao();
        var containerId = containerId();

        var result = list(dao, containerId);

        assertEquals(List.of(), result);
    }

    @Test
    public void listContainers() {
        // arrange
        var dao = getDao();

        var containerId1 = containerId();
        var containerId2 = containerId();

        var operation1_1 = operation().toBuilder().setContainerId(containerId1).build();
        var operation1_2 = operation().toBuilder().setContainerId(containerId1).build();
        var operation2_1 = operation().toBuilder().setContainerId(containerId2).build();

        assumeTrue(join(dao.insert(operation1_1)));
        assumeTrue(join(dao.insert(operation1_2)));
        assumeTrue(join(dao.insert(operation2_1)));

        // act
        var list1 = list(dao, containerId1);
        var list2 = list(dao, containerId2);

        // assert
        var expectedList1 = Stream.of(operation1_1, operation1_2)
            .sorted(expectedListComparator)
            .collect(toList());

        assertEquals(expectedList1, list1);
        assertEquals(List.of(operation2_1), list2);
    }

    @Test
    public void listManyPages() {
        // arrange
        var dao = getDao();
        var containerId = containerId();

        var pageSize = 2;
        var ops333 = operations(containerId, b -> b.setCreatedAt(now + 333)).limit(3);
        var ops222 = operations(containerId, b -> b.setCreatedAt(now + 222)).limit(2);
        var ops111 = operations(containerId, b -> b.setCreatedAt(now + 111)).limit(2);
        var operations = Streams.concat(ops333, ops222, ops111).collect(toList());
        shuffle(operations);

        assumeTrue(
            operations.stream()
                .map(dao::insert)
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .join()
                .reduceLeft(Boolean::logicalAnd));

        // act
        var token = "";
        var pages = new ArrayList<List<LongRunningOperation>>();
        do {
            var pageOpts = new TokenPageOptions(pageSize, token);
            var page = listPaged(dao, containerId, pageOpts);
            pages.add(page.getItems());
            token = page.getNextPageToken();
        } while (!token.isEmpty());

        // assert
        var expected = operations.stream()
            .sorted(expectedListComparator)
            .collect(toList());

        var actual = pages.stream()
            .flatMap(Collection::stream)
            .collect(toList());

        assertEquals(expected, actual);
    }

    @Test
    public void count() {
        // arrange
        var dao = getDao();
        var containerId = containerId();

        var limit = 6;
        var ops444 = operations(containerId, b -> b.setCreatedAt(now + 444)).limit(3);
        var ops333 = operations(containerId, b -> b.setCreatedAt(now + 333)).limit(2);
        var ops222 = operations(containerId, b -> b.setCreatedAt(now + 222)).limit(1);
        var ops111 = operations(containerId, b -> b.setCreatedAt(now + 111)).limit(1);
        var operations = Streams.concat(ops444, ops333, ops222, ops111).collect(toList());
        shuffle(operations);

        assumeTrue(
            operations.stream()
                .map(dao::insert)
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .join()
                .reduceLeft(Boolean::logicalAnd));

        // act
        var count500 = count(dao, containerId, now + 500, limit);
        var count400 = count(dao, containerId, now + 400, limit);
        var count300 = count(dao, containerId, now + 300, limit);
        var count200 = count(dao, containerId, now + 200, limit);
        var count100 = count(dao, containerId, now + 100, limit);

        // assert
        assertEquals(0, count500);
        assertEquals(3, count400);
        assertEquals(5, count300);
        assertEquals(6, count200);
        assertEquals(6, count100);
    }

    @Test
    public void heavyOperations() {
        // arrange
        var dao = getDao();
        var containerId = containerId();

        var pageSize = 100;
        var operations = operations(containerId, b -> b.setData(heavyData()))
            .limit(843)
            .collect(toList());
        shuffle(operations);

        var inFlightLimiter = new InFlightLimiter(500);
        assumeTrue(
            operations.stream()
                .map(operation -> {
                    var f = new CompletableFuture<Boolean>();
                    inFlightLimiter.run(() -> {
                        var insertFuture = dao.insert(operation);
                        CompletableFutures.whenComplete(insertFuture, f);
                        return insertFuture;
                    });
                    return f;
                })
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .join()
                .reduceLeft(Boolean::logicalAnd));

        // act
        var token = "";
        var pages = new ArrayList<List<LongRunningOperation>>();
        do {
            var pageOpts = new TokenPageOptions(pageSize, token);
            var page = listPaged(dao, containerId, pageOpts);
            pages.add(page.getItems());
            token = page.getNextPageToken();
        } while (!token.isEmpty());

        var count = count(dao, containerId, 42, 1000);

        // assert
        var expected = operations.stream()
            .sorted(expectedListComparator)
            .collect(toList());
        var actual = pages.stream()
            .flatMap(Collection::stream)
            .collect(toList());
        assertEquals(expected, actual);

        assertEquals(843, count);
    }

    private static Stream<LongRunningOperation> operations(
        String containerId,
        Consumer<LongRunningOperation.Builder> builderConsumer)
    {
        return Stream.generate(LongRunningOperationDaoTest::operation)
            .map(op -> {
                var builder = op.toBuilder();
                builderConsumer.accept(builder);
                return builder.setContainerId(containerId).build();
            });
    }

    private static List<LongRunningOperation> list(LongRunningOperationDao dao, String containerId) {
        return listPaged(dao, containerId, new TokenPageOptions(10, "")).getItems();
    }

    private static TokenBasePage<LongRunningOperation> listPaged(
        LongRunningOperationDao dao,
        String containerId,
        TokenPageOptions pageOpts)
    {
        return join(dao.list(
            ContainerType.PROJECT,
            containerId,
            LongRunningOperationType.DELETE_METRICS,
            pageOpts
        ));
    }

    private static int count(
        LongRunningOperationDao dao,
        String containerId,
        long createdSince,
        int limit)
    {
        return join(dao.count(
            ContainerType.PROJECT,
            containerId,
            LongRunningOperationType.DELETE_METRICS,
            createdSince,
            limit
        )).intValue();
    }

    private static LongRunningOperation operation() {
        return LongRunningOperation.newBuilder()
            .setOperationId(operationId())
            .setOperationType(LongRunningOperationType.DELETE_METRICS)
            .setContainerId(containerId())
            .setContainerType(ContainerType.PROJECT)
            .setDescription("description " + randomAlphanumeric(8))
            .setCreatedAt(now + random().nextLong(10_000))
            .setCreatedBy("createdBy" + randomAlphanumeric(8))
            .setUpdatedAt(now + random().nextLong(10_000))
            .setStatus(random().nextInt())
            .setData(Any.pack(StringValue.of("data " + randomAlphanumeric(8))))
            .setVersion(random().nextInt(1, 100))
            .build();
    }

    private static Any heavyData() {
        return Any.pack(StringValue.of("data " + randomAlphanumeric(100_000)));
    }

    private static String operationId() {
        return randomAlphanumeric(8);
    }

    private static String containerId() {
        return randomAlphanumeric(10);
    }

    private static ThreadLocalRandom random() {
        return ThreadLocalRandom.current();
    }
}
