package ru.yandex.solomon.gateway.operations.db.memory;

import java.util.Comparator;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.LongRunningOperationType;
import ru.yandex.solomon.gateway.operations.db.LongRunningOperationDao;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static java.util.Comparator.reverseOrder;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.toList;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public class InMemoryLongRunningOperationDao implements LongRunningOperationDao {

    private final ConcurrentNavigableMap<String, LongRunningOperation> table = new ConcurrentSkipListMap<>();
    private final ConcurrentNavigableMap<Key, String> index = new ConcurrentSkipListMap<>();

    public volatile Supplier<CompletableFuture<?>> beforeSupplier;
    public volatile Supplier<CompletableFuture<?>> afterSupplier;

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return async(() -> null);
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return async(() -> null);
    }

    @Override
    public CompletableFuture<Boolean> insert(LongRunningOperation operation) {
        return async(() -> {
            if (table.putIfAbsent(operation.operationId(), operation) != null) {
                return false;
            }

            index.put(Key.of(operation), operation.operationId());

            return true;
        });
    }

    @Override
    public CompletableFuture<Optional<LongRunningOperation>> insertIfAbsent(LongRunningOperation operation) {
        return async(() -> {
            var old = table.putIfAbsent(operation.operationId(), operation);
            if (old != null) {
                return Optional.of(old);
            }

            index.put(Key.of(operation), operation.operationId());

            return Optional.empty();
        });

    }

    @Override
    public CompletableFuture<Optional<LongRunningOperation>> findOne(String operationId) {
        return async(() -> Optional.ofNullable(table.get(operationId)));
    }

    @Override
    public CompletableFuture<Optional<LongRunningOperation>> update(LongRunningOperation operation) {
        return async(() -> {
            var updated = new AtomicBoolean(false);
            var result = table.computeIfPresent(
                operation.operationId(),
                (id, old) -> {
                    if (old.version() != operation.version()) {
                        return old;
                    }

                    updated.set(true);
                    return operation.toBuilder()
                        .setCreatedAt(old.createdAt())
                        .setCreatedBy(old.createdBy())
                        .setVersion(operation.version() + 1)
                        .build();
                });

            return updated.get() ? Optional.of(result) : Optional.empty();
        });
    }

    @Override
    public CompletableFuture<TokenBasePage<LongRunningOperation>> list(
        ContainerType containerType,
        String containerId,
        LongRunningOperationType operationType,
        TokenPageOptions pageOpts)
    {
        return async(() -> {
            var slice = index.subMap(
                new Key(containerType, containerId, operationType, Long.MAX_VALUE, ""),
                new Key(containerType, containerId, operationType, Long.MIN_VALUE, "")
            );

            var ops = slice.values().stream()
                .skip(pageOpts.getOffset())
                .limit(pageOpts.getSize())
                .map(table::get)
                .collect(toList());

            return new TokenBasePage<>(
                ops,
                ops.isEmpty() ? "" : String.valueOf(pageOpts.getSize() + pageOpts.getOffset())
            );
        });
    }

    @Override
    public CompletableFuture<Long> count(
        ContainerType containerType,
        String containerId,
        LongRunningOperationType operationType,
        long createdSince,
        int limit)
    {
        return async(() -> {
            var slice = index.subMap(
                new Key(containerType, containerId, operationType, Long.MAX_VALUE, ""),
                new Key(containerType, containerId, operationType, Long.MIN_VALUE, "")
            );

            return slice.values().stream()
                .map(table::get)
                .filter(op -> op.createdAt() > createdSince)
                .limit(limit)
                .count();
        });
    }

    private <T> CompletableFuture<T> async(Supplier<T> fn) {
        return before().thenComposeAsync(i -> {
            var result = fn.get();
            return after().thenApply(a -> result);
        });
    }

    private CompletableFuture<?> before() {
        var copy = beforeSupplier;
        if (copy == null) {
            return completedFuture(null);
        }

        return copy.get();
    }

    private CompletableFuture<?> after() {
        var copy = afterSupplier;
        if (copy == null) {
            return completedFuture(null);
        }

        return copy.get();
    }

    private static record Key(
        ContainerType containerType,
        String containerId,
        LongRunningOperationType operationType,
        long createdAt,
        String operationId) implements Comparable<Key>
    {

        private static final Comparator<Key> cmp =
            Comparator.comparing(Key::containerType)
                .thenComparing(Key::containerId)
                .thenComparing(Key::operationType)
                .thenComparing(Key::createdAt, reverseOrder())
                .thenComparing(Key::operationId);

        static Key of(LongRunningOperation operation) {
            return new Key(
                operation.containerType(),
                operation.containerId(),
                operation.operationType(),
                operation.createdAt(),
                operation.operationId());
        }

        @Override
        public int compareTo(Key that) {
            return cmp.compare(this, that);
        }
    }
}
