package ru.yandex.solomon.gateway.push;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.collect.ImmutableList;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.util.CharsetUtil;
import org.apache.commons.lang3.StringUtils;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.discovery.cluster.ClusterMapperStub;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.monitoring.coremon.TPushedDataRequest;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationType;
import ru.yandex.solomon.auth.roles.RoleSet;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.coremon.client.CoremonClientStub;
import ru.yandex.solomon.coremon.client.ShardInfo;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.UnknownShardLocation;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Sergey Polovko
 */
public class PushMetricProcessorTest {

    private final PushDisabler pushDisabler = new PushDisabler(new ClusterMapperStub());
    private final FeatureFlagsHolder featureFlagsHolder = makeFlagsHolder();

    @Test
    public void pushToReadOnlyShard() {
        ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.READ_ONLY);
        var processor = new PushMetricProcessor(new CoremonClientStub(), pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, MetricFormat.JSON, Unpooled.EMPTY_BUFFER, 0, Account.ANONYMOUS);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(0, response.getSuccessMetricCount());
        Assert.assertEquals(UrlStatusType.SHARD_IS_NOT_WRITABLE, response.getStatus());
        Assert.assertEquals(
                "shard project=solomon;cluster=production;service=fetcher in state READ_ONLY, write is impossible",
                response.getErrorMessage());
    }

    @Test
    public void pushToShardWithUnknownLocation() {
        var coremonClient = new CoremonClientStub() {
            @Override
            public ImmutableList<String> shardHosts(int shardId) {
                throw new UnknownShardLocation(shardId);
            }
        };

        ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");
        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, MetricFormat.JSON, Unpooled.EMPTY_BUFFER, 0, Account.ANONYMOUS);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(0, response.getSuccessMetricCount());
        Assert.assertEquals(UrlStatusType.UNKNOWN_SHARD, response.getStatus());
        Assert.assertEquals(
                "shard project=solomon;cluster=production;service=fetcher was not found",
                response.getErrorMessage());
    }

    @Test
    public void pushToHalfFailedCluster() {
        ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");
        final String content = "test-data";
        final ByteBuf contentBuf = Unpooled.copiedBuffer(content, CharsetUtil.UTF_8);
        final int contentSize = measureContentSize(content);
        final MetricFormat format = MetricFormat.JSON;

        AtomicInteger fails = new AtomicInteger(0);
        var coremonClient = new CoremonClientStub() {
            @Override
            public CompletableFuture<TDataProcessResponse> processPushedData(String host, TPushedDataRequest request) {
                if ("host-a".equals(host)) {
                    fails.incrementAndGet();
                    return failedFuture(new RuntimeException(host + " is not available"));
                }

                Assert.assertEquals("host-b", host);

                Assert.assertEquals(format, request.getFormat());
                Assert.assertEquals(content, request.getContent().toString(CharsetUtil.UTF_8));
                Assert.assertEquals(shardKey.toUniqueId().hashCode(), request.getNumId());

                return completedFuture(TDataProcessResponse.newBuilder()
                        .setStatus(UrlStatusType.OK)
                        .setSuccessMetricCount(37)
                        .build());
            }

            @Override
            public ImmutableList<String> shardHosts(int shardId) {
                Assert.assertEquals(shardKey.toUniqueId().hashCode(), shardId);
                return ImmutableList.of("host-a", "host-b");
            }
        };

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, format, contentBuf, contentSize, Account.ANONYMOUS);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(UrlStatusType.OK, response.getStatus());
        Assert.assertEquals(37, response.getSuccessMetricCount());
        Assert.assertEquals(1, fails.get());
    }

    @Test
    public void pushToFailedCluster() {
        ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");

        AtomicInteger fails = new AtomicInteger(0);
        var coremonClient = new CoremonClientStub() {
            @Override
            public CompletableFuture<TDataProcessResponse> processPushedData(String host, TPushedDataRequest request) {
                fails.incrementAndGet();
                return failedFuture(new RuntimeException(host + " is not available"));
            }

            @Override
            public ImmutableList<String> shardHosts(int shardId) {
                Assert.assertEquals(shardKey.toUniqueId().hashCode(), shardId);
                return ImmutableList.of("host-a", "host-b");
            }
        };

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        try {
            var pushRequest = new PushRequest(shardKey, MetricFormat.PROMETHEUS, Unpooled.EMPTY_BUFFER, 0, Account.ANONYMOUS);
            processor.processPush(pushRequest, false).join();
            Assert.fail("expected exception was not thrown");
        } catch (CompletionException e) {
            String message = e.getCause().getMessage();
            Assert.assertTrue("host-a is not available".equals(message) || "host-b is not available".equals(message));
        }
    }

    @Test
    public void anonymousPushToUnknownShard() {
        ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");

        SolomonConfHolder confHolder = makeConfHolder(new ShardKey("some", "other", "shard"), ShardState.RW);
        var processor = new PushMetricProcessor(new CoremonClientStub(), pushDisabler, confHolder, featureFlagsHolder);

        try {
            var pushRequest = new PushRequest(shardKey, MetricFormat.JSON, Unpooled.EMPTY_BUFFER, 0, Account.ANONYMOUS);
            processor.processPush(pushRequest, false).join();
            Assert.fail("expected exception was not thrown");
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            Assert.assertNotNull(cause);
            Assert.assertEquals(
                    "Shard with key 'project=solomon;cluster=production;service=fetcher' was not found",
                    cause.getMessage());
        }
    }

    @Test
    public void anonymousPushToKnownShard() {
        ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");
        final String content = "test-data";
        final ByteBuf contentBuf = Unpooled.copiedBuffer(content, CharsetUtil.UTF_8);
        final int contentSize = measureContentSize(content);
        final MetricFormat format = MetricFormat.JSON;

        var coremonClient = makeCoremonClient(shardKey, content, format, "host-a", "host-b", 10);

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, format, contentBuf, contentSize, Account.ANONYMOUS);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(UrlStatusType.OK, response.getStatus());
        Assert.assertEquals(10, response.getSuccessMetricCount());
    }

    @Test
    public void authenticatedPushToUnknownShard() {
        final ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");
        final String content = "test-data";
        final ByteBuf contentBuf = Unpooled.copiedBuffer(content, CharsetUtil.UTF_8);
        final int contentSize = measureContentSize(content);
        final MetricFormat format = MetricFormat.JSON;
        final Account account = new Account("jamel", AuthType.OAuth, AuthorizationType.PROJECT_ACL, RoleSet.ALL);

        var coremonClient = new CoremonClientStub() {
            @Override
            public CompletableFuture<ShardInfo> createShard(String projectId, String cluster, String service, String createdBy) {
                var newShardKey = new ShardKey(projectId, cluster, service);
                Assert.assertEquals(shardKey, newShardKey);
                Assert.assertEquals(account.getId(), createdBy);

                String shardId = "my-awesome-shard";
                int numId = 777;
                return completedFuture(new ShardInfo(shardId, numId, ImmutableList.of("host-1", "host-1")));
            }

            @Override
            public CompletableFuture<TDataProcessResponse> processPushedData(String host, TPushedDataRequest request) {
                Assert.assertTrue("host-1".equals(host) || "host-2".equals(host));

                Assert.assertEquals(format, request.getFormat());
                Assert.assertEquals(content, request.getContent().toString(CharsetUtil.UTF_8));
                Assert.assertEquals(777, request.getNumId());

                return completedFuture(TDataProcessResponse.newBuilder()
                        .setStatus(UrlStatusType.OK)
                        .setSuccessMetricCount(42)
                        .build());
            }

            @Override
            public ImmutableList<String> shardHosts(int shardId) {
                Assert.fail("must not be called");
                return null;
            }
        };

        SolomonConfHolder confHolder = makeConfHolder(new ShardKey("some", "other", "shard"), ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, format, contentBuf, contentSize, account);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(UrlStatusType.OK, response.getStatus());
        Assert.assertEquals(42, response.getSuccessMetricCount());
    }

    @Test
    public void authenticatedPushToKnownShard() {
        ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");
        final String content = "test-data";
        final ByteBuf contentBuf = Unpooled.copiedBuffer(content, CharsetUtil.UTF_8);
        final int contentSize = measureContentSize(content);
        final MetricFormat format = MetricFormat.SPACK;
        final Account account = new Account("jamel", AuthType.OAuth, AuthorizationType.PROJECT_ACL, RoleSet.ALL);

        var coremonClient = makeCoremonClient(shardKey, content, format, "host-d", "host-e", 100);

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, format, contentBuf, contentSize, account);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(UrlStatusType.OK, response.getStatus());
        Assert.assertEquals(100, response.getSuccessMetricCount());
    }

    @Test
    public void requestSizeLimitExceeded() {
        ShardKey shardKey = new ShardKey("solomon", "production", "fetcher");
        String content = "{\"metrics\": [" + StringUtils.repeat("{\"name\": \"cpu\", \"value\": 5}", ", ", 1000) + "]}";
        ByteBuf contentBuf = Unpooled.copiedBuffer(content, CharsetUtil.UTF_8);
        int contentSize = measureContentSize(content);

        var coremonClient = makeCoremonClient(shardKey);

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, MetricFormat.JSON, contentBuf, contentSize, Account.ANONYMOUS);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(UrlStatusType.RESPONSE_TOO_LARGE, response.getStatus());
    }

    @Test
    public void pushToUnknownCloudService() {
        // yc.public.cloud.id is public cloud project and unknown service is unknown service provider
        ShardKey shardKey = new ShardKey("yc.public.cloud.id", "folder_id", "unknown_service");
        final String content = "test-data";
        final ByteBuf contentBuf = Unpooled.copiedBuffer(content, CharsetUtil.UTF_8);
        final int contentSize = measureContentSize(content);
        final MetricFormat format = MetricFormat.SPACK;
        final Account account = new Account("jamel", AuthType.IAM, AuthorizationType.IAM, RoleSet.ALL);

        var coremonClient = makeCoremonClient(shardKey);

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);

        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, format, contentBuf, contentSize, account);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(UrlStatusType.AUTH_ERROR, response.getStatus());
    }

    @Test
    public void pushToIntenralCloudProject() {
        // yc.cloud.id project has INTERNAL_CLOUD flag
        ShardKey shardKey = new ShardKey("yc.cloud.id", "folder_id", "unknown_service");
        final String content = "test-data";
        final ByteBuf contentBuf = Unpooled.copiedBuffer(content, CharsetUtil.UTF_8);
        final int contentSize = measureContentSize(content);
        final MetricFormat format = MetricFormat.SPACK;
        final Account account = new Account("jamel", AuthType.IAM, AuthorizationType.IAM, RoleSet.ALL);

        var coremonClient = makeCoremonClient(shardKey);

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, format, contentBuf, contentSize, account);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(UrlStatusType.OK, response.getStatus());
    }

    @Test
    public void pushToKnownCloudService() {
        // known_service is a predefined service provider
        ShardKey shardKey = new ShardKey("yc.public.cloud.id", "folder_id", "known_service");
        final String content = "test-data";
        final ByteBuf contentBuf = Unpooled.copiedBuffer(content, CharsetUtil.UTF_8);
        final int contentSize = measureContentSize(content);
        final MetricFormat format = MetricFormat.SPACK;
        final Account account = new Account("jamel", AuthType.IAM, AuthorizationType.IAM, RoleSet.ALL);

        var coremonClient = makeCoremonClient(shardKey);

        SolomonConfHolder confHolder = makeConfHolder(shardKey, ShardState.RW);
        var processor = new PushMetricProcessor(coremonClient, pushDisabler, confHolder, featureFlagsHolder);

        var pushRequest = new PushRequest(shardKey, format, contentBuf, contentSize, account);
        var response = processor.processPush(pushRequest, false).join();

        Assert.assertEquals(UrlStatusType.OK, response.getStatus());
    }

    private CoremonClientStub makeCoremonClient(ShardKey shardKey) {
        return new CoremonClientStub() {
            @Override
            public CompletableFuture<TDataProcessResponse> processPushedData(String host, TPushedDataRequest request) {
                return completedFuture(TDataProcessResponse.newBuilder()
                        .setStatus(UrlStatusType.OK)
                        .setSuccessMetricCount(10)
                        .build());
            }

            @Override
            public ImmutableList<String> shardHosts(int shardId) {
                Assert.assertEquals(shardKey.toUniqueId().hashCode(), shardId);
                return ImmutableList.of("host-a", "host-b");
            }
        };
    }

    private CoremonClientStub makeCoremonClient(
        ShardKey shardKey,
        String content,
        MetricFormat format,
        String host1,
        String host2,
        int metricsWritten)
    {
        return new CoremonClientStub() {
            @Override
            public CompletableFuture<TDataProcessResponse> processPushedData(String host, TPushedDataRequest request) {
                Assert.assertTrue(host1.equals(host) || host2.equals(host));

                Assert.assertEquals(format, request.getFormat());
                Assert.assertEquals(content, request.getContent().toString(CharsetUtil.UTF_8));
                Assert.assertEquals(shardKey.toUniqueId().hashCode(), request.getNumId());

                return completedFuture(TDataProcessResponse.newBuilder()
                        .setStatus(UrlStatusType.OK)
                        .setSuccessMetricCount(metricsWritten)
                        .build());
            }

            @Override
            public ImmutableList<String> shardHosts(int shardId) {
                Assert.assertEquals(shardKey.toUniqueId().hashCode(), shardId);
                return ImmutableList.of(host1, host2);
            }
        };
    }

    private static int measureContentSize(String content) {
        return content.getBytes(CharsetUtil.UTF_8).length;
    }

    private static FeatureFlagsHolder makeFlagsHolder() {
        FeatureFlagHolderStub flagsHolder = new FeatureFlagHolderStub();
        flagsHolder.setFlag("yc.cloud.id", FeatureFlag.INTERNAL_CLOUD, true);
        return flagsHolder;
    }

    private static SolomonConfHolder makeConfHolder(ShardKey shardKey, ShardState state) {
        Project project = Project.newBuilder()
                .setId(shardKey.getProject())
                .setName(shardKey.getProject())
                .setOwner("robot-solomon")
                .build();

        Cluster cluster = Cluster.newBuilder()
                .setProjectId(shardKey.getProject())
                .setId(shardKey.getProject() + '_' + shardKey.getCluster())
                .setName(shardKey.getCluster())
                .build();

        Service service = Service.newBuilder()
                .setProjectId(shardKey.getProject())
                .setId(shardKey.getProject() + '_' + shardKey.getService())
                .setName(shardKey.getService())
                .build();

        Shard shard = Shard.newBuilder()
                .setProjectId(project.getId())
                .setId(shardKey.toUniqueId())
                .setNumId(shardKey.toUniqueId().hashCode())
                .setClusterId(cluster.getId())
                .setClusterName(cluster.getName())
                .setServiceId(service.getId())
                .setServiceName(service.getName())
                .setMaxResponseSizeBytes(1000)
                .setState(state)
                .build();

        ServiceProvider serviceProvider = ServiceProvider.newBuilder()
                .setId("known_service")
                .setCloudId("cloud_id")
                .build();

        var confHolder = new SolomonConfHolder();
        confHolder.onConfigurationLoad(SolomonConfWithContext.create(new SolomonRawConf(
                List.of(serviceProvider),
                List.of(project),
                List.of(cluster),
                List.of(service),
                List.of(shard)
        )));
        return confHolder;
    }
}
