package ru.yandex.solomon.gateway.tasks;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Supplier;

import com.google.protobuf.Any;
import com.google.protobuf.StringValue;
import io.grpc.Status;
import org.apache.commons.lang3.RandomStringUtils;

import ru.yandex.solomon.scheduler.proto.GetTaskRequest;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.supplyAsync;

/**
 * @author Stanislav Kashirin
 */
class RemoteTaskClientStub extends RemoteTask.RemoteTaskClient {

    volatile Supplier<CompletableFuture<?>> beforeSupplier;
    volatile String interruptId;
    private final ConcurrentMap<String, Task> taskById = new ConcurrentHashMap<>();

    RemoteTaskClientStub(RetryConfig retry) {
        super(retry);
    }

    @Override
    protected CompletableFuture<String> interrupt() {
        return before().thenApply(i -> {;
            var task = taskById.get(interruptId);
            return task == null ? "" : task.getId();
        });
    }

    @Override
    protected CompletableFuture<String> schedule() {
        return before().thenApply(ignore -> prepareTask().getId());
    }

    @Override
    protected CompletableFuture<Task> getTask(GetTaskRequest request) {
        return before().thenApply(ignore -> {
            var task = taskById.get(request.getId());
            if (task == null) {
                throw Status.NOT_FOUND.withDescription("not found task").asRuntimeException();
            }
            return task;
        });
    }

    Task prepareTask() {
        var id = RandomStringUtils.randomAlphanumeric(8);
        var task = Task.newBuilder()
            .setId(id)
            .setType("test_type")
            .setExecuteAt(System.currentTimeMillis())
            .setParams(Any.pack(StringValue.of("pam-param")))
            .build();

        taskById.putIfAbsent(id, task);
        return task;
    }

    Task getTaskById(String id) {
        return getTask(GetTaskRequest.newBuilder().setId(id).build()).join();
    }

    void putTask(Task task) {
        taskById.put(task.getId(), task);
    }

    void removeTaskById(String id) {
        taskById.remove(id);
    }

    private CompletableFuture<?> before() {
        var copy = beforeSupplier;
        if (copy == null) {
            return completedFuture(null);
        }

        return supplyAsync(beforeSupplier);
    }
}
