package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.google.protobuf.Any;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.coremon.api.task.DeleteMetricsCheckResult;
import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.gateway.api.task.DeleteMetricsProgress;
import ru.yandex.gateway.api.task.DeleteMetricsProgress.PhaseOnReplicaProgress;
import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.util.time.InstantUtils;

import static java.util.concurrent.TimeUnit.DAYS;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * @author Stanislav Kashirin
 */
public class DeleteMetricsTaskHandlerTest {

    private SolomonConfWithContext conf;

    @Before
    public void setUp() throws Exception {
        var projects = List.of(
            Project.newBuilder()
                .setId("p")
                .setName("p")
                .setOwner("any")
                .build());

        var clusters = List.of(
            Cluster.newBuilder()
                .setProjectId("p")
                .setId("c")
                .setName("c")
                .build());

        var services = List.of(
            Service.newBuilder()
                .setProjectId("p")
                .setId("s1")
                .setName("s1")
                .build(),
            Service.newBuilder()
                .setProjectId("p")
                .setId("s2")
                .setName("s2")
                .build(),
            Service.newBuilder()
                .setProjectId("p")
                .setId("s3")
                .setName("s3")
                .build());
        var shards = List.of(
            Shard.newBuilder()
                .setProjectId("p")
                .setId("p" + "_" + "c" + "_" + "s1")
                .setNumId(1)
                .setClusterId("c")
                .setClusterName("c")
                .setServiceId("s1")
                .setServiceName("s1")
                .setShardSettings(shardSettings(7))
                .build(),
            Shard.newBuilder()
                .setProjectId("p")
                .setId("p" + "_" + "c" + "_" + "s2")
                .setNumId(2)
                .setClusterId("c")
                .setClusterName("c")
                .setServiceId("s2")
                .setServiceName("s2")
                .setShardSettings(shardSettings(8))
                .build(),
            Shard.newBuilder()
                .setProjectId("p")
                .setId("no_ttl_shard")
                .setNumId(3)
                .setClusterId("c")
                .setClusterName("c")
                .setServiceId("s3")
                .setServiceName("s3")
                .setShardSettings(shardSettings(0))
                .build());

        var rawConf = new SolomonRawConf(List.of(), projects, clusters, services, shards);
        this.conf = SolomonConfWithContext.create(rawConf);
    }

    @Test
    public void noInterferenceWithTtl() {
        var progress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(
                shuffledList(
                    PhaseOnReplicaProgress.newBuilder()
                        .addAllOnShards(
                            shuffledList(
                                remoteTaskProgress(true, 1, currentTimeSeconds()),
                                remoteTaskProgress(true, 2, currentTimeSeconds())))
                        .build(),
                    PhaseOnReplicaProgress.newBuilder()
                        .addAllOnShards(
                            shuffledList(
                                remoteTaskProgress(true, 1, currentTimeSeconds()),
                                remoteTaskProgress(true, 2, currentTimeSeconds())))
                        .build()
                ))
            .build();

        var result = DeleteMetricsTaskHandler.validateNoTtlInterference(Instant.now(), conf, progress);

        assertNull(result);
    }

    @Test
    public void interferenceWithTtl() {
        var progress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(
                shuffledList(
                    PhaseOnReplicaProgress.newBuilder()
                        .addAllOnShards(
                            shuffledList(
                                remoteTaskProgress(true, 1, currentTimeSeconds() - daysToSeconds(30)),
                                remoteTaskProgress(true, 2, currentTimeSeconds() - daysToSeconds(7))))
                        .build(),
                    PhaseOnReplicaProgress.newBuilder()
                        .addAllOnShards(
                            shuffledList(
                                remoteTaskProgress(true, 1, currentTimeSeconds() - daysToSeconds(30)),
                                remoteTaskProgress(true, 2, currentTimeSeconds())))
                        .build()
                ))
            .build();

        var result = DeleteMetricsTaskHandler.validateNoTtlInterference(Instant.now(), conf, progress);

        assertNotNull(result);
    }

    @Test
    public void skipInterferenceValidationWhenCheckIsNotFinished() {
        var progress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(
                shuffledList(
                    PhaseOnReplicaProgress.newBuilder()
                        .addAllOnShards(
                            shuffledList(
                                remoteTaskProgress(true, 1, currentTimeSeconds() - daysToSeconds(30)),
                                remoteTaskProgress(true, 2, currentTimeSeconds() - daysToSeconds(30))))
                        .build(),
                    PhaseOnReplicaProgress.newBuilder()
                        .addAllOnShards(
                            shuffledList(
                                remoteTaskProgress(true, 1, currentTimeSeconds() - daysToSeconds(30)),
                                remoteTaskProgress(false, 2, currentTimeSeconds())))
                        .build()))
            .build();

        var result = DeleteMetricsTaskHandler.validateNoTtlInterference(Instant.now(), conf, progress);

        assertNull(result);
    }

    @Test
    public void noInterferenceWhenNoTtlConfigured() {
        var progress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(
                shuffledList(
                    PhaseOnReplicaProgress.newBuilder()
                        .addAllOnShards(
                            shuffledList(
                                remoteTaskProgress(true, 3, currentTimeSeconds()),
                                remoteTaskProgress(true, 2, currentTimeSeconds()),
                                remoteTaskProgress(true, 1, currentTimeSeconds())))
                        .build(),
                    PhaseOnReplicaProgress.newBuilder()
                        .addAllOnShards(
                            shuffledList(
                                remoteTaskProgress(true, 3, currentTimeSeconds() - daysToSeconds(365 * 5)),
                                remoteTaskProgress(true, 2, currentTimeSeconds()),
                                remoteTaskProgress(true, 1, currentTimeSeconds())))
                        .build()))
            .build();

        var result = DeleteMetricsTaskHandler.validateNoTtlInterference(Instant.now(), conf, progress);

        assertNull(result);
    }

    private static RemoteTaskProgress remoteTaskProgress(boolean complete, int numId, int minLastPointSeconds) {
        return RemoteTaskProgress.newBuilder()
            .setComplete(complete)
            .setRemoteTask(
                Task.newBuilder()
                    .setParams(
                        Any.pack(
                            DeleteMetricsParams.newBuilder()
                                .setNumId(numId)
                                .build()))
                    .setResult(
                        Any.pack(
                            DeleteMetricsCheckResult.newBuilder()
                                .setMinLastPointSeconds(minLastPointSeconds)
                                .build())))
            .build();
    }

    private static int currentTimeSeconds() {
        return InstantUtils.millisecondsToSeconds(System.currentTimeMillis());
    }

    private static int daysToSeconds(int days) {
        return (int) DAYS.toSeconds(days);
    }

    private static ShardSettings shardSettings(int metricsTtlDays) {
        return ShardSettings.of(
            ShardSettings.Type.PULL,
            null,
            42,
            metricsTtlDays,
            DecimPolicy.DEFAULT,
            ShardSettings.AggregationSettings.EMPTY,
            null);
    }

    @SafeVarargs
    private static <T> List<T> shuffledList(T... items) {
        var result = Arrays.asList(items);
        Collections.shuffle(result);
        return result;
    }
}
