package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.function.BooleanSupplier;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.protobuf.Any;
import com.google.protobuf.Message;
import io.grpc.Status;
import io.grpc.Status.Code;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.coremon.api.task.DeleteMetricsMoveProgress;
import ru.yandex.coremon.api.task.DeleteMetricsMoveProgress.MoveToDeletedMetricsProgress;
import ru.yandex.coremon.api.task.DeleteMetricsMoveResult;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackProgress;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackProgress.RollbackDeletedMetricsProgress;
import ru.yandex.coremon.api.task.DeleteMetricsTerminateResult;
import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.gateway.api.task.DeleteMetricsProgress;
import ru.yandex.gateway.api.task.DeleteMetricsProgress.RollbackRequested;
import ru.yandex.gateway.api.task.DeleteMetricsResult.ReplicaResult;
import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.misc.random.Random2;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.frontend.DeleteMetricsConfig;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.coremon.client.CoremonClientStub;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.LongRunningOperationType;
import ru.yandex.solomon.gateway.operations.db.memory.InMemoryLongRunningOperationDao;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationMetrics;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationStatus;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationTrackerNoOp;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.ExecutionContextStub;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Complete;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Fail;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Reschedule;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.grpc.Proto;
import ru.yandex.solomon.scheduler.proto.Task.State;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.Matchers.lessThanOrEqualTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.gateway.api.task.DeleteMetricsProgress.PhaseOnReplicaProgress;
import static ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager.unpackData;
import static ru.yandex.solomon.gateway.tasks.deleteMetrics.DeleteMetricsRandom.params;

/**
 * @author Stanislav Kashirin
 */
public class DeleteMetricsTaskTest {

    private static final String REPLICA_0 = "replica_0";
    private static final String REPLICA_1 = "replica_1";

    private static final Duration IDLE_RESCHEDULE_MAX_DELAY = Duration.ofSeconds(20);

    @Rule
    public Timeout globalTimeout = Timeout.builder()
            .withTimeout(1, TimeUnit.MINUTES)
            .withLookingForStuckThread(true)
            .build();

    private RetryConfig retryConfig;
    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private DeleteMetricsOperationManager manager;
    private CoremonClientStub coremonClient;

    @Before
    public void setUp() throws Exception {
        retryConfig = RetryConfig.DEFAULT
                .withNumRetries(Integer.MAX_VALUE)
                .withMaxDelay(0);
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        manager = new DeleteMetricsOperationManager(new InMemoryLongRunningOperationDao());
        coremonClient = new CoremonClientStub();
        coremonClient.addCluster(REPLICA_0);
        coremonClient.addCluster(REPLICA_1);
    }

    @After
    public void tearDown() {
        if (timer != null) {
            timer.shutdownNow();
        }
    }

    @Test
    public void rescheduleOnError() {
        // arrange
        retryConfig = RetryConfig.DEFAULT
            .withNumRetries(3)
            .withMaxDelay(0);
        coremonClient.beforeSupplier = () -> failedFuture(Status.ABORTED.withDescription("hi").asRuntimeException());

        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        // act
        proc.start().join();

        // assert
        var event = context.takeDoneEvent(Reschedule.class);
        var delay = event.executeAt() - System.currentTimeMillis();
        assertThat(delay, allOf(
            greaterThanOrEqualTo(0L),
            lessThanOrEqualTo(TimeUnit.DAYS.toMillis(1))));

        var operation = getOperation(context.task().id());
        assertEquals(DeleteMetricsOperationStatus.DELETING.value, operation.status());
    }

    @Test
    public void rescheduleWhenCheckShardIdleOnReplica0() throws InterruptedException {
        // arrange
        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        // act
        var future = proc.start();

        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(0));
        var progress = proc.progress();
        rescheduleTasksOnReplica(progress.getCheckOnReplicas(0), rescheduleAt);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        assertEquals(List.of(), coremonClient.tasks(REPLICA_1));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.DELETING.value, operation.status());
        assertEquals(0.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void rescheduleWhenCheckShardIdleOnReplica1() throws InterruptedException {
        // arrange
        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        // act
        var future = proc.start();

        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(0));
        completeTasksOnReplica(proc.progress().getCheckOnReplicas(0));

        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(1));
        var progress = proc.progress();
        rescheduleTasksOnReplica(progress.getCheckOnReplicas(1), rescheduleAt);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicaComplete(eventProgress.getCheckOnReplicas(0));

        coremonClient.tasks(REPLICA_0)
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.DELETING.value, operation.status());
        assertEquals(0.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    /**
     * Ideally, it should be "fail-fast" instead of "reschedule", but it's done in this way for simplicity
     */
    @Test
    public void rescheduleWhenOneCheckShardCompletedWithNotOkButOtherIdle() throws InterruptedException {
        // arrange
        coremonClient.beforeCluster(REPLICA_1, () -> {
            throw Status.ABORTED.withDescription("unexpected call to " + REPLICA_1).asRuntimeException();
        });

        final int atLeastShards = 2;
        var params = params(atLeastShards);
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        // act
        var future = proc.start();

        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(0));
        var progress = proc.progress();
        var shards = progress.getCheckOnReplicas(0).getOnShardsList();
        var notOkTaskId = random().randomElement(shards).getRemoteTaskId();
        for (var shard : shards) {
            var task = coremonClient.taskById(REPLICA_0, shard.getRemoteTaskId()).toBuilder();
            if (shard.getRemoteTaskId().equals(notOkTaskId)) {
                task.setState(State.COMPLETED)
                    .setStatus(Proto.toProto(Status.INTERNAL.withDescription("hi")));
            } else {
                task.setState(State.SCHEDULED)
                    .setExecuteAt(rescheduleAt);
            }
            coremonClient.putTask(REPLICA_0, task.build());
        }

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        var operation = getOperation(context.task().id());
        assertEquals(DeleteMetricsOperationStatus.DELETING.value, operation.status());
    }

    @Test
    public void failWhenAllCheckShardsCompletedAndOneWithNotOk() throws InterruptedException {
        // arrange
        coremonClient.beforeCluster(REPLICA_1, () -> {
            throw Status.ABORTED.withDescription("unexpected call to " + REPLICA_1).asRuntimeException();
        });

        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        // act
        var future = proc.start();

        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(0));
        var progress = proc.progress();
        var shards = progress.getCheckOnReplicas(0).getOnShardsList();
        var notOkTaskId = random().randomElement(shards).getRemoteTaskId();
        for (var shard : shards) {
            var taskCompleted = coremonClient.taskById(REPLICA_0, shard.getRemoteTaskId()).toBuilder()
                .setState(State.COMPLETED)
                .setStatus(
                    shard.getRemoteTaskId().equals(notOkTaskId)
                        ? Proto.toProto(Status.FAILED_PRECONDITION.withDescription("hi"))
                        : Proto.toProto(Status.OK))
                .build();

            coremonClient.putTask(REPLICA_0, taskCompleted);
        }

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Fail.class);
        var status = Status.fromThrowable(event.throwable());
        assertEquals(status.toString(), Code.FAILED_PRECONDITION, status.getCode());
        assertEquals(status.toString(), "hi", status.getDescription());

        var operation = getOperation(context.task().id());
        assertEquals(System.currentTimeMillis(), operation.updatedAt(), 10_000d);

        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.METRICS_HAVE_RECENT_WRITES.value, operation.status());
        assertEquals("hi", data.getStatusMessage());
        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void rescheduleWhenMoveShardIdleOnReplica0() throws InterruptedException {
        // arrange
        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        // act
        var future = proc.start();

        completeCheckPhaseOnBothReplicas(proc);

        // run move phase
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(0));
        var progress = proc.progress();
        rescheduleTasksOnReplica(progress.getMoveOnReplicas(0), rescheduleAt);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());

        coremonClient.tasks(REPLICA_1)
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.DELETING.value, operation.status());
        assertEquals(0.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void rescheduleWhenMoveShardIdleWithSomeProgressOnReplica0() throws InterruptedException {
        // arrange
        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        var moveProgress = DeleteMetricsMoveProgress.newBuilder()
            .setMoveToDeletedMetrics(
                MoveToDeletedMetricsProgress.newBuilder()
                    .setProgress(0.17)
                    .setEstimatedTotalMetrics(9000))
            .build();

        // act
        var future = proc.start();

        completeCheckPhaseOnBothReplicas(proc);

        // run move phase
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(0));
        var progress = proc.progress();
        rescheduleTasksOnReplica(progress.getMoveOnReplicas(0), rescheduleAt, moveProgress);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());

        coremonClient.tasks(REPLICA_1)
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.DELETING.value, operation.status());
        assertEquals(17.0 / 2, data.getProgressPercentage(), 0.0);
        assertEquals(9000 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void rescheduleWhenMoveShardIdleOnReplica1() throws InterruptedException {
        // arrange
        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        var moveResult0 = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(9001)
            .build();
        var moveProgress1 = DeleteMetricsMoveProgress.newBuilder()
            .setMoveToDeletedMetrics(
                MoveToDeletedMetricsProgress.newBuilder()
                    .setProgress(0.20)
                    .setEstimatedTotalMetrics(9000))
            .build();

        // act
        var future = proc.start();

        completeCheckPhaseOnBothReplicas(proc);

        // run move phase
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(0));
        completeTasksOnReplica(proc.progress().getMoveOnReplicas(0), moveResult0);

        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(1));
        var progress = proc.progress();
        rescheduleTasksOnReplica(progress.getMoveOnReplicas(1), rescheduleAt, moveProgress1);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());
        assertReplicaComplete(eventProgress.getMoveOnReplicas(0));

        coremonClient.tasks(REPLICA_0)
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.DELETING.value, operation.status());
        assertEquals(60.0, data.getProgressPercentage(), 0.0);
        assertEquals(9001 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void rescheduleToPermDelAtWhenMoveReplicasComplete() throws InterruptedException {
        // arrange
        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var moveResult0 = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(8999)
            .build();
        var moveResult1 = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(9009)
            .build();

        // act
        var future = proc.start();

        completeCheckPhaseOnBothReplicas(proc);

        // complete move phase
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(0));
        completeTasksOnReplica(proc.progress().getMoveOnReplicas(0), moveResult0);
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(1));
        var progress = proc.progress();
        completeTasksOnReplica(progress.getMoveOnReplicas(1), moveResult1);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertEquals(params.getPermanentDeletionAt(), event.executeAt());

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());
        assertReplicasComplete(eventProgress.getMoveOnReplicasList());

        coremonClient.tasks()
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.WAITING_FOR_PERMANENT_DELETION.value, operation.status());
        assertEquals(99.0, data.getProgressPercentage(), 0.0);
        assertEquals(9009 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    /**
     * It could be better: short-circuit completion instead of pointless waiting.
     * But, to enable it MoveTask must report a precise number of moved metrics in its result.
     */
    @Test
    public void rescheduleToPermDelAtWhenMoveReplicasWithZeroMetricsComplete() throws InterruptedException {
        // arrange
        var params = params();
        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(0)
            .build();

        // act
        var future = proc.start();

        completeCheckPhaseOnBothReplicas(proc);

        // complete move phase
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(0));
        completeTasksOnReplica(proc.progress().getMoveOnReplicas(0), moveResult);
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(1));
        var progress = proc.progress();
        completeTasksOnReplica(progress.getMoveOnReplicas(1), moveResult);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertEquals(params.getPermanentDeletionAt(), event.executeAt());

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());
        assertReplicasComplete(eventProgress.getMoveOnReplicasList());

        coremonClient.tasks()
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.WAITING_FOR_PERMANENT_DELETION.value, operation.status());
        assertEquals(99.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void rescheduleWhenTerminateShardIdleOnReplica0() throws InterruptedException {
        // arrange
        var params = params().toBuilder()
            .setPermanentDeletionAt(System.currentTimeMillis())
            .build();

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(9000)
            .build();

        var completedAt = Instant.now().minus(Duration.ofDays(31)).toEpochMilli();
        var initialProgress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(completedReplicas(params, completedAt))
            .addAllMoveOnReplicas(completedReplicas(params, completedAt, moveResult))
            .build();

        var context = context(params, initialProgress);
        var proc = task(context);

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        // act
        var future = proc.start();

        // run terminate phase
        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(0));
        var progress = proc.progress();
        rescheduleTasksOnReplica(progress.getTerminateOnReplicas(0), rescheduleAt);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());
        assertReplicasComplete(eventProgress.getMoveOnReplicasList());

        coremonClient.tasks(REPLICA_1)
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.DELETING_PERMANENTLY.value, operation.status());
        assertEquals(99.0, data.getProgressPercentage(), 0.0);
        assertEquals(9000 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void rescheduleWhenTerminateShardIdleOnReplica1() throws InterruptedException {
        // arrange
        var params = params().toBuilder()
            .setPermanentDeletionAt(System.currentTimeMillis())
            .build();

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(9000)
            .build();

        var completedAt = Instant.now().minus(Duration.ofDays(31)).toEpochMilli();
        var initialProgress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(completedReplicas(params, completedAt))
            .addAllMoveOnReplicas(completedReplicas(params, completedAt, moveResult))
            .build();

        var context = context(params, initialProgress);
        var proc = task(context);

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        // act
        var future = proc.start();

        // run terminate phase
        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(0));
        completeTasksOnReplica(proc.progress().getTerminateOnReplicas(0));

        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(1));
        var progress = proc.progress();
        rescheduleTasksOnReplica(progress.getTerminateOnReplicas(1), rescheduleAt);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());
        assertReplicasComplete(eventProgress.getMoveOnReplicasList());
        assertReplicaComplete(eventProgress.getTerminateOnReplicas(0));

        coremonClient.tasks(REPLICA_0)
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.DELETING_PERMANENTLY.value, operation.status());
        assertEquals(99.0, data.getProgressPercentage(), 0.0);
        assertEquals(9000 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void completeWhenTerminateReplicasComplete() throws InterruptedException {
        // arrange
        var params = params().toBuilder()
            .setPermanentDeletionAt(System.currentTimeMillis())
            .build();
        var shardCount = params.getNumIdsCount();

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(43)
            .build();

        var completedAt = Instant.now().minus(Duration.ofDays(31)).toEpochMilli();
        var initialProgress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(completedReplicas(params, completedAt))
            .addAllMoveOnReplicas(completedReplicas(params, completedAt, moveResult))
            .build();

        var context = context(params, initialProgress);
        var proc = task(context);

        manager.tryCreateOperation(context.task().id(), params);

        // act
        var future = proc.start();

        // complete terminate phase
        var terminateResult0 = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(42)
            .build();
        var terminateResult1 = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(40)
            .build();

        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(0));
        completeTasksOnReplica(proc.progress().getTerminateOnReplicas(0), terminateResult0);

        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(1));
        var progress = proc.progress();
        completeTasksOnReplica(progress.getTerminateOnReplicas(1), terminateResult1);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Complete.class);
        assertNotNull(event);

        var result = DeleteMetricsTaskProto.result(event.result());

        assertThat(
            result.getResultsList(),
            containsInAnyOrder(
                ReplicaResult.newBuilder().setClusterId(REPLICA_0).setDeletedMetrics(shardCount * 42).build(),
                ReplicaResult.newBuilder().setClusterId(REPLICA_1).setDeletedMetrics(shardCount * 40).build()));

        coremonClient.tasks()
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(params.getDescription(), operation.description());
        assertEquals(params.getCreatedAt(), operation.createdAt());
        assertEquals(params.getCreatedBy(), operation.createdBy());
        assertEquals(System.currentTimeMillis(), operation.updatedAt(), 10_000d);
        assertEquals(DeleteMetricsOperationStatus.COMPLETED.value, operation.status());

        assertEquals(params.getSelectors(), data.getSelectors());
        assertEquals(params.getPermanentDeletionAt(), data.getPermanentDeletionAt());
        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(43 * shardCount, data.getEstimatedMetricsCount());
        assertEquals(42 * shardCount, data.getPermanentlyDeletedMetricsCount());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void completeWhenTerminateReplicasWithZeroMetricsComplete() throws InterruptedException {
        // arrange
        var params = params().toBuilder()
            .setPermanentDeletionAt(System.currentTimeMillis())
            .build();

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(0)
            .build();

        var completedAt = Instant.now().minus(Duration.ofDays(31)).toEpochMilli();
        var initialProgress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(completedReplicas(params, completedAt))
            .addAllMoveOnReplicas(completedReplicas(params, completedAt, moveResult))
            .build();

        var context = context(params, initialProgress);
        var proc = task(context);

        manager.tryCreateOperation(context.task().id(), params);

        // act
        var future = proc.start();

        // complete terminate phase
        var terminateResult = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(0)
            .build();

        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(0));
        completeTasksOnReplica(proc.progress().getTerminateOnReplicas(0), terminateResult);

        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(1));
        var progress = proc.progress();
        completeTasksOnReplica(progress.getTerminateOnReplicas(1), terminateResult);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Complete.class);
        assertNotNull(event);

        var result = DeleteMetricsTaskProto.result(event.result());

        assertThat(
            result.getResultsList(),
            containsInAnyOrder(
                ReplicaResult.newBuilder().setClusterId(REPLICA_0).setDeletedMetrics(0).build(),
                ReplicaResult.newBuilder().setClusterId(REPLICA_1).setDeletedMetrics(0).build()));

        coremonClient.tasks()
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(params.getDescription(), operation.description());
        assertEquals(params.getCreatedAt(), operation.createdAt());
        assertEquals(params.getCreatedBy(), operation.createdBy());
        assertEquals(System.currentTimeMillis(), operation.updatedAt(), 10_000d);
        assertEquals(DeleteMetricsOperationStatus.COMPLETED.value, operation.status());

        assertEquals(params.getSelectors(), data.getSelectors());
        assertEquals(params.getPermanentDeletionAt(), data.getPermanentDeletionAt());
        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void completeWhenPermDelAtAlreadyReachedOnStart() throws Exception {
        // arrange
        var params = params().toBuilder()
            .setPermanentDeletionAt(System.currentTimeMillis())
            .build();
        var shardCount = params.getNumIdsCount();

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(43)
            .build();

        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        // act
        var future = proc.start();

        completeCheckPhaseOnBothReplicas(proc);
        completeMovePhaseOnBothReplicas(proc, moveResult, moveResult);

        // complete terminate phase
        var terminateResult0 = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(42)
            .build();
        var terminateResult1 = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(40)
            .build();

        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(0));
        completeTasksOnReplica(proc.progress().getTerminateOnReplicas(0), terminateResult0);

        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(1));
        var progress = proc.progress();
        completeTasksOnReplica(progress.getTerminateOnReplicas(1), terminateResult1);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Complete.class);
        assertNotNull(event);

        var result = DeleteMetricsTaskProto.result(event.result());

        assertThat(
            result.getResultsList(),
            containsInAnyOrder(
                ReplicaResult.newBuilder().setClusterId(REPLICA_0).setDeletedMetrics(shardCount * 42).build(),
                ReplicaResult.newBuilder().setClusterId(REPLICA_1).setDeletedMetrics(shardCount * 40).build()));

        coremonClient.tasks()
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(params.getDescription(), operation.description());
        assertEquals(params.getCreatedAt(), operation.createdAt());
        assertEquals(params.getCreatedBy(), operation.createdBy());
        assertEquals(System.currentTimeMillis(), operation.updatedAt(), 10_000d);
        assertEquals(DeleteMetricsOperationStatus.COMPLETED.value, operation.status());

        assertEquals(params.getSelectors(), data.getSelectors());
        assertEquals(params.getPermanentDeletionAt(), data.getPermanentDeletionAt());
        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(43 * shardCount, data.getEstimatedMetricsCount());
        assertEquals(42 * shardCount, data.getPermanentlyDeletedMetricsCount());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void rescheduleToPermDelAtWhenRestartedFromTheMiddleOfMovePhase() throws InterruptedException {
        // arrange
        var params = params();

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(9000)
            .build();

        var completedAt = Instant.now().minus(Duration.ofDays(42)).toEpochMilli();
        var initialProgress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(completedReplicas(params, completedAt))
            .addAllMoveOnReplicas(completedReplicas(params, completedAt, moveResult).subList(0, 1))
            .build();

        var context = context(params, initialProgress);
        var proc = task(context);

        // act
        var future = proc.start();

        // complete move phase
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(1));
        var progress = proc.progress();
        completeTasksOnReplica(progress.getMoveOnReplicas(1));

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertEquals(params.getPermanentDeletionAt(), event.executeAt());

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());
        assertReplicasComplete(eventProgress.getMoveOnReplicasList());

        coremonClient.tasks()
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.WAITING_FOR_PERMANENT_DELETION.value, operation.status());
        assertEquals(99.0, data.getProgressPercentage(), 0.0);
        assertEquals(9000 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void rescheduleToPermDelAtWhenRestartedFromTheMiddleOfCheckPhase() throws InterruptedException {
        // arrange
        var params = params();

        var completedAt = Instant.now().minus(Duration.ofDays(42)).toEpochMilli();
        var initialProgress = DeleteMetricsProgress.newBuilder()
            .addAllCheckOnReplicas(completedReplicas(params, completedAt).subList(0, 1))
            .build();

        var context = context(params, initialProgress);
        var proc = task(context);

        var moveResult0 = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(13)
            .build();
        var moveResult1 = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(42)
            .build();

        // act
        var future = proc.start();

        // complete check phase
        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(1));
        completeTasksOnReplica(proc.progress().getCheckOnReplicas(1));

        // complete move phase
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(0));
        completeTasksOnReplica(proc.progress().getMoveOnReplicas(0), moveResult0);
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(1));
        var progress = proc.progress();
        completeTasksOnReplica(progress.getMoveOnReplicas(1), moveResult1);

        awaitAndJoin(future);

        // assert
        assertNotEquals(progress, proc.progress());

        var event = context.takeDoneEvent(Reschedule.class);
        assertEquals(params.getPermanentDeletionAt(), event.executeAt());

        var eventProgress = DeleteMetricsTaskProto.progress(event.progress());
        assertReplicasComplete(eventProgress.getCheckOnReplicasList());
        assertReplicasComplete(eventProgress.getMoveOnReplicasList());

        coremonClient.tasks()
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(context.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.WAITING_FOR_PERMANENT_DELETION.value, operation.status());
        assertEquals(99.0, data.getProgressPercentage(), 0.0);
        assertEquals(42 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
    }

    @Test
    public void completeWhenRollbackFromNothing() throws InterruptedException {
        // arrange
        var params = params();

        var rollbackProgress = DeleteMetricsRollbackProgress.newBuilder()
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(1.0)
                    .setTotalMetrics(0)
                    .build())
            .build();
        var terminateResult = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(0)
            .build();

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        // act
        var rollbackFuture = rollbackProc.start();

        completeRollbackPhaseOnBothReplicas(rollbackProc, rollbackProgress, rollbackProgress);
        completeTerminatePhaseOnBothReplicas(rollbackProc, terminateResult, terminateResult);

        awaitAndJoin(rollbackFuture);

        // assert
        var event = rollbackContext.takeDoneEvent(Complete.class);
        assertNotNull(event);
        var result = DeleteMetricsTaskProto.result(event.result());

        assertThat(
            result.getResultsList(),
            containsInAnyOrder(
                ReplicaResult.newBuilder().setClusterId(REPLICA_0).setDeletedMetrics(0).build(),
                ReplicaResult.newBuilder().setClusterId(REPLICA_1).setDeletedMetrics(0).build()));

        coremonClient.tasks()
            .forEach(task -> assertEquals(task.toString(), State.COMPLETED, task.getState()));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.CANCELLED.value, operation.status());
        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void completeWhenRollbackFromTheMiddleOfCheckPhase() throws InterruptedException {
        // arrange
        var params = params();

        var rollbackProgress = DeleteMetricsRollbackProgress.newBuilder()
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(1.0)
                    .setTotalMetrics(0)
                    .build())
            .build();
        var terminateResult = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(0)
            .build();

        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            context.task().id(),
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        // act

        // init
        var future = proc.start();
        // run check phase
        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(0));
        context.markCompleted();
        awaitAndJoin(future);

        // rollback
        var rollbackFuture = rollbackProc.start();
        completeRollbackPhaseOnBothReplicas(rollbackProc, rollbackProgress, rollbackProgress);
        completeTerminatePhaseOnBothReplicas(rollbackProc, terminateResult, terminateResult);
        awaitAndJoin(rollbackFuture);

        // assert
        var event = rollbackContext.takeDoneEvent(Complete.class);
        assertNotNull(event);
        var result = DeleteMetricsTaskProto.result(event.result());

        assertThat(
            result.getResultsList(),
            containsInAnyOrder(
                ReplicaResult.newBuilder().setClusterId(REPLICA_0).setDeletedMetrics(0).build(),
                ReplicaResult.newBuilder().setClusterId(REPLICA_1).setDeletedMetrics(0).build()));

        coremonClient.tasks()
            .forEach(task -> assertEquals(task.toString(), State.COMPLETED, task.getState()));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.CANCELLED.value, operation.status());
        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void completeWhenRollbackFromTheMiddleOfMovePhase() throws InterruptedException {
        // arrange
        var params = params();

        var rollbackProgress = DeleteMetricsRollbackProgress.newBuilder()
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(1.0)
                    .setTotalMetrics(4200)
                    .build())
            .build();
        var terminateResult = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(0)
            .build();

        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            context.task().id(),
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        // act

        // init
        var future = proc.start();
        completeCheckPhaseOnBothReplicas(proc);
        // run move phase
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(0));
        context.markCompleted();
        awaitAndJoin(future);

        // rollback
        var rollbackFuture = rollbackProc.start();
        completeRollbackPhaseOnBothReplicas(rollbackProc, rollbackProgress, rollbackProgress);
        completeTerminatePhaseOnBothReplicas(rollbackProc, terminateResult, terminateResult);
        awaitAndJoin(rollbackFuture);

        // assert
        var event = rollbackContext.takeDoneEvent(Complete.class);
        assertNotNull(event);
        var result = DeleteMetricsTaskProto.result(event.result());

        assertThat(
            result.getResultsList(),
            containsInAnyOrder(
                ReplicaResult.newBuilder().setClusterId(REPLICA_0).setDeletedMetrics(0).build(),
                ReplicaResult.newBuilder().setClusterId(REPLICA_1).setDeletedMetrics(0).build()));

        coremonClient.tasks()
            .forEach(task -> assertEquals(task.toString(), State.COMPLETED, task.getState()));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.CANCELLED.value, operation.status());
        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(4200 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void completeWhenRollbackFromBothReplicasMoved() throws InterruptedException {
        // arrange
        var params = params();

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(42)
            .build();
        var rollbackProgress = DeleteMetricsRollbackProgress.newBuilder()
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(1.0)
                    .setTotalMetrics(42)
                    .build())
            .build();
        var terminateResult = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(0)
            .build();

        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        // act

        // init
        var future = proc.start();
        completeCheckPhaseOnBothReplicas(proc);
        completeMovePhaseOnBothReplicas(proc, moveResult, moveResult);
        awaitAndJoin(future);

        // rollback
        var rollbackFuture = rollbackProc.start();

        completeRollbackPhaseOnBothReplicas(rollbackProc, rollbackProgress, rollbackProgress);

        // complete terminate phase
        awaitScheduleTask(rollbackProc, p -> p.getTerminateOnReplicas(0));
        completeTasksOnReplica(rollbackProc.progress().getTerminateOnReplicas(0), terminateResult);

        awaitScheduleTask(rollbackProc, p -> p.getTerminateOnReplicas(1));
        var progress = rollbackProc.progress();
        completeTasksOnReplica(progress.getTerminateOnReplicas(1), terminateResult);

        awaitAndJoin(rollbackFuture);

        // assert
        assertNotEquals(progress, rollbackProc.progress());

        var event = rollbackContext.takeDoneEvent(Complete.class);
        assertNotNull(event);
        var result = DeleteMetricsTaskProto.result(event.result());

        assertThat(
            result.getResultsList(),
            containsInAnyOrder(
                ReplicaResult.newBuilder().setClusterId(REPLICA_0).setDeletedMetrics(0).build(),
                ReplicaResult.newBuilder().setClusterId(REPLICA_1).setDeletedMetrics(0).build()));

        coremonClient.tasks()
            .forEach(task -> assertEquals(task.toString(), State.COMPLETED, task.getState()));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(params.getDescription(), operation.description());
        assertEquals(params.getCreatedAt(), operation.createdAt());
        assertEquals(params.getCreatedBy(), operation.createdBy());
        assertEquals(System.currentTimeMillis(), operation.updatedAt(), 10_000d);
        assertEquals(DeleteMetricsOperationStatus.CANCELLED.value, operation.status());

        assertEquals(params.getSelectors(), data.getSelectors());
        assertThat(data.getPermanentDeletionAt(), greaterThan(params.getPermanentDeletionAt()));
        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(42 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void rescheduleWhenRollbackShardIdleOnReplica0() throws InterruptedException {
        // arrange
        var params = params();

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        // act
        var rollbackFuture = rollbackProc.start();

        awaitScheduleTask(rollbackProc, p -> p.getRollbackOnReplicas(0));
        rescheduleTasksOnReplica(rollbackProc.progress().getRollbackOnReplicas(0), rescheduleAt);

        awaitAndJoin(rollbackFuture);

        // assert
        var event = rollbackContext.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        assertEquals(List.of(), coremonClient.tasks(REPLICA_1));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.CANCELLING.value, operation.status());
        assertEquals(0.0, data.getProgressPercentage(), 0.0);
        assertEquals(0, data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void rescheduleWhenRollbackShardIdleWithSomeProgressOnReplica0() throws InterruptedException {
        // arrange
        var params = params();

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        var rollbackProgress = DeleteMetricsRollbackProgress.newBuilder()
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(0.13)
                    .setTotalMetrics(1_000_000)
                    .build())
            .build();

        // act
        var rollbackFuture = rollbackProc.start();

        awaitScheduleTask(rollbackProc, p -> p.getRollbackOnReplicas(0));
        rescheduleTasksOnReplica(rollbackProc.progress().getRollbackOnReplicas(0), rescheduleAt, rollbackProgress);

        awaitAndJoin(rollbackFuture);

        // assert
        var event = rollbackContext.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        assertEquals(List.of(), coremonClient.tasks(REPLICA_1));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.CANCELLING.value, operation.status());
        assertEquals(13.0 / 2, data.getProgressPercentage(), 0.0);
        assertEquals(1_000_000 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void rescheduleWhenRollbackShardIdleOnReplica1() throws InterruptedException {
        // arrange
        var params = params();

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            context.task().id(),
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(1_111_111)
            .build();
        var rollbackProgress0 = DeleteMetricsRollbackProgress.newBuilder()
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(1.0)
                    .setTotalMetrics(1_000_000)
                    .build())
            .build();
        var rollbackProgress1 = DeleteMetricsRollbackProgress.newBuilder()
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(0.42)
                    .setTotalMetrics(1_000_100)
                    .build())
            .build();

        // act

        // init
        var future = proc.start();
        completeCheckPhaseOnBothReplicas(proc);
        completeMovePhaseOnBothReplicas(proc, moveResult, moveResult);
        awaitAndJoin(future);

        // rollback
        var rollbackFuture = rollbackProc.start();

        awaitScheduleTask(rollbackProc, p -> p.getRollbackOnReplicas(0));
        rescheduleTasksOnReplica(
            rollbackProc.progress().getRollbackOnReplicas(0),
            System.currentTimeMillis(),
            rollbackProgress0);
        completeTasksOnReplica(rollbackProc.progress().getRollbackOnReplicas(0));

        awaitScheduleTask(rollbackProc, p -> p.getRollbackOnReplicas(1));
        rescheduleTasksOnReplica(
            rollbackProc.progress().getRollbackOnReplicas(1),
            rescheduleAt,
            rollbackProgress1);

        awaitAndJoin(rollbackFuture);

        // assert
        var event = rollbackContext.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        coremonClient.tasks(REPLICA_0)
            .forEach(task -> assertEquals(State.COMPLETED, task.getState()));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.CANCELLING.value, operation.status());
        assertEquals(50.0 + 42.0 / 2, data.getProgressPercentage(), 0.0);
        assertEquals(1_000_100 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertEquals("", data.getStatusMessage());
    }

    @Test
    public void rescheduleWhenRollbackShardStuckWithNotOkStatus() throws InterruptedException {
        // arrange
        var params = params(2); // because single shard will be treated as a stuck replica

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        var rollbackProgress = DeleteMetricsRollbackProgress.newBuilder()
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(0.33)
                    .setTotalMetrics(888)
                    .build())
            .build();
        var notOkRollbackProgress = rollbackProgress.toBuilder()
            .setStatus(
                Proto.toProto(
                    Status.FAILED_PRECONDITION.withDescription(
                        "It's been a bad day! Please don't take a picture")))
            .build();

        // act
        var rollbackFuture = rollbackProc.start();

        awaitScheduleTask(rollbackProc, p -> p.getRollbackOnReplicas(0));
        var replica = rollbackProc.progress().getRollbackOnReplicas(0);
        var shards = replica.getOnShardsList();
        var notOkTaskId = random().randomElement(shards).getRemoteTaskId();
        for (var shard : replica.getOnShardsList()) {
            var taskRescheduled = coremonClient.taskById(replica.getClusterId(), shard.getRemoteTaskId()).toBuilder()
                .setState(State.SCHEDULED)
                .setExecuteAt(rescheduleAt)
                .setProgress(Any.pack(
                    shard.getRemoteTaskId().equals(notOkTaskId)
                        ? notOkRollbackProgress
                        : rollbackProgress));

            coremonClient.putTask(replica.getClusterId(), taskRescheduled.build());
        }

        awaitAndJoin(rollbackFuture);

        // assert
        var event = rollbackContext.takeDoneEvent(Reschedule.class);
        assertRescheduleAtWithMaxDelay(rescheduleAt, event, IDLE_RESCHEDULE_MAX_DELAY);

        assertEquals(List.of(), coremonClient.tasks(REPLICA_1));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(DeleteMetricsOperationStatus.CANCELLING.value, operation.status());
        assertEquals(33.0 / 2, data.getProgressPercentage(), 0.0);
        assertEquals(888 * params.getNumIdsCount(), data.getEstimatedMetricsCount());
        assertEquals(0, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertThat(
            data.getStatusMessage(),
            containsString("It's been a bad day! Please don't take a picture"));
    }

    @Test
    public void completeWhenRollbackDonePartially() throws InterruptedException {
        // arrange
        var params = params();
        var shardCount = params.getNumIdsCount();

        // tricks with params instead of clocks
        var terminateParams = params.toBuilder()
            .setPermanentDeletionAt(System.currentTimeMillis() - TimeUnit.DAYS.toMillis(14))
            .build();

        var rescheduleAt = System.currentTimeMillis() + TimeUnit.DAYS.toMillis(5);

        var moveResult = DeleteMetricsMoveResult.newBuilder()
            .setMovedMetrics(42)
            .build();
        var rollbackProgress = DeleteMetricsRollbackProgress.newBuilder()
            .setStatus(Proto.toProto(Status.INTERNAL.withDescription("I'm stuck")))
            .setRollbackDeletedMetrics(
                RollbackDeletedMetricsProgress.newBuilder()
                    .setProgress(0.78)
                    .setTotalMetrics(42)
                    .setStillDeletedMetrics(7)
                    .build())
            .build();
        var terminateResult = DeleteMetricsTerminateResult.newBuilder()
            .setDeletedMetrics(7)
            .build();

        var context = context(params, DeleteMetricsProgress.getDefaultInstance());
        var proc = task(context);

        var rollbackRequested = rollbackRequested();
        var rollbackContext = context(
            context.task().id(),
            params,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var rollbackProc = task(rollbackContext);

        var terminateContext = context(
            context.task().id(),
            terminateParams,
            DeleteMetricsProgress.newBuilder()
                .setRollbackRequested(rollbackRequested)
                .build());
        var terminateProc = task(terminateContext);

        // act

        // init
        var future = proc.start();
        completeCheckPhaseOnBothReplicas(proc);
        completeMovePhaseOnBothReplicas(proc, moveResult, moveResult);
        awaitAndJoin(future);

        // rollback
        var rollbackFuture = rollbackProc.start();

        awaitScheduleTask(rollbackProc, p -> p.getRollbackOnReplicas(0));
        rescheduleTasksOnReplica(
            rollbackProc.progress().getRollbackOnReplicas(0),
            rescheduleAt,
            rollbackProgress);

        awaitScheduleTask(rollbackProc, p -> p.getRollbackOnReplicas(1));
        rescheduleTasksOnReplica(
            rollbackProc.progress().getRollbackOnReplicas(1),
            rescheduleAt,
            rollbackProgress);

        awaitAndJoin(rollbackFuture);

        // terminate
        var terminateFuture = terminateProc.start();
        completeTerminatePhaseOnBothReplicas(terminateProc, terminateResult, terminateResult);
        awaitAndJoin(terminateFuture);

        // assert
        var event = terminateContext.takeDoneEvent(Complete.class);
        assertNotNull(event);
        var result = DeleteMetricsTaskProto.result(event.result());

        assertThat(
            result.getResultsList(),
            containsInAnyOrder(
                ReplicaResult.newBuilder().setClusterId(REPLICA_0).setDeletedMetrics(7 * shardCount).build(),
                ReplicaResult.newBuilder().setClusterId(REPLICA_1).setDeletedMetrics(7 * shardCount).build()));

        coremonClient.tasks()
            .forEach(task -> assertEquals(task.toString(), State.COMPLETED, task.getState()));

        var operation = getOperation(rollbackContext.task().id());
        var data = unpackData(operation);
        assertEquals(System.currentTimeMillis(), operation.updatedAt(), 10_000d);
        assertEquals(DeleteMetricsOperationStatus.COMPLETED.value, operation.status());

        assertEquals(100.0, data.getProgressPercentage(), 0.0);
        assertEquals(42 * shardCount, data.getEstimatedMetricsCount());
        assertEquals(7 * shardCount, data.getPermanentlyDeletedMetricsCount());
        assertEquals(rollbackRequested.getRequestedAt(), data.getCancelledAt());
        assertEquals(rollbackRequested.getRequestedBy(), data.getCancelledBy());
        assertThat(data.getStatusMessage(), containsString("I'm stuck"));
    }

    private void assertRescheduleAtWithMaxDelay(
        long expectedAt,
        Reschedule reschedule,
        Duration maxDelay)
    {
        var executeAt = Instant.ofEpochMilli(reschedule.executeAt());
        var expectedAtLo = Instant.ofEpochMilli(expectedAt);
        var expectedAtHi = Instant.ofEpochMilli(expectedAt).plus(maxDelay);

        assertThat(
            "bad reschedule: " + executeAt,
            executeAt,
            allOf(
                greaterThanOrEqualTo(expectedAtLo),
                lessThanOrEqualTo(expectedAtHi)));
    }

    private void assertReplicasComplete(List<PhaseOnReplicaProgress> phaseOnReplicas) {
        for (var replica : phaseOnReplicas) {
            assertReplicaComplete(replica);
        }
    }

    private void assertReplicaComplete(PhaseOnReplicaProgress replica) {
        for (var shard : replica.getOnShardsList()) {
            assertTrue(shard.getComplete());
            assertNotEquals("", shard.getRemoteTaskId());
            assertTrue(shard.hasRemoteTask());
        }
    }

    private LongRunningOperation getOperation(String operationId) {
        var result = manager.getOperation(operationId).join();
        assertTrue(result.isPresent());
        var operation = result.orElseThrow();

        assertEquals(LongRunningOperationType.DELETE_METRICS, operation.operationType());
        assertEquals(ContainerType.PROJECT, operation.containerType());

        return operation;
    }

    private void awaitScheduleTask(
        DeleteMetricsTask proc,
        Function<DeleteMetricsProgress, PhaseOnReplicaProgress> phaseMapper) throws InterruptedException
    {
        awaitCondition(
            () -> phaseMapper.apply(proc.progress()).getOnShardsList().stream()
                .noneMatch(
                    r -> r.getRemoteTaskId().isEmpty()
                        || r.getRemoteTask().equals(ru.yandex.solomon.scheduler.proto.Task.getDefaultInstance())));
    }

    private void awaitAndJoin(CompletableFuture<?> future) throws InterruptedException {
        awaitCondition(future::isDone);
        future.join();
    }

    private void awaitCondition(BooleanSupplier interrupted) throws InterruptedException {
        while (!interrupted.getAsBoolean()) {
            awaitCoremonCall(interrupted);
        }
    }

    private void awaitCoremonCall(BooleanSupplier interrupted) throws InterruptedException {
        CountDownLatch sync = new CountDownLatch(1);
        coremonClient.beforeSupplier = () -> {
            sync.countDown();
            return CompletableFuture.completedFuture(null);
        };

        while (!sync.await(1, TimeUnit.NANOSECONDS)) {
            clock.passedTime(10, TimeUnit.SECONDS);
            if (interrupted.getAsBoolean()) {
                return;
            }
        }
    }

    private void completeTasksOnReplica(PhaseOnReplicaProgress replica) {
        completeTasksOnReplica(replica, null);
    }

    private void completeTasksOnReplica(PhaseOnReplicaProgress replica, @Nullable Message result) {
        for (var shard : replica.getOnShardsList()) {
            var task = coremonClient.taskById(replica.getClusterId(), shard.getRemoteTaskId()).toBuilder()
                .setState(State.COMPLETED)
                .setStatus(Proto.toProto(Status.OK))
                .setResult(result == null ? Any.getDefaultInstance() : Any.pack(result))
                .build();
            coremonClient.putTask(replica.getClusterId(), task);
        }
    }

    private void completeCheckPhaseOnBothReplicas(DeleteMetricsTask proc) throws InterruptedException {
        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(0));
        completeTasksOnReplica(proc.progress().getCheckOnReplicas(0));
        awaitScheduleTask(proc, p -> p.getCheckOnReplicas(1));
        completeTasksOnReplica(proc.progress().getCheckOnReplicas(1));
    }

    private void completeMovePhaseOnBothReplicas(
        DeleteMetricsTask proc,
        DeleteMetricsMoveResult moveResult0,
        DeleteMetricsMoveResult moveResult1) throws InterruptedException
    {
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(0));
        completeTasksOnReplica(proc.progress().getMoveOnReplicas(0), moveResult0);
        awaitScheduleTask(proc, p -> p.getMoveOnReplicas(1));
        completeTasksOnReplica(proc.progress().getMoveOnReplicas(1), moveResult1);
    }

    private void completeRollbackPhaseOnBothReplicas(
        DeleteMetricsTask proc,
        DeleteMetricsRollbackProgress rollbackProgress0,
        DeleteMetricsRollbackProgress rollbackProgress1) throws InterruptedException
    {
        awaitScheduleTask(proc, p -> p.getRollbackOnReplicas(0));
        rescheduleTasksOnReplica(proc.progress().getRollbackOnReplicas(0), System.currentTimeMillis(), rollbackProgress0);
        completeTasksOnReplica(proc.progress().getRollbackOnReplicas(0));

        awaitScheduleTask(proc, p -> p.getRollbackOnReplicas(1));
        rescheduleTasksOnReplica(proc.progress().getRollbackOnReplicas(1), System.currentTimeMillis(), rollbackProgress1);
        completeTasksOnReplica(proc.progress().getRollbackOnReplicas(1));
    }

    private void completeTerminatePhaseOnBothReplicas(
        DeleteMetricsTask proc,
        DeleteMetricsTerminateResult terminateResult0,
        DeleteMetricsTerminateResult terminateResult1) throws InterruptedException
    {
        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(0));
        completeTasksOnReplica(proc.progress().getTerminateOnReplicas(0), terminateResult0);
        awaitScheduleTask(proc, p -> p.getTerminateOnReplicas(1));
        completeTasksOnReplica(proc.progress().getTerminateOnReplicas(1), terminateResult1);
    }

    private void rescheduleTasksOnReplica(PhaseOnReplicaProgress replica, long rescheduleAt) {
        rescheduleTasksOnReplica(replica, rescheduleAt, null);
    }

    private void rescheduleTasksOnReplica(
        PhaseOnReplicaProgress replica,
        long rescheduleAt,
        @Nullable Message progress)
    {
        for (var shard : replica.getOnShardsList()) {
            var task = coremonClient.taskById(replica.getClusterId(), shard.getRemoteTaskId());
            var taskRescheduled = task.toBuilder()
                .setState(State.SCHEDULED)
                .setProgress(progress == null ? Any.getDefaultInstance() : Any.pack(progress))
                .setExecuteAt(rescheduleAt)
                .build();
            coremonClient.putTask(replica.getClusterId(), taskRescheduled);
        }
    }

    private DeleteMetricsTask task(ExecutionContext context) {
        return new DeleteMetricsTask(
            retryConfig,
            confHolder(DeleteMetricsTaskProto.params(context.task().params())),
            coremonClient,
            manager,
            new DeleteMetricsOperationMetrics(
                new MetricRegistry(),
                DeleteMetricsConfig.newBuilder().setReportVerboseMetrics(true).build()),
            new DeleteMetricsOperationTrackerNoOp(),
            ForkJoinPool.commonPool(),
            timer,
            context);
    }

    private static SolomonConfHolder confHolder(DeleteMetricsParams params) {
        var projects = List.of(
            Project.newBuilder()
                .setId("p")
                .setName("p")
                .setOwner("any")
                .build());

        var clusters = List.of(
            Cluster.newBuilder()
                .setProjectId("p")
                .setId("c")
                .setName("c")
                .build());

        var services = params.getNumIdsList().stream()
            .mapToLong(Integer::toUnsignedLong)
            .mapToObj(
                i -> Service.newBuilder()
                    .setProjectId("p")
                    .setId("s" + i)
                    .setName("s" + i)
                    .build())
            .collect(toList());

        var shards = params.getNumIdsList().stream()
            .skip(1)
            .mapToLong(Integer::toUnsignedLong)
            .mapToObj(
                i -> Shard.newBuilder()
                    .setProjectId("p")
                    .setId("p" + "_" + "c" + "_" + "s" + i)
                    .setNumId((int) i)
                    .setClusterId("c")
                    .setClusterName("c")
                    .setServiceId("s" + i)
                    .setServiceName("s" + i)
                    .build())
            .collect(toList());

        var confHolder = new SolomonConfHolder();
        confHolder.onConfigurationLoad(SolomonConfWithContext.create(
            new SolomonRawConf(List.of(), projects, clusters, services, shards)));
        return confHolder;
    }

    private static List<PhaseOnReplicaProgress> completedReplicas(DeleteMetricsParams params, long completedAt) {
        return completedReplicas(params, completedAt, null);
    }

    private static List<PhaseOnReplicaProgress> completedReplicas(
        DeleteMetricsParams params,
        long completedAt,
        @Nullable Message result)
    {
        return Stream.of(REPLICA_0, REPLICA_1)
            .map(
                clusterId -> PhaseOnReplicaProgress.newBuilder()
                    .setClusterId(clusterId)
                    .addAllOnShards(
                        params.getNumIdsList().stream()
                            .map(numId -> {
                                var id = randomAlphanumeric(10);
                                return RemoteTaskProgress.newBuilder()
                                    .setClusterId(clusterId)
                                    .setComplete(true)
                                    .setRemoteTaskId(id)
                                    .setRemoteTask(
                                        ru.yandex.solomon.scheduler.proto.Task.newBuilder()
                                            .setId(id)
                                            .setParams(Any.pack(
                                                ru.yandex.coremon.api.task.DeleteMetricsParams.newBuilder()
                                                    .setNumId(numId)
                                                    .build()))
                                            .setResult(result == null ? Any.getDefaultInstance() : Any.pack(result))
                                            .build())
                                    .setRemoteTaskCompletedAt(completedAt)
                                    .build();
                            })
                            .collect(toList()))
                    .build())
            .collect(toList());
    }

    private static ExecutionContextStub context(DeleteMetricsParams params, DeleteMetricsProgress progress) {
        return context(UUID.randomUUID().toString(), params, progress);
    }

    private static ExecutionContextStub context(
        String id,
        DeleteMetricsParams params,
        DeleteMetricsProgress progress)
    {
        var task = Task.newBuilder()
            .setId(id)
            .setType("delete_metrics")
            .setExecuteAt(System.currentTimeMillis())
            .setProgress(Any.pack(progress))
            .setParams(Any.pack(params))
            .build();

        return new ExecutionContextStub(task);
    }

    private static RollbackRequested rollbackRequested() {
        return RollbackRequested.newBuilder()
            .setRequestedAt(System.currentTimeMillis() - TimeUnit.SECONDS.toMillis(5))
            .setRequestedBy(randomAlphanumeric(8))
            .build();
    }

    private static Random2 random() {
        return Random2.threadLocal();
    }
}
