package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import io.grpc.Status;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.db.memory.InMemoryLongRunningOperationDao;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.gateway.tasks.deleteMetrics.DeleteMetricsRandom.params;

/**
 * @author Stanislav Kashirin
 */
public class PrepareOperationTest {

    @Rule
    public Timeout globalTimeout = Timeout.builder()
        .withTimeout(1, TimeUnit.MINUTES)
        .withLookingForStuckThread(true)
        .build();

    private RetryConfig retryConfig;
    private InMemoryLongRunningOperationDao dao;
    private DeleteMetricsOperationManager manager;

    @Before
    public void setUp() throws Exception {
        retryConfig = RetryConfig.DEFAULT
            .withNumRetries(Integer.MAX_VALUE)
            .withMaxDelay(0);
        dao = new InMemoryLongRunningOperationDao();
        manager = new DeleteMetricsOperationManager(dao);
    }

    @Test
    public void createNewOne() {
        // arrange
        var operationId = operationId();
        var params = params();
        var proc = prepareOperation(operationId, params);

        // act
        var operation = proc.prepare().join();

        // assert
        assertOperationInDao(operation);
    }

    @Test
    public void alreadyCreated() {
        // arrange
        var operationId = operationId();
        var params = params();
        var alreadyCreated = manager.tryCreateOperation(operationId, params).join().operation();

        var proc = prepareOperation(operationId, params());

        // act
        var operation = proc.prepare().join();

        // assert
        assertEquals(operation, alreadyCreated);
        assertOperationInDao(operation);
    }

    @Test
    public void retryError() {
        // arrange
        var remaining = new AtomicInteger(5);
        dao.beforeSupplier = () -> {
            var r = remaining.get();
            if (r == 5 || r > 0 && ThreadLocalRandom.current().nextBoolean()) {
                remaining.decrementAndGet();
                return failedFuture(Status.UNAVAILABLE.asRuntimeException());
            }

            return completedFuture(null);
        };

        var operationId = operationId();
        var params = params();
        var proc = prepareOperation(operationId, params);

        // act
        var operation = proc.prepare().join();

        // assert
        remaining.set(0);
        assertOperationInDao(operation);
    }

    @Test
    public void whenAlwaysError() {
        // arrange
        retryConfig = retryConfig.withNumRetries(3);
        dao.beforeSupplier = unavailable();

        var operationId = operationId();
        var params = params();
        var proc = prepareOperation(operationId, params);

        // act
        var status = proc.prepare().thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();

        // assert
        assertNotEquals(Status.Code.OK, status.getCode());
        assertNotNull(status.getDescription());
        assertNotNull(status.getCause());
    }

    private PrepareOperation prepareOperation(
        String id,
        DeleteMetricsParams params)
    {
        return new PrepareOperation(retryConfig, manager, id, params);
    }

    private void assertOperationInDao(LongRunningOperation operation) {
        var persisted = manager.getOperation(operation.operationId()).join();
        assertTrue(persisted.isPresent());
        assertEquals(operation, persisted.orElseThrow());
    }

    private static String operationId() {
        return randomAlphanumeric(8);
    }

    private static Supplier<CompletableFuture<?>> unavailable() {
        return () -> failedFuture(Status.UNAVAILABLE.asRuntimeException());
    }
}
