package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.google.protobuf.Any;
import com.google.protobuf.Message;
import io.grpc.Status;
import org.junit.Test;

import ru.yandex.coremon.api.task.DeleteMetricsMoveProgress;
import ru.yandex.coremon.api.task.DeleteMetricsMoveProgress.MoveToDeletedMetricsProgress;
import ru.yandex.coremon.api.task.DeleteMetricsMoveResult;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackProgress;
import ru.yandex.coremon.api.task.DeleteMetricsRollbackProgress.RollbackDeletedMetricsProgress;
import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.gateway.api.task.DeleteMetricsProgress;
import ru.yandex.gateway.api.task.DeleteMetricsProgress.PhaseOnReplicaProgress;
import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.solomon.gateway.tasks.deleteMetrics.ProgressEstimation.ProgressStatus;
import ru.yandex.solomon.scheduler.grpc.Proto;
import ru.yandex.solomon.scheduler.proto.Task;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.junit.Assert.assertEquals;

/**
 * @author Stanislav Kashirin
 */
public class ProgressEstimationTest {

    private static final DeleteMetricsParams PARAMS = DeleteMetricsParams.newBuilder()
        .addAllNumIds(List.of(1, 2, 3))
        .build();

    private static final PhaseOnReplicaProgress MOVE_REPLICA_0 = PhaseOnReplicaProgress.newBuilder()
        .addAllOnShards(
            List.of(
                remoteTaskProgress(
                    DeleteMetricsMoveProgress.newBuilder()
                        .setMoveToDeletedMetrics(
                            MoveToDeletedMetricsProgress.newBuilder()
                                .setEstimatedTotalMetrics(100)
                                .setProgress(0.00)
                        )),
                remoteTaskProgress(
                    DeleteMetricsMoveProgress.newBuilder()
                        .setMoveToDeletedMetrics(
                            MoveToDeletedMetricsProgress.newBuilder()
                                .setEstimatedTotalMetrics(200)
                                .setProgress(0.00)
                        )),
                remoteTaskProgress(
                    DeleteMetricsMoveProgress.newBuilder()
                        .setMoveToDeletedMetrics(
                            MoveToDeletedMetricsProgress.newBuilder()
                                .setEstimatedTotalMetrics(300)
                                .setProgress(0.00)
                        ))))
        .build();

    private static final PhaseOnReplicaProgress MOVE_REPLICA_1 = PhaseOnReplicaProgress.newBuilder()
        .addAllOnShards(
            List.of(
                remoteTaskResult(
                    DeleteMetricsMoveResult.newBuilder()
                        .setMovedMetrics(101)),
                remoteTaskProgress(
                    DeleteMetricsMoveProgress.newBuilder()
                        .setMoveToDeletedMetrics(
                            MoveToDeletedMetricsProgress.newBuilder()
                                .setEstimatedTotalMetrics(202)
                                .setProgress(0.55)
                        )),
                remoteTaskProgress(
                    DeleteMetricsMoveProgress.newBuilder()
                        .setMoveToDeletedMetrics(
                            MoveToDeletedMetricsProgress.newBuilder()
                                .setEstimatedTotalMetrics(303)
                                .setProgress(0.25)
                        ))))
        .build();

    private static final PhaseOnReplicaProgress MOVE_REPLICA_2 = PhaseOnReplicaProgress.newBuilder()
        .addAllOnShards(
            List.of(
                remoteTaskResult(
                    DeleteMetricsMoveResult.newBuilder()
                        .setMovedMetrics(111)),
                remoteTaskResult(
                    DeleteMetricsMoveResult.newBuilder()
                        .setMovedMetrics(222)),
                remoteTaskProgress(
                    DeleteMetricsMoveProgress.newBuilder()
                        .setMoveToDeletedMetrics(
                            MoveToDeletedMetricsProgress.newBuilder()
                                .setEstimatedTotalMetrics(333)
                                .setProgress(1.00)
                        ))))
        .build();

    private static final PhaseOnReplicaProgress ROLLBACK_REPLICA_0 = PhaseOnReplicaProgress.newBuilder()
        .addAllOnShards(
            List.of(
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(100)
                                .setProgress(0.00)
                        )),
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(200)
                                .setProgress(0.00)
                        )),
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(300)
                                .setProgress(0.00)
                        ))))
        .build();

    private static final PhaseOnReplicaProgress ROLLBACK_REPLICA_1 = PhaseOnReplicaProgress.newBuilder()
        .addAllOnShards(
            List.of(
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(101)
                                .setProgress(1.00)
                        )),
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(202)
                                .setProgress(0.55)
                        )),
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(303)
                                .setProgress(0.25)
                        ))))
        .build();

    private static final PhaseOnReplicaProgress ROLLBACK_REPLICA_2 = PhaseOnReplicaProgress.newBuilder()
        .addAllOnShards(
            List.of(
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(111)
                                .setProgress(1.00)
                        )),
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(222)
                                .setProgress(1.00)
                        )),
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(333)
                                .setProgress(1.00)
                        ))))
        .build();

    private static final PhaseOnReplicaProgress ROLLBACK_REPLICA_NOT_OK = PhaseOnReplicaProgress.newBuilder()
        .addAllOnShards(
            List.of(
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(101)
                                .setProgress(1.00)
                        )),
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setStatus(Proto.toProto(Status.FAILED_PRECONDITION))
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(202)
                                .setProgress(0.55)
                        ),
                    2),
                remoteTaskProgress(
                    DeleteMetricsRollbackProgress.newBuilder()
                        .setStatus(Proto.toProto(Status.ABORTED))
                        .setRollbackDeletedMetrics(
                            RollbackDeletedMetricsProgress.newBuilder()
                                .setTotalMetrics(303)
                                .setProgress(0.25)
                        ),
                    3)))
        .build();

    @Test
    public void moveReplica00() {
        var progress = moveProgress(MOVE_REPLICA_0, MOVE_REPLICA_0);

        var estimation = ProgressEstimation.onMove(PARAMS, progress);

        assertEquals(600, estimation.estimatedMetrics());
        assertEquals(0.0, estimation.progressPercentage(), 0.0);
        assertEquals(List.of(), estimation.progressStatuses());
    }

    @Test
    public void moveReplica01() {
        var progress = moveProgress(MOVE_REPLICA_0, MOVE_REPLICA_1);

        var estimation = ProgressEstimation.onMove(PARAMS, progress);

        assertEquals(606, estimation.estimatedMetrics());
        assertEquals(30.0, estimation.progressPercentage(), 0.0);
        assertEquals(List.of(), estimation.progressStatuses());
    }

    @Test
    public void moveReplica12() {
        var progress = moveProgress(MOVE_REPLICA_1, MOVE_REPLICA_2);

        var estimation = ProgressEstimation.onMove(PARAMS, progress);

        assertEquals(666, estimation.estimatedMetrics());
        assertEquals(80.0, estimation.progressPercentage(), 0.0);
        assertEquals(List.of(), estimation.progressStatuses());
    }

    @Test
    public void moveReplica22() {
        var progress = moveProgress(MOVE_REPLICA_2, MOVE_REPLICA_2);

        var estimation = ProgressEstimation.onMove(PARAMS, progress);

        assertEquals(666, estimation.estimatedMetrics());
        assertEquals(99.0, estimation.progressPercentage(), 0.0);
        assertEquals(List.of(), estimation.progressStatuses());
    }

    @Test
    public void rollbackReplica00() {
        var progress = rollbackProgress(ROLLBACK_REPLICA_0, ROLLBACK_REPLICA_0);

        var estimation = ProgressEstimation.onRollback(PARAMS, progress);

        assertEquals(600, estimation.estimatedMetrics());
        assertEquals(0.0, estimation.progressPercentage(), 0.0);
        assertEquals(List.of(), estimation.progressStatuses());
    }

    @Test
    public void rollbackReplica01() {
        var progress = rollbackProgress(ROLLBACK_REPLICA_0, ROLLBACK_REPLICA_1);

        var estimation = ProgressEstimation.onRollback(PARAMS, progress);

        assertEquals(606, estimation.estimatedMetrics());
        assertEquals(30.0, estimation.progressPercentage(), 0.0);
        assertEquals(List.of(), estimation.progressStatuses());
    }

    @Test
    public void rollbackReplica12() {
        var progress = rollbackProgress(ROLLBACK_REPLICA_1, ROLLBACK_REPLICA_2);

        var estimation = ProgressEstimation.onRollback(PARAMS, progress);

        assertEquals(666, estimation.estimatedMetrics());
        assertEquals(80.0, estimation.progressPercentage(), 0.0);
        assertEquals(List.of(), estimation.progressStatuses());
    }

    @Test
    public void rollbackReplica22() {
        var progress = rollbackProgress(ROLLBACK_REPLICA_2, ROLLBACK_REPLICA_2);

        var estimation = ProgressEstimation.onRollback(PARAMS, progress);

        assertEquals(666, estimation.estimatedMetrics());
        assertEquals(99.0, estimation.progressPercentage(), 0.0);
        assertEquals(List.of(), estimation.progressStatuses());
    }

    @Test
    public void rollbackNotOk() {
        var a = ROLLBACK_REPLICA_NOT_OK.toBuilder().setClusterId("a").build();
        var b = ROLLBACK_REPLICA_NOT_OK.toBuilder().setClusterId("b").build();
        var progress = rollbackProgress(a, b);

        var estimation = ProgressEstimation.onRollback(PARAMS, progress);

        assertEquals(606, estimation.estimatedMetrics());
        assertEquals(60.0, estimation.progressPercentage(), 0.0);

        var expectedStatuses = List.of(
            new ProgressStatus("a", 2, Status.FAILED_PRECONDITION),
            new ProgressStatus("a", 3, Status.ABORTED),
            new ProgressStatus("b", 2, Status.FAILED_PRECONDITION),
            new ProgressStatus("b", 3, Status.ABORTED));

        assertThat(estimation.progressStatuses(), containsInAnyOrder(expectedStatuses.toArray()));
    }

    private static DeleteMetricsProgress moveProgress(PhaseOnReplicaProgress... progressOnReplicas) {
        var replicas = Arrays.asList(progressOnReplicas);
        Collections.shuffle(replicas);

        return DeleteMetricsProgress.newBuilder()
            .addAllMoveOnReplicas(replicas)
            .build();
    }

    private static DeleteMetricsProgress rollbackProgress(PhaseOnReplicaProgress... progressOnReplicas) {
        var replicas = Arrays.asList(progressOnReplicas);
        Collections.shuffle(replicas);

        return DeleteMetricsProgress.newBuilder()
            .addAllRollbackOnReplicas(replicas)
            .build();
    }

    private static RemoteTaskProgress remoteTaskProgress(Message.Builder proto) {
        return remoteTaskProgress(proto, 0);
    }

    private static RemoteTaskProgress remoteTaskProgress(Message.Builder proto, int numId) {
        return RemoteTaskProgress.newBuilder()
            .setRemoteTask(
                Task.newBuilder()
                    .setProgress(Any.pack(proto.build()))
                    .setParams(
                        Any.pack(
                            ru.yandex.coremon.api.task.DeleteMetricsParams.newBuilder()
                                .setNumId(numId)
                                .build())))
            .build();
    }

    private static RemoteTaskProgress remoteTaskResult(Message.Builder proto) {
        return RemoteTaskProgress.newBuilder()
            .setComplete(true)
            .setRemoteTask(
                Task.newBuilder()
                    .setResult(Any.pack(proto.build())))
            .build();
    }
}
