package ru.yandex.solomon.gateway.tasks.removeShard;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import io.grpc.Status;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.gateway.api.task.RemoveShardParams;
import ru.yandex.gateway.api.task.RemoveShardProgress.RemoveConf;
import ru.yandex.solomon.core.db.dao.memory.InMemoryShardDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.util.future.RetryConfig;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class RemoveShardFromConfTest {

    @Rule
    public Timeout globalTimeout = Timeout.builder()
            .withTimeout(1, TimeUnit.MINUTES)
            .withLookingForStuckThread(true)
            .build();

    private RetryConfig retryConfig;
    private InMemoryShardDao dao;

    @Before
    public void setUp() throws Exception {
        dao = new InMemoryShardDao();
        retryConfig = RetryConfig.DEFAULT
                .withNumRetries(Integer.MAX_VALUE)
                .withMaxDelay(0);
    }

    @Test
    public void alreadyDone() {
        var shard = randomShard();
        var progress = RemoveConf.newBuilder()
                .setComplete(true)
                .build();

        dao.beforeSupplier = () -> {
            throw Status.UNAVAILABLE.asRuntimeException();
        };

        var proc = remove(params(shard), progress);
        proc.start().join();
        assertEquals(progress, proc.progress());
    }

    @Test
    public void nothingToRemove() {
        var shard = randomShard();
        var proc = remove(params(shard), RemoveConf.getDefaultInstance());
        proc.start().join();

        var expected = RemoveConf.newBuilder()
                .setComplete(true)
                .build();

        assertEquals(expected, proc.progress());
    }

    @Test
    public void removeShard() {
        var shard = randomShard();
        dao.insert(shard).join();
        var proc = remove(params(shard), RemoveConf.getDefaultInstance());
        proc.start().join();

        var expected = RemoveConf.newBuilder()
                .setComplete(true)
                .build();
        assertEquals(expected, proc.progress());
        assertNull(dao.getById(shard.getProjectId(), shard.getId()));
    }

    @Test
    public void avoidRemoveDifferentNumId() {
        var shardV1 = randomShard();
        dao.insert(shardV1).join();
        assertTrue(dao.deleteOne(shardV1.getProjectId(), shardV1.getFolderId(), shardV1.getId()).join());
        var shardV2 = shardV1.toBuilder()
                .setNumId(ThreadLocalRandom.current().nextInt(1, Integer.MAX_VALUE))
                .build();
        assertTrue(dao.insert(shardV2).join());

        var proc = remove(params(shardV1), RemoveConf.getDefaultInstance());
        proc.start().join();

        var expected = RemoveConf.newBuilder()
                .setComplete(true)
                .build();
        assertEquals(expected, proc.progress());
        assertEquals(shardV2, dao.getById(shardV2.getProjectId(), shardV2.getId()));
    }

    @Test
    public void retryError() {
        var shard = randomShard();
        dao.insert(shard).join();
        AtomicInteger remaining = new AtomicInteger(5);
        dao.beforeSupplier = () -> {
            if (ThreadLocalRandom.current().nextBoolean() && remaining.get() > 0) {
                remaining.decrementAndGet();
                return CompletableFuture.failedFuture(Status.UNAVAILABLE.asRuntimeException());
            }

            return CompletableFuture.completedFuture(null);
        };

        var proc = remove(params(shard), RemoveConf.getDefaultInstance());
        proc.start().join();

        var expected = RemoveConf.newBuilder()
                .setComplete(true)
                .build();

        assertEquals(expected, proc.progress());
        assertNull(dao.getById(shard.getProjectId(), shard.getId()));
    }

    private Shard randomShard() {
        var random = ThreadLocalRandom.current();
        return Shard.newBuilder()
                .setProjectId("project_id_" + random.nextLong())
                .setId("shard_id_" + random.nextLong())
                .setClusterId("cluster_id_" + random.nextLong())
                .setClusterName("cluster_name_" + random.nextLong())
                .setServiceId("service_id_" + random.nextLong())
                .setServiceName("service_name_" + random.nextLong())
                .build();
    }

    private RemoveShardParams params(Shard shard) {
        return RemoveShardParams.newBuilder()
                .setProjectId(shard.getProjectId())
                .setShardId(shard.getId())
                .setNumId(shard.getNumId())
                .build();
    }

    private RemoveShardFromConf remove(RemoveShardParams params, RemoveConf process) {
        return new RemoveShardFromConf(retryConfig, dao, params, process);
    }
}
