package ru.yandex.solomon.gateway.utils.url;

import java.util.AbstractMap;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Random;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.junit.Assert;
import org.junit.Test;

/**
 * @author Stepan Koltsov
 */
public class UrlUtilsTest {

    private Random random = new Random(17);

    @Test
    public void addParameter() {
        Assert.assertEquals("/?foo=bar", UrlUtils.addParameter("/", "foo", "bar"));
        Assert.assertEquals("/?a=b&foo=bar#fragment", UrlUtils.addParameter("/?a=b#fragment", "foo", "bar"));
        Assert.assertEquals("http://yandex.ru/?q=solomon#result", UrlUtils.addParameter("http://yandex.ru/#result", "q", "solomon"));
        Assert.assertEquals("/?a=%26", UrlUtils.addParameter("/", "a", "&"));
    }

    private void testRemoveQueryArgs(String expected, String url, String... args) {
        String message = "input: " + url + ", rm: " + Arrays.toString(args);

        Assert.assertEquals(message, expected, UrlUtils.removeQueryArgs(url, args));
    }

    @Test
    public void removeQueryArgs() {
        testRemoveQueryArgs("/", "/", "a");
        testRemoveQueryArgs("/foo", "/foo", "aaa");
        testRemoveQueryArgs("/foo", "/foo?", "aaa");
        testRemoveQueryArgs("/?b=c", "/?b=c", "x");
        testRemoveQueryArgs("/?b=c", "/?b=c", "x", "y");
        testRemoveQueryArgs("/", "/?b=c", "b");
        testRemoveQueryArgs("/", "/?b=c&b=e", "b");
        testRemoveQueryArgs("/?x=y", "/?b=c&b=e&x=y", "b");
        testRemoveQueryArgs("/?x=y&x=y", "/?x=y&b=e&x=y", "b");
        testRemoveQueryArgs("/?b=e", "/?x=y&b=e&x=y", "x");
        testRemoveQueryArgs("/?x=%20", "/?b=c&x=%20", "b");

        // parameter without value
        testRemoveQueryArgs("/?x=11", "/?b&x=11", "b");

        // extra amps
        testRemoveQueryArgs("/", "/?&&xx=yy&&", "xx");
    }

    private static String addSep(String url) {
        if (url.contains("?")) {
            return url + "&";
        } else {
            return url + "?";
        }
    }

    private void removeQueryArgsRandomIteration() {
        int length = random.nextInt(4);

        String expected = random.nextBoolean() ? "/foo" : "/";
        String input = expected;

        String remove = "bar";

        String[] parameterNames = new String[] { remove, "baz", "qux", "quux" };

        for (int i = 0; i < length; ++i) {
            input = addSep(input);

            if (random.nextInt(3) == 0) {
                input += "&";
            }

            String param = parameterNames[random.nextInt(parameterNames.length)];
            int value = random.nextInt(1000);

            if (param.equals(remove)) {
                input = addSep(input);
                if (value < 100) {
                    input += param + "=";
                } else if (value < 200) {
                    input += param;
                } else {
                    input += param + "=" + value;
                }
            } else {
                expected = addSep(expected);
                expected += param + "=" + value;

                input = addSep(input);
                input += param + "=" + value;
            }
        }

        if (random.nextBoolean()) {
            input = addSep(input);
        }

        testRemoveQueryArgs(expected, input, remove);
    }

    @Test
    public void removeQueryArgsRandom() {
        for (int i = 0; i < 10000; ++i) {
            removeQueryArgsRandomIteration();
        }
    }

    private static <A, B> Map<A, B> map(A k1, B v1, A k2, B v2) {
        return Stream.of(new AbstractMap.SimpleEntry<>(k1, v1), new AbstractMap.SimpleEntry<>(k2, v2))
            .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    @Test
    public void fillDefaults() {
        Assert.assertEquals("/?a=b", UrlUtils.fillDefault("/", "a", "b"));
        Assert.assertEquals("/?a=b&c=d", UrlUtils.fillDefaultsMap("/", map("a", "b", "c", "d")));
        Assert.assertEquals("/?a=b", UrlUtils.fillDefault("/?a=b", "a", "x"));
        Assert.assertEquals("/?a=b&a=c", UrlUtils.fillDefault("/?a=b&a=c", "a", "x"));
        Assert.assertEquals("/?a=b&c=d", UrlUtils.fillDefault("/?a=b&c=d", "a", "x"));
    }

    @Test
    public void parseQueryArgs() {
        LinkedHashMap<String, String> expected = new LinkedHashMap<>();
        expected.put("a", "b");
        expected.put("c", "d=e");
        Assert.assertEquals(expected, UrlUtils.parseQueryArgs("a=b&c=d=e"));

        Assert.assertEquals("+", UrlUtils.parseQueryArgs("x=%2B").get("x"));
        Assert.assertEquals("a b", UrlUtils.parseQueryArgs("x=a+b").get("x"));
    }

    private void updateParameterTest(String expected, String url, String name, String value) {
        String message = "input: " + url + "; name: " + name + "; value: " + value;
        Assert.assertEquals(message, expected, UrlUtils.updateParameter(url, name, value));
    }

    @Test
    public void decodeArg() {
        Assert.assertEquals("", UrlUtils.decodeArg(""));
        Assert.assertEquals("a+b", UrlUtils.decodeArg("a%2Bb"));
        Assert.assertEquals("a b", UrlUtils.decodeArg("a+b"));
        Assert.assertEquals("a b", UrlUtils.decodeArg("a%20b"));
        Assert.assertEquals("a b c", UrlUtils.decodeArg("a+b+c"));
        Assert.assertEquals("a b c", UrlUtils.decodeArg("a%20b%20c"));
    }

    @Test
    public void enodeArg() {
        Assert.assertEquals("", UrlUtils.encodeArg(""));
        Assert.assertEquals("a%2Bb", UrlUtils.encodeArg("a+b"));
        Assert.assertEquals("a%20b", UrlUtils.encodeArg("a b"));
        Assert.assertEquals("a%20b%20c", UrlUtils.encodeArg("a b c"));
    }

    @Test
    public void updateParameter() {
        // add parameter
        updateParameterTest("/?x=y&z=w", "/?x=y", "z", "w");
        // replace preserving order
        updateParameterTest("/?a=b&x=2&z=w", "/?a=b&x=y&z=w", "x", "2");
        // removes multiple entries
        updateParameterTest("/?a=b&x=2&z=w", "/?a=b&x=y1&x=y2&z=w", "x", "2");

        updateParameterTest("/?aa=cc&cc=dd", "/?aa=bb&cc=dd", "aa", "cc");
        updateParameterTest("/?aa=cc", "/?aa=bb", "aa", "cc");
        updateParameterTest("/?cc=dd&aa=cc", "/?cc=dd&aa=bb", "aa", "cc");

        updateParameterTest("/?aa=bb", "/?aa=bb", "aa", "bb");

        // some partial strings
        updateParameterTest("/?ab=cd", "/?", "ab", "cd");
        updateParameterTest("/?ab=cd", "/?ab=cc&", "ab", "cd");
        updateParameterTest("/?ab=cd", "/?ab", "ab", "cd");
        updateParameterTest("/?ab=cd", "/?ab&", "ab", "cd");
        updateParameterTest("/?ab=cd&xx=yy", "/?ab&xx=yy", "ab", "cd");
    }

    @Test
    public void updateOrRemoveParameter() {
        // removes parameter if value == null
        Assert.assertEquals("/?a=b&z=w", UrlUtils.updateOrRemoveParameter("/?a=b&x=y1&z=w", "x", null));
    }

}
