package ru.yandex.solomon.gateway.viewHistory;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Predicate;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.core.db.model.ViewHistory;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.utils.url.QueryArg;
import ru.yandex.solomon.gateway.utils.url.UrlUtils;
import ru.yandex.solomon.util.UrlBuilder;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;


/**
 * @author albazh
 */
public class ViewHistoryUrlUtilsTest {

    @Test
    public void sortQueryArgsByName() {
        {
            String orig = "/some/path";
            Assert.assertEquals("/some/path",
                ViewHistoryUrlUtils.sortQueryArgsByName(UrlUtils.parseUrl(orig)).reconstruct());
        }
        {
            String orig = "/some/path?";
            Assert.assertEquals("/some/path",
                ViewHistoryUrlUtils.sortQueryArgsByName(UrlUtils.parseUrl(orig)).reconstruct());
        }
        {
            String orig = "/some/path?a=x";
            Assert.assertEquals("/some/path?a=x",
                ViewHistoryUrlUtils.sortQueryArgsByName(UrlUtils.parseUrl(orig)).reconstruct());
        }
        {
            String orig = "/some/path?a=x&b=x";
            Assert.assertEquals("/some/path?a=x&b=x",
                ViewHistoryUrlUtils.sortQueryArgsByName(UrlUtils.parseUrl(orig)).reconstruct());
        }
        {
            String orig = "/some/path?b=x&a=x&c=x";
            Assert.assertEquals("/some/path?a=x&b=x&c=x",
                ViewHistoryUrlUtils.sortQueryArgsByName(UrlUtils.parseUrl(orig)).reconstruct());
        }
    }

    @Test
    public void isQueryArgsSortedByName() {
        {
            ArrayList<QueryArg> orig = new ArrayList<>(Collections.emptyList());
            assertTrue(ViewHistoryUrlUtils.isQueryArgsSortedByName(orig));
        }
        {
            ArrayList<QueryArg> orig = new ArrayList<>(Collections.singletonList(
                QueryArg.decoded("a", "x")
            ));
            assertTrue(ViewHistoryUrlUtils.isQueryArgsSortedByName(orig));
        }
        {
            ArrayList<QueryArg> orig = new ArrayList<>(Arrays.asList(
                QueryArg.decoded("a", "x"),
                QueryArg.decoded("b", "x")
            ));
            assertTrue(ViewHistoryUrlUtils.isQueryArgsSortedByName(orig));
        }
        {
            ArrayList<QueryArg> orig = new ArrayList<>(Arrays.asList(
                QueryArg.decoded("b", "x"),
                QueryArg.decoded("a", "x")
            ));
            assertFalse(ViewHistoryUrlUtils.isQueryArgsSortedByName(orig));
        }
    }

    @Test
    public void matchByUrl() throws URISyntaxException {
        Map<String, String> pageUrlParams = new HashMap<>();
        pageUrlParams.put(UserLinksBasic.PROJECT, "solomon");
        pageUrlParams.put(UserLinksBasic.CLUSTER, "salmon");
        pageUrlParams.put(UserLinksBasic.SERVICE, "sys");
        pageUrlParams.put("path", "/Io/Disks/Reads");
        pageUrlParams.put("host", "Sas");
        pageUrlParams.put("disk", "sda");
        pageUrlParams.put(UserLinksBasic.GRAPH_PARAM, "auto");

        //check the same
        String pageUrl = toUriString(pageUrlParams);
        String historyUrl = pageUrl;
        Predicate<ViewHistory> matcher = ViewHistoryUrlUtils.matchByUrl(pageUrl);
        assertTrue(matcher.test(mocked(historyUrl)));

        Map<String, String> historyUrlParams = new HashMap<>(pageUrlParams);
        historyUrlParams.put(UserLinksBasic.B_QA, "2016-06-03T11:12:19.057Z");
        historyUrlParams.put(UserLinksBasic.E_QA, "2016-06-09T02:35:27.496Z");
        historyUrl = toUriString(historyUrlParams);
        assertTrue(matcher.test(mocked(historyUrl)));

        historyUrlParams.put(UserLinksBasic.PROJECT, "other");
        historyUrl = toUriString(historyUrlParams);
        assertFalse(matcher.test(mocked(historyUrl)));

        pageUrlParams.put(UserLinksBasic.REVIVER_CLIENT_ID, "client1");
        pageUrl = toUriString(pageUrlParams);
        matcher = ViewHistoryUrlUtils.matchByUrl(pageUrl);
        assertFalse(matcher.test(mocked(historyUrl)));

        historyUrlParams.put(UserLinksBasic.PROJECT, "solomon");
        historyUrl = toUriString(historyUrlParams);
        assertTrue(matcher.test(mocked(historyUrl)));
    }

    private static ViewHistory mocked(String url) {
        return new ViewHistory("some-id", url, "some-page-title", false);
    }

    private static String toUriString(Map<String, String> parameters) {
        return toUriString("/some/path", parameters);
    }

    private static String toUriString(String path, Map<String, String> parameters) {
        UrlBuilder uriBuilder = new UrlBuilder(path);
        parameters.forEach(uriBuilder::appendQueryArg);
        return uriBuilder.build();
    }
}
