package ru.yandex.solomon.gateway.viewHistory;

import java.util.Arrays;
import java.util.List;
import java.util.function.Function;
import java.util.stream.IntStream;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.tuple.Triple;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.solomon.core.db.model.ViewHistory;

import static org.junit.Assert.assertEquals;


/**
 * @author albazh
 */
@YaIgnore
public class ViewHistoryUrlUtilsUrlPcsTest {

    private static final String URL_A_MAIN_SYS = "/?project=A&cluster=MAIN&service=SYS";
    private static final Triple<String, String, String> PCS_A_MAIN_SYS = Triple.of("A", "MAIN", "SYS");
    private static final UrlAndPcs A_MAIN_SYS = new UrlAndPcs(URL_A_MAIN_SYS, PCS_A_MAIN_SYS);

    private static final String URL_PINNED = "/?project=SOLOMON&cluster=SALMON&dashboard=SOLOMON-CUMULATIVE-TIME-SYS";
    private static final Triple<String, String, String> PCS_PINNED = Triple.of("SOLOMON", "SALMON", "");
    private static final UrlAndPcs PINNED = new UrlAndPcs(URL_PINNED, PCS_PINNED);

    private ImmutableMap<String, List<TestItem>> suite = ImmutableMap.<String, List<TestItem>>builder()
        .put(
            "common positive tests", Arrays.asList(
                new TestItem(
                    new UrlAndPcs("localhost:8080/?project=A&cluster=MAIN&service=SYS", Triple.of("A", "MAIN", "SYS")),
                    A_MAIN_SYS,
                    true),
                new TestItem(
                    new UrlAndPcs("localhost:8080/?project=A&cluster=MAIN", Triple.of("A", "MAIN", "")),
                    A_MAIN_SYS,
                    true),
                new TestItem(
                    new UrlAndPcs("localhost:8080/?project=A", Triple.of("A", "", "")),
                    A_MAIN_SYS,
                    true),
                new TestItem(
                    new UrlAndPcs("localhost:8080", Triple.of("", "", "")),
                    A_MAIN_SYS,
                    true),
                new TestItem(
                    new UrlAndPcs("localhost:8080/?randomArg=1&x=2&y=true", Triple.of("", "", "")),
                    A_MAIN_SYS,
                    true)
            )
        )
        .put(
            "common negative tests", Arrays.asList(
                new TestItem(
                    new UrlAndPcs("localhost:8080/?project=A&cluster=MAIN&service=NOT_SYS",
                        Triple.of("A", "MAIN", "NOT_SYS")),
                    A_MAIN_SYS,
                    false),
                new TestItem(
                    new UrlAndPcs("localhost:8080/?project=A&cluster=NOT_MAIN", Triple.of("A", "NOT_MAIN", "")),
                    A_MAIN_SYS,
                    false),
                new TestItem(
                    new UrlAndPcs("localhost:8080/?project=NOT_AA", Triple.of("NOT_AA", "", "")),
                    A_MAIN_SYS,
                    false)
            )
        )
        .put(
            "pinned dashboard tests", Arrays.asList(
                new TestItem(
                    new UrlAndPcs("localhost:8080/?project=SOLOMON&cluster=SALMON&service=SYS",
                        Triple.of("SOLOMON", "SALMON", "SYS")),
                    PINNED,
                    true),
                new TestItem(
                    new UrlAndPcs("localhost:8080/?cluster=SALMON&service=SYS", Triple.of("", "SALMON", "SYS")),
                    PINNED,
                    true),
                new TestItem(
                    new UrlAndPcs("localhost:8080/?project=SOLOMON&service=SYS", Triple.of("SOLOMON", "", "SYS")),
                    PINNED,
                    true),
                new TestItem(
                    new UrlAndPcs("localhost:8080/?service=SYS", Triple.of("", "", "SYS")),
                    PINNED,
                    true)
            )
        )
        .build();

    @Test
    public void pcs() {
        forEachTest(TestItem::testPcs);
    }

    @Test
    public void url() {
        forEachTest(TestItem::testUrl);
    }

    private void forEachTest(Function<TestItem, Boolean> tester) {
        suite.forEach((desc, items) -> {
            IntStream.range(0, items.size()).forEach(idx -> {
                TestItem testItem = items.get(idx);
                assertEquals(desc + ": " + idx, testItem.result, tester.apply(testItem));
            });
        });
    }

    private static class TestItem {

        private final UrlAndPcs page;
        private final UrlAndPcs history;
        private final ViewHistory view;
        private final boolean result;

        private TestItem(
            UrlAndPcs page,
            UrlAndPcs history,
            boolean result)
        {
            this.page = page;
            this.history = history;
            this.view = new ViewHistory("login", history.url, "title", true);
            this.result = result;
        }

        private boolean testUrl() {
            return ViewHistoryUrlUtils.matchByUrlPcsArgs(page.url).test(view);
        }

        private boolean testPcs() {
            return ViewHistoryUrlUtils.equalPcs(page.pcs, history.pcs);
        }
    }

    private static class UrlAndPcs {

        private final String url;
        private final Triple<String, String, String> pcs;

        private UrlAndPcs(
            String url,
            Triple<String, String, String> pcs)
        {
            this.url = url;
            this.pcs = pcs;
        }
    }

}
